#include <ctype.h>
#include <stdlib.h>
#include <string.h>

#include "base.h"
#include "log.h"
#include "buffer.h"

#include "plugin.h"

#ifdef HAVE_CONFIG_H
#include "config.h"
#endif

/**
 * this is a skeleton for a lighttpd plugin
 * 
 * just replaces every occurance of 'skeleton' by your plugin name
 * 
 * e.g. in vim:
 * 
 *   :%s/skeleton/myhandler/
 * 
 */



/* plugin config for all request/connections */

typedef struct {
	PLUGIN_DATA;
	buffer *match_buf;
	buffer *conf_match;
} plugin_data;

typedef struct {
	size_t foo;
} handler_ctx;

static handler_ctx * handler_ctx_init() {
	handler_ctx * hctx;
	
	hctx = calloc(1, sizeof(*hctx));
	
	return hctx;
}

static void handler_ctx_free(handler_ctx *hctx) {
	
	free(hctx);
}

/* init the plugin data */
INIT_FUNC(mod_skeleton_init) {
	plugin_data *p;
	
	p = calloc(1, sizeof(*p));
	
	p->match_buf = buffer_init();
	p->conf_match = buffer_init();
	
	return p;
}

/* detroy the plugin data */
FREE_FUNC(mod_skeleton_free) {
	plugin_data *p = p_d;
	
	if (!p) return HANDLER_GO_ON;
	
	buffer_free(p->match_buf);
	buffer_free(p->conf_match);
	
	free(p);
	
	return HANDLER_GO_ON;
}

/* handle plugin config and check values */

SETDEFAULTS_FUNC(mod_skeleton_set_defaults) {
	plugin_data *p = p_d;
	size_t i = 0;
	
	config_values_t cv[] = { 
		{ "skeleton.string",            NULL, T_CONFIG_STRING },       /* 0 */
		{ NULL,                         NULL, T_CONFIG_UNSET }
	};
	
	if (!p) return HANDLER_ERROR;
	
	/* 0 */
	cv[i++].destination = p->conf_match;
	
	if (0 != config_insert_values(srv, cv)) {
		return HANDLER_ERROR;
	}
	
	return HANDLER_GO_ON;
}

URIHANDLER_FUNC(mod_skeleton_uri_handler) {
	plugin_data *p = p_d;
	int s_len;
	size_t k;
	
	if (con->uri.path->used == 0) return HANDLER_GO_ON;
	
	s_len = con->uri.path->used - 1;
	
	for (k = 0; k < p->conf_access_deny->used; k++) {
		data_string *ds = (data_string *)p->conf_access_deny->data[k];
		int ct_len = ds->value->used - 1;
		
		if (ct_len > s_len) continue;
		if (ds->value->used == 0) continue;
		
		if (0 == strncmp(con->uri.path->ptr + s_len - ct_len, ds->value->ptr, ct_len)) {
			con->http_status = 403;
	
			return HANDLER_FINISHED;
		}
	}
	
	/* not found */
	return HANDLER_GO_ON;
}

/* this function is called at dlopen() time and inits the callbacks */

int mod_skeleton_plugin_init(plugin *p) {
	p->name        = buffer_init_string("skeleton");
	
	p->init        = mod_skeleton_init;
	p->handle_uri_clean  = mod_skeleton_uri_handler;
	p->set_defaults  = mod_skeleton_set_defaults;
	p->cleanup     = mod_skeleton_free;
	
	p->data        = NULL;
	
	return 0;
}
