#define _GNU_SOURCE
#include <sys/types.h>
#include <sys/socket.h>

#include <netinet/in.h>

#include <arpa/inet.h>


#include <fcntl.h>
#include <stdlib.h>
#include <string.h>
#include <unistd.h>
#include <errno.h>
#include <time.h>

#include "server.h"
#include "connections.h"
#include "mod_status.h"
#include "response.h"
#include "connections.h"
#include "log.h"

#include "plugin.h"

#include "inet_ntop_cache.h"

typedef struct {
	PLUGIN_DATA;
	
	double traffic_out;
	double requests;
	
	double mod_5s_traffic_out[5];
	double mod_5s_requests[5];
	size_t mod_5s_ndx;
	
	double rel_traffic_out;
	double rel_requests;
	
	double abs_traffic_out;
	double abs_requests;
	
	double bytes_written;
	
	buffer *module_list;
	buffer *config_url;
	buffer *status_url;
} plugin_data;

INIT_FUNC(mod_status_init) {
	plugin_data *p;
	size_t i;
	
	p = calloc(1, sizeof(*p));
	
	p->traffic_out = p->requests = 0;
	p->rel_traffic_out = p->rel_requests = 0;
	p->abs_traffic_out = p->abs_requests = 0;
	p->bytes_written = 0;
	p->module_list = buffer_init();
	p->status_url = buffer_init();
	p->config_url = buffer_init();
	
	for (i = 0; i < 5; i++) {
		p->mod_5s_traffic_out[i] = p->mod_5s_requests[i] = 0;
	}
	
	return p;
}

FREE_FUNC(mod_status_free) {
	plugin_data *p = p_d;
	
	UNUSED(srv);

	if (!p) return HANDLER_GO_ON;
	
	buffer_free(p->module_list);
	buffer_free(p->status_url);
	buffer_free(p->config_url);
	
	free(p);
	
	return HANDLER_GO_ON;
}

SETDEFAULTS_FUNC(mod_status_set_defaults) {
	plugin_data *p = p_d;
	
	config_values_t cv[] = { 
		{ "status.status-url",           NULL, T_CONFIG_STRING, T_CONFIG_SCOPE_CONNECTION },
		{ "status.config-url",           NULL, T_CONFIG_STRING, T_CONFIG_SCOPE_CONNECTION },
		{ NULL,                          NULL, T_CONFIG_UNSET, T_CONFIG_SCOPE_UNSET }
	};
	
	if (!p) return HANDLER_ERROR;
	cv[0].destination = p->status_url;
	cv[1].destination = p->config_url;
	
	if (0 != config_insert_values(srv, cv)) {
		return HANDLER_ERROR;
	}
	
	/* check for dir */
	return HANDLER_GO_ON;
}



static int mod_status_row_append(buffer *b, const char *key, const char *value) {
	BUFFER_APPEND_STRING_CONST(b, "   <tr>\n");
	BUFFER_APPEND_STRING_CONST(b, "    <td><b>");
	buffer_append_string(b, key);
	BUFFER_APPEND_STRING_CONST(b, "</b></td>\n");
	BUFFER_APPEND_STRING_CONST(b, "    <td>");
	buffer_append_string(b, value);
	BUFFER_APPEND_STRING_CONST(b, "</td>\n");
	BUFFER_APPEND_STRING_CONST(b, "   </tr>\n");
	
	return 0;
}

static int mod_status_header_append(buffer *b, const char *key) {
	BUFFER_APPEND_STRING_CONST(b, "   <tr>\n");
	BUFFER_APPEND_STRING_CONST(b, "    <th colspan=\"2\">");
	buffer_append_string(b, key);
	BUFFER_APPEND_STRING_CONST(b, "</th>\n");
	BUFFER_APPEND_STRING_CONST(b, "   </tr>\n");
	
	return 0;
}

static handler_t mod_status_handle_server_status(server *srv, connection *con, void *p_d) {
	plugin_data *p = p_d;
	buffer *b;
	size_t j;
	double avg;
	
	b = chunkqueue_get_append_buffer(con->write_queue);
	
	BUFFER_COPY_STRING_CONST(b, 
		      "<?xml version=\"1.0\" encoding=\"iso-8859-1\"?>\n"
		      "<!DOCTYPE html PUBLIC \"-//W3C//DTD XHTML 1.0 Transitional//EN\"\n"
		      "         \"http://www.w3.org/TR/xhtml1/DTD/xhtml1-transitional.dtd\">\n"
		      "<html xmlns=\"http://www.w3.org/1999/xhtml\" xml:lang=\"en\" lang=\"en\">\n"
		      " <head>\n"
		      "  <title>Status</title>\n"
		      " </head>\n"
		      " <body>\n");
	
	/* connection listing */
	BUFFER_APPEND_STRING_CONST(b, "<pre>");
	
	BUFFER_APPEND_STRING_CONST(b, "<b>absolute</b> (since start)\n");
	
	BUFFER_APPEND_STRING_CONST(b, " <b>Requests</b>: ");
	buffer_append_long(b, p->abs_requests);
	BUFFER_APPEND_STRING_CONST(b, " req\n");
	
	BUFFER_APPEND_STRING_CONST(b, " <b>Traffic</b> : ");
	buffer_append_long(b, p->abs_traffic_out / 1024);
	BUFFER_APPEND_STRING_CONST(b, " kbytes\n");
	
	
	BUFFER_APPEND_STRING_CONST(b, "<b>average</b> (5s sliding average)\n");
	for (j = 0, avg = 0; j < 5; j++) {
		avg += p->mod_5s_requests[j];
	}
	
	avg /= 5;
	
	BUFFER_APPEND_STRING_CONST(b, " <b>Requests</b>: ");
	buffer_append_long(b, avg);
	BUFFER_APPEND_STRING_CONST(b, " req/s\n");
	
	for (j = 0, avg = 0; j < 5; j++) {
		avg += p->mod_5s_traffic_out[j];
	}
	
	avg /= 5;
	
	BUFFER_APPEND_STRING_CONST(b, " <b>Traffic</b> : ");
	buffer_append_long(b, avg / 1024);
	BUFFER_APPEND_STRING_CONST(b, " kb/s\n");
	
	
	BUFFER_APPEND_STRING_CONST(b, "\n<b>legend</b>\n");
	BUFFER_APPEND_STRING_CONST(b, ". = connect, C = close, E = hard error\n");
	BUFFER_APPEND_STRING_CONST(b, "r = read, R = read-POST, W = write, h = handle-request\n");
	BUFFER_APPEND_STRING_CONST(b, "q = request-start,  Q = request-end\n");
	BUFFER_APPEND_STRING_CONST(b, "s = response-start, S = response-end\n<hr />\n");
	
	for (j = 0; j < srv->conns->used; j++) {
		connection *c = srv->conns->ptr[j];
		const char *state = connection_get_short_state(c->state);
		
		buffer_append_string_len(b, state, 1);
		
		if (((j + 1) % 50) == 0) {
			BUFFER_APPEND_STRING_CONST(b, "\n");
		}
	}
	
	BUFFER_APPEND_STRING_CONST(b, "\n<hr />\n<b>Connections</b>\n");
	
	for (j = 0; j < srv->conns->used; j++) {
		connection *c = srv->conns->ptr[j];
		
		buffer_append_string(b, inet_ntop_cache_get_ip(srv, &(c->dst_addr)));
	
		BUFFER_APPEND_STRING_CONST(b, " ");
		
		buffer_append_off_t(b, chunkqueue_written(c->write_queue));
		
		BUFFER_APPEND_STRING_CONST(b, " ");
		
		buffer_append_string(b, connection_get_state(c->state));
		
		BUFFER_APPEND_STRING_CONST(b, " ");
		
		buffer_append_string_buffer(b, c->physical.path);
		
		BUFFER_APPEND_STRING_CONST(b, "\n");
	}
	
	
	BUFFER_APPEND_STRING_CONST(b, 
		      "</pre>\n");
	
	
	BUFFER_APPEND_STRING_CONST(b, 
		      " </body>\n"
		      "</html>\n"
		      );
	
	response_header_overwrite(srv, con, CONST_STR_LEN("Content-Type"), CONST_STR_LEN("text/html"));
	
	con->http_status = 200;
	con->file_finished = 1;
	
	return HANDLER_FINISHED;
}

static handler_t mod_status_handle_server_config(server *srv, connection *con, void *p_d) {
	plugin_data *p = p_d;
	buffer *b, *m = p->module_list;
	size_t i;
	
	struct ev_map { fdevent_handler_t et; const char *name; } event_handlers[] = 
	{ 
		/* - poll is most reliable
		 * - select works everywhere
		 * - linux-* are experimental
		 */
#ifdef USE_POLL
		{ FDEVENT_HANDLER_POLL,           "poll" },
#endif
#ifdef USE_SELECT
		{ FDEVENT_HANDLER_SELECT,         "select" },
#endif
#ifdef USE_LINUX_EPOLL
		{ FDEVENT_HANDLER_LINUX_SYSEPOLL, "linux-sysepoll" },
#endif
#ifdef USE_LINUX_SIGIO
		{ FDEVENT_HANDLER_LINUX_RTSIG,    "linux-rtsig" },
#endif
#ifdef USE_SOLARIS_DEVPOLL
		{ FDEVENT_HANDLER_SOLARIS_DEVPOLL,"solaris-devpoll" },
#endif
#ifdef USE_FREEBSD_KQUEUE
		{ FDEVENT_HANDLER_FREEBSD_KQUEUE, "freebsd-kqueue" },
#endif
		{ FDEVENT_HANDLER_UNSET,          NULL }
	};
	
	b = chunkqueue_get_append_buffer(con->write_queue);
	
	BUFFER_COPY_STRING_CONST(b, 
			   "<?xml version=\"1.0\" encoding=\"iso-8859-1\"?>\n"
			   "<!DOCTYPE html PUBLIC \"-//W3C//DTD XHTML 1.0 Transitional//EN\"\n"
			   "         \"http://www.w3.org/TR/xhtml1/DTD/xhtml1-transitional.dtd\">\n"
			   "<html xmlns=\"http://www.w3.org/1999/xhtml\" xml:lang=\"en\" lang=\"en\">\n"
			   " <head>\n"
			   "  <title>Status</title>\n"
			   " </head>\n"
			   " <body>\n"
			   "  <h1>"PACKAGE_NAME" "PACKAGE_VERSION"</h1>\n"
			   "  <table border=\"1\">\n");
	
	mod_status_header_append(b, "Server-Features");
#ifdef HAVE_PCRE_H
	mod_status_row_append(b, "Rewrite Engine", "enabled");
#else
	mod_status_row_append(b, "Rewrite Engine", "disabled - pcre missing");
#endif
#ifdef HAVE_ZLIB_H
	mod_status_row_append(b, "On-the-Fly Output Compression", "enabled");
#else
	mod_status_row_append(b, "On-the-Fly Output Compression", "disabled - zlib missing");
#endif
	mod_status_header_append(b, "Network Engine");
	
	for (i = 0; event_handlers[i].name; i++) {
		if (event_handlers[i].et == srv->event_handler) {
			mod_status_row_append(b, "fd-Event-Handler", event_handlers[i].name);
			break;
		}
	}
	
	mod_status_header_append(b, "Config-File-Settings");
	mod_status_row_append(b, "Directory Listings", con->conf.dir_listing ? "enabled" : "disabled");
	
	for (i = 0; i < srv->plugins.used; i++) {
		plugin **ps = srv->plugins.ptr;
		
		plugin *pl = ps[i];
	
		if (i == 0) {
			buffer_copy_string_buffer(m, pl->name);
		} else {
			BUFFER_APPEND_STRING_CONST(m, "<br />");
			buffer_append_string_buffer(m, pl->name);
		}
	}
	
	mod_status_row_append(b, "Loaded Modules", m->ptr);
	
	BUFFER_APPEND_STRING_CONST(b, "  </table>\n");
	
	BUFFER_APPEND_STRING_CONST(b, 
		      " </body>\n"
		      "</html>\n"
		      );
	
	response_header_overwrite(srv, con, CONST_STR_LEN("Content-Type"), CONST_STR_LEN("text/html"));
	
	con->http_status = 200;
	con->file_finished = 1;
	
	return HANDLER_FINISHED;
}

static handler_t mod_status_handler(server *srv, connection *con, void *p_d) {
	plugin_data *p = p_d;
	
	if (!buffer_is_empty(p->status_url) && 
	    buffer_is_equal(p->status_url, con->uri.path)) {
		return mod_status_handle_server_status(srv, con, p_d);
	} else if (!buffer_is_empty(p->config_url) && 
	    buffer_is_equal(p->config_url, con->uri.path)) {
		return mod_status_handle_server_config(srv, con, p_d);
	}
	
	return HANDLER_GO_ON;
}

TRIGGER_FUNC(mod_status_trigger) {
	plugin_data *p = p_d;
	size_t i;
	
	/* check all connections */
	for (i = 0; i < srv->conns->used; i++) {
		connection *c = srv->conns->ptr[i];
		
		p->bytes_written += c->bytes_written - c->bytes_written_last_sec;
		c->bytes_written_last_sec = c->bytes_written;
	}
	
	/* a sliding average */
	p->mod_5s_traffic_out[p->mod_5s_ndx] = p->bytes_written;
	p->mod_5s_requests   [p->mod_5s_ndx] = p->requests;
	
	p->mod_5s_ndx = (p->mod_5s_ndx+1) % 5;
	
	p->abs_traffic_out += p->bytes_written;
	p->rel_traffic_out += p->bytes_written;
	
	p->bytes_written = 0;
	
	/* reset storage - second */
	p->traffic_out = 0;
	p->requests    = 0;
	
	return HANDLER_GO_ON;
}

REQUESTDONE_FUNC(mod_status_account) {
	plugin_data *p = p_d;
	
	UNUSED(srv);

	p->requests++;
	p->rel_requests++;
	p->abs_requests++;
	
	p->bytes_written += con->bytes_written;
	
	return HANDLER_GO_ON;
}

int mod_status_plugin_init(plugin *p) {
	p->name        = buffer_init_string("status");
	
	p->init        = mod_status_init;
	p->cleanup     = mod_status_free;
	p->set_defaults= mod_status_set_defaults;
	
	p->handle_uri_clean    = mod_status_handler;
	p->handle_trigger      = mod_status_trigger;
	p->handle_request_done = mod_status_account;
	
	p->data        = NULL;
	
	return 0;
}
