# bltHiertable.tcl
# ----------------------------------------------------------------------
# Bindings for the BLT hiertable widget
# ----------------------------------------------------------------------
#   AUTHOR:  George Howlett
#            Bell Labs Innovations for Lucent Technologies
#            gah@lucent.com
#            http://www.tcltk.com/blt
#
#      RCS:  $Id: bltHiertable.tcl,v 1.2 2000/01/24 18:52:30 gah Exp $
#
# ----------------------------------------------------------------------
# Copyright (c) 1998  Lucent Technologies, Inc.
# ======================================================================
#
# Permission to use, copy, modify, and distribute this software and its
# documentation for any purpose and without fee is hereby granted,
# provided that the above copyright notice appear in all copies and that
# both that the copyright notice and warranty disclaimer appear in
# supporting documentation, and that the names of Lucent Technologies
# any of their entities not be used in advertising or publicity
# pertaining to distribution of the software without specific, written
# prior permission.
#
# Lucent Technologies disclaims all warranties with regard to this
# software, including all implied warranties of merchantability and
# fitness.  In no event shall Lucent be liable for any special, indirect
# or consequential damages or any damages whatsoever resulting from loss
# of use, data or profits, whether in an action of contract, negligence
# or other tortuous action, arising out of or in connection with the use
# or performance of this software.
#
# ======================================================================

array set bltHiertable {
    afterId ""
    space   off
    action  ""
    x       0
    y       0
    toggle  yes
    select  off
}

# 
# ButtonPress assignments
#
#	B1-Enter	start auto-scrolling
#	B1-Leave	stop auto-scrolling
#	ButtonPress-2	start scan
#	B2-Motion	adjust scan
#	ButtonRelease-2 stop scan
#

bind Hiertable <ButtonPress-2> {
    set bltHiertable(cursor) [%W cget -cursor]
    %W configure -cursor hand1
    %W scan mark %x %y
}

bind Hiertable <B2-Motion> {
    %W scan dragto %x %y
}

bind Hiertable <ButtonRelease-2> {
    %W configure -cursor $bltHiertable(cursor)
}

bind Hiertable <B1-Leave> {
    blt::HiertableAutoScan %W 
}

bind Hiertable <B1-Enter> {
    after cancel $bltHiertable(afterId)
}


# 
# KeyPress assignments
#
#	Up			
#	Down
#	Shift-Up
#	Shift-Down
#	Prior (PageUp)
#	Next  (PageDn)
#	Left
#	Right
#	space		Start selection toggle of entry currently with focus.
#	Return		Start selection toggle of entry currently with focus.
#	Home
#	End
#	F1
#	F2
#	ASCII char	Go to next open entry starting with character.
#
# KeyRelease
#
#	space		Stop selection toggle of entry currently with focus.
#	Return		Stop selection toggle of entry currently with focus.


bind Hiertable <KeyPress-Up> {
    blt::HiertableTraverse %W up
    if { $bltHiertable(space) } {
	%W selection toggle focus
    }
}

bind Hiertable <KeyPress-Down> {
    blt::HiertableTraverse %W down
    if { $bltHiertable(space) } {
	%W selection toggle focus
    }
}

bind Hiertable <Shift-KeyPress-Up> {
    blt::HiertableTraverse %W prevsibling
}

bind Hiertable <Shift-KeyPress-Down> {
    blt::HiertableTraverse %W nextsibling
}

bind Hiertable <KeyPress-Prior> {
    blt::HiertablePage %W top
}

bind Hiertable <KeyPress-Next> {
    blt::HiertablePage %W bottom
}

bind Hiertable <KeyPress-Left> {
    %W close focus
}
bind Hiertable <KeyPress-Right> {
    %W open focus
    %W see focus -anchor w
}

bind Hiertable <KeyPress-space> {
    blt::HiertableToggle %W focus
    set bltHiertable(space) on
}

bind Hiertable <KeyRelease-space> { 
    set bltHiertable(space) off
}

bind Hiertable <KeyPress-Return> {
    blt::HiertableToggle %W focus
    set bltHiertable(space) on
}

bind Hiertable <KeyRelease-Return> { 
    set bltHiertable(space) off
}

bind Hiertable <KeyPress> {
    blt::HiertableSearch %W %A
}

bind Hiertable <KeyPress-Home> {
    blt::HiertableTraverse %W root
}

bind Hiertable <KeyPress-End> {
    blt::HiertableTraverse %W end
}

bind Hiertable <KeyPress-F1> {
    %W open -r root
}

bind Hiertable <KeyPress-F2> {
    eval %W close -r [%W entry children root 0 end] 
}

# ----------------------------------------------------------------------
#
# USAGE: blt::HiertableInit <hiertable> 
#
# Invoked by internally by Hiertable_Init routine.  Initializes the 
# default bindings for the hiertable widget entries.  These are local
# to the widget, so they can't be set through the widget's class
# bind tags.
#
# Arguments:	hiertable		hierarchy widget
#
# ----------------------------------------------------------------------
proc blt::HiertableInit { widget } {
    $widget bind SelectionMode <Enter> { 
	%W entry activate current 
    }
    $widget bind SelectionMode <Leave> { 
	%W entry activate "" 
    }
    $widget bind SelectionMode <ButtonPress-1> { 	
	blt::HiertableSelect %W 
    }
    $widget bind SelectionMode <Double-ButtonPress-1> {
	%W toggle current
    }
    $widget bind SelectionMode <Shift-ButtonPress-1> { 
	blt::HiertableStartSelect %W "set"
    }
    $widget bind SelectionMode <Control-ButtonPress-1> { 
	blt::HiertableStartSelect %W "toggle"
    }
    $widget bind SelectionMode <B1-Motion> { 
	if { $bltHiertable(action) != "" } {
	    set bltHiertable(select) on
	}
	set bltHiertable(x) %x
	set bltHiertable(y) %y
	blt::HiertableMotion %W %x %y
    }
    $widget bind SelectionMode <ButtonRelease-1> { 
	if { $bltHiertable(action) != "" } {
	    blt::HiertableEndSelect %W %x %y
	}
	after cancel $bltHiertable(afterId)
    }
    #
    # Button bindings
    #
    $widget button bind all <ButtonRelease-1> {
	%W see -anchor nw current
	%W toggle current
    }
    $widget button bind all <Enter> {
	%W button activate current
    }
    $widget button bind all <Leave> {
	%W button activate ""
    }
}

# ----------------------------------------------------------------------
# USAGE: blt::HiertableAutoScan <hiertable>
#
# Invoked when the user is selecting elements in a hiertable widget
# and drags the mouse pointer outside of the widget.  Scrolls the
# view in the direction of the pointer.
#
# Arguments:	hiertable		hierarchy widget
#
# ----------------------------------------------------------------------
proc blt::HiertableAutoScan { widget } {
    global bltHiertable
    if { ![winfo exists $widget] } {
	return
    }
    set x $bltHiertable(x)
    set y $bltHiertable(y)
    if { $y >= [winfo height $widget] } {
	$widget yview scroll 1 units
    } elseif { $y < 0 } {
	$widget yview scroll -1 units
    } 
    blt::HiertableMotion $widget $x $y
    set bltHiertable(afterId) [after 10 blt::HiertableAutoScan $widget]
}

# ----------------------------------------------------------------------
# USAGE blt::HiertableSelect <hiertable> 
#
# Invoked when the user clicks on a <hiertable> widget. The currently
# picked entry is available using the "current" index. The selection
# is cleared and the element at that coordinate is selected.  
#
# Arguments:	hiertable		hierarchy widget
#
# ----------------------------------------------------------------------
proc blt::HiertableSelect { widget } {
    global bltHiertable

    set bltHiertable(action) ""

    # Clear all selections before setting the new anchor.
    $widget selection clear 0 end
    $widget selection set current
    $widget selection anchor current

    $widget see current
    $widget focus current
}

# ----------------------------------------------------------------------
# USAGE: blt::HiertableStartSelect <hiertable> <action>
#
# Invoked when the user clicks on a <hiertable> widget at the screen
# coordinate <x>,<y>.  Marks the start of a "drag" operation for
# selecting multiple elements.  See related HiertableMotion and
# HiertableEndSelect procedures.
#
# Arguments:	hiertable		hierarchy widget
#		action		"set", "clear", "toggle", or "".  
#				Indicates how to manage the selection.
#
# ----------------------------------------------------------------------
proc blt::HiertableStartSelect { widget action } {
    global bltHiertable

    set bltHiertable(action) $action
    switch -- [$widget cget -selectmode] {
	single - active {
	    blt::HiertableSelect $widget
	}
	multiple {
	    $widget selection anchor current
	}
    }
}

# ----------------------------------------------------------------------
# USAGE:	blt::HiertableMotion 
#
# Invoked when the user has clicked on a <hiertable> widget and has
# dragged the mouse to the coordinate <x>,<y>.  Updates a preview
# of the elements being selected.  See related HiertableStartSelect
# and HiertableEndSelect procedures.
#
# Arguments:	hiertable		hierarchy widget
#		x		X-coordinate of mouse pointer
#		y		Y-coordinate of mouse pointer
#
# ----------------------------------------------------------------------
proc blt::HiertableMotion { widget x y } {
    global bltHiertable

    if { $bltHiertable(action) == "" } {
	return
    }
    if {[$widget cget -selectmode] == "multiple" } {
	set node [$widget nearest $x $y]
	$widget selection dragto $node $bltHiertable(action)
    }
}

# ----------------------------------------------------------------------
# USAGE: blt::HiertableEndSelect <hiertable> <x> <y>
#
# Invoked when the user has clicked on a <hiertable> widget and has
# then released the mouse at coordinate <x>,<y>.  Finalizes the
# current selection.  See related HiertableStartSelect and HiertableMotion
# procedures.
#
# Arguments:	hiertable		hierarchy widget
#		x		X-coordinate of mouse pointer
#		y		Y-coordinate of mouse pointer
#
# ----------------------------------------------------------------------
proc blt::HiertableEndSelect { widget x y } {
    global bltHiertable

    if { $bltHiertable(action) != "" } {
	after cancel $bltHiertable(afterId)
    }
    if {[$widget cget -selectmode] == "multiple" } {
	set node [$widget nearest $x $y]
        $widget selection $bltHiertable(action) anchor $node
        $widget selection anchor $node
    }
    set bltHiertable(select) "off"
}

# ----------------------------------------------------------------------
# USAGE: blt::HiertableToggle <hiertable> <index>
# Arguments:	hiertable		hierarchy widget
#
# Invoked when the user presses the space bar.  Toggles the selection
# for the entry at <index>.
# ----------------------------------------------------------------------
proc blt::HiertableToggle { widget index } {
    switch -- [$widget cget -selectmode] {
        single - active {
            set i [$widget index $index]
            if {[lsearch [$widget curselection] $i] < 0} {
                $widget selection clear 0 end
            }
            $widget selection toggle $index
        }
        multiple {
            $widget selection toggle $index
        }
    }
}

# ----------------------------------------------------------------------
# USAGE: blt::HiertableTraverse <hiertable> <where>
#
# Invoked by KeyPress bindings.  Moves the active selection to the
# entry <where>, which is an index such as "up", "down", "prevsibling",
# "nextsibling", etc.
# ----------------------------------------------------------------------
proc blt::HiertableTraverse { widget where } {
    catch {$widget focus $where}
    if { [$widget cget -selectmode] == "active" } {
        $widget selection clear 0 end
        $widget selection set focus
    }
    $widget see focus
}

# ----------------------------------------------------------------------
# USAGE: blt::HiertablePage <hiertable> <where>
# Arguments:	hiertable		hierarchy widget
#
# Invoked by KeyPress bindings.  Pages the current view up or down.
# The <where> argument should be either "top" or "bottom".
# ----------------------------------------------------------------------
proc blt::HiertablePage { widget where } {
    if { [$widget index focus] == [$widget index view.$where] } {
        if {$where == "top"} {
	    $widget yview scroll -1 pages
	    $widget yview scroll 1 units
        } else {
	    $widget yview scroll 1 pages
	    $widget yview scroll -1 units
        }
    }
    update
    #
    # Adjust the entry focus and the view.  Also activate the entry.
    # just in case the mouse point is not in the widget.
    #
    $widget entry activate view.$where
    $widget focus view.$where
    $widget see view.$where
    if { [$widget cget -selectmode] == "active" } {
        $widget selection clear 0 end
        $widget selection set focus
    }
}

# ----------------------------------------------------------------------
# USAGE: blt::HiertableSearch <hiertable> <char>
# Arguments:	hiertable		hierarchy widget
#
# Invoked by KeyPress bindings.  Searches for an entry that starts
# with the letter <char> and makes that entry active.
# ----------------------------------------------------------------------

proc blt::HiertableSearch { widget key } {
    if {[string match {[ -~]} $key]} {
	set last [$widget index focus]
	set next [$widget index next]
	while { $next != $last } {
	    set label [$widget entry cget $next -label]
	    if { [string index $label 0] == $key } {
		break
	    }
	    set next [$widget index -at $next next]
	}
	$widget focus $next
        if {[$widget cget -selectmode] == "active"} {
            $widget selection clear 0 end
            $widget selection set focus
        }
	$widget see focus
    }
}

#
#  Edit mode assignments
#
#	ButtonPress-3   Enables/disables edit mode on entry.  Sets focus to 
#			entry.
#
#  KeyPress
#
#	Left		Move insertion position to previous.
#	Right		Move insertion position to next.
#	Up		Move insertion position up one line.
#	Down		Move insertion position down one line.
#	Return		End edit mode.
#	Shift-Return	Line feed.
#	Home		Move to first position.
#	End		Move to last position.
#	ASCII char	Insert character left of insertion point.
#	Del		Delete character right of insertion point.
#	Delete		Delete character left of insertion point.
#	Ctrl-X		Cut
#	Ctrl-V		Copy
#	Ctrl-P		Paste
#	
#  KeyRelease
#
#  ButtonPress-1	Start selection if in entry, otherwise clear selection.
#  B1-Motion		Extend/reduce selection.
#  ButtonRelease-1      End selection if in entry, otherwise use last selection.
#  B1-Enter		Disabled.
#  B1-Leave		Disabled.
#  ButtonPress-2	Same as above.
#  B2-Motion		Same as above.
#  ButtonRelease-2	Same as above.
#	
# All bindings in editting mode will "break" to override other bindings.
#
#

bind xEditor <ButtonPress-3> {
    set node [%W nearest %x %y]
    %W entry insert $node @%x,%y ""
#    %W entry insert $node 2 ""
}

image create photo blt::HiertableCloseNormalFolder -format gif -data {
R0lGODlhEAANAPIAAAAAAH9/f7+/v///////AAAAAAAAAAAAACH+JEZpbGUgd3JpdHRlbiBi
eSBBZG9iZSBQaG90b3Nob3CoIDUuMAAsAAAAABAADQAAAzk4Gsz6cIQ44xqCZCGbk4MmclAA
gNs4ml7rEaxVAkKc3gTAnBO+sbyQT6M7gVQpk9HlAhgHzqhUmgAAOw==
}
image create photo blt::HiertableOpenNormalFolder -format gif -data {
R0lGODlhEAANAPIAAAAAAH9/f7+/v///AP///wAAAAAAAAAAACH+JEZpbGUgd3JpdHRlbiBi
eSBBZG9iZSBQaG90b3Nob3CoIDUuMAAsAAAAABAADQAAAzNIGsz6kAQxqAjxzcpvc1KWBUDY
nRQZWmilYi37EmztlrAt43R8mzrO60P8lAiApHK5TAAAOw==
}
image create photo blt::HiertableCloseActiveFolder -format gif -data {
R0lGODlhEAANAPIAAAAAAH9/f7+/v/////+/AAAAAAAAAAAAACH+JEZpbGUgd3JpdHRlbiBi
eSBBZG9iZSBQaG90b3Nob3CoIDUuMAAsAAAAABAADQAAAzk4Gsz6cIQ44xqCZCGbk4MmclAA
gNs4ml7rEaxVAkKc3gTAnBO+sbyQT6M7gVQpk9HlAhgHzqhUmgAAOw==
}
image create photo blt::HiertableOpenActiveFolder -format gif -data {
R0lGODlhEAANAPIAAAAAAH9/f7+/v/+/AP///wAAAAAAAAAAACH+JEZpbGUgd3JpdHRlbiBi
eSBBZG9iZSBQaG90b3Nob3CoIDUuMAAsAAAAABAADQAAAzNIGsz6kAQxqAjxzcpvc1KWBUDY
nRQZWmilYi37EmztlrAt43R8mzrO60P8lAiApHK5TAAAOw==
}

# Standard Motif bindings:

bind HiertableEditor <ButtonPress-1> {
    [winfo parent %W] text icursor @%x,%y
}

bind HiertableEditor <Left> {
    [winfo parent %W] text icursor last
}
bind HiertableEditor <Right> {
    [winfo parent %W] text icursor next
}
bind HiertableEditor <Shift-Left> {
    tkEntryKeySelect %W [expr {[%W index insert] - 1}]
    tkEntrySeeInsert %W
}
bind HiertableEditor <Shift-Right> {
    tkEntryKeySelect %W [expr {[%W index insert] + 1}]
    tkEntrySeeInsert %W
}

bind HiertableEditor <Home> {
    [winfo parent %W] text icursor 0
}
bind HiertableEditor <Shift-Home> {
    tkEntryKeySelect %W 0
    tkEntrySeeInsert %W
}
bind HiertableEditor <End> {
    [winfo parent %W] text icursor end
}
bind HiertableEditor <Shift-End> {
    tkEntryKeySelect %W end
    tkEntrySeeInsert %W
}

bind HiertableEditor <Delete> {
    if {[[winfo parent %W] text selection present]} {
	[winfo parent %W] delete sel.first sel.last
    } else {
	[winfo parent %W] delete insert
    }
}
bind HiertableEditor <BackSpace> {
    blt::HiertableEditorBackspace [winfo parent %W]
}

bind HiertableEditor <Control-space> {
    [winfo parent %W] text selection from insert
}
bind HiertableEditor <Select> {
    [winfo parent %W] text selection from insert
}
bind HiertableEditor <Control-Shift-space> {
    [winfo parent %W] text selection adjust insert
}
bind HiertableEditor <Shift-Select> {
    [winfo parent %W] text selection adjust insert
}
bind HiertableEditor <Control-slash> {
    [winfo parent %W] text selection range 0 end
}
bind HiertableEditor <Control-backslash> {
    [winfo parent %W] text selection clear
}
bind HiertableEditor <KeyPress> {
    blt::HiertableInsert [winfo parent %W] %A
}

# Ignore all Alt, Meta, and Control keypresses unless explicitly bound.
# Otherwise, if a widget binding for one of these is defined, the
# <KeyPress> class binding will also fire and insert the character,
# which is wrong.  Ditto for Escape, Return, and Tab.

bind HiertableEditor <Alt-KeyPress> {# nothing}
bind HiertableEditor <Meta-KeyPress> {# nothing}
bind HiertableEditor <Control-KeyPress> {# nothing}
bind HiertableEditor <Escape> {# nothing}
bind HiertableEditor <Shift-Return> {
    blt::HiertableInsert [winfo parent %W] "\n"
}
bind HiertableEditor <KP_Enter> {# nothing}
bind HiertableEditor <Tab> {# nothing}
if {![string compare $tcl_platform(platform) "macintosh"]} {
	bind HiertableEditor <Command-KeyPress> {# nothing}
}

# On Windows, paste is done using Shift-Insert.  Shift-Insert already
# generates the <<Paste>> event, so we don't need to do anything here.
if {[string compare $tcl_platform(platform) "windows"]} {
    bind HiertableEditor <Insert> {
	catch {tkEntryInsert %W [selection get -displayof %W]}
    }
}

# Additional emacs-like bindings:

bind HiertableEditor <Control-a> {
    tkEntrySetCursor %W 0
}
bind HiertableEditor <Control-b> {
    tkEntrySetCursor %W [expr {[%W index insert] - 1}]
}

bind HiertableEditor <Control-d> {
    %W text delete insert
}
bind HiertableEditor <Control-e> {
    tkEntrySetCursor %W end
}
bind HiertableEditor <Control-f> {
    tkEntrySetCursor %W [expr {[%W index insert] + 1}]
}
bind HiertableEditor <Control-h> {
    blt::HiertableEditorBackspace [winfo parent %W]
}
bind HiertableEditor <Control-k> {
    %W text delete insert end
}
bind HiertableEditor <Control-t> {
    tkEntryTranspose %W
}
bind HiertableEditor <Meta-b> {
    tkEntrySetCursor %W [tkEntryPreviousWord %W insert]
}
bind HiertableEditor <Meta-d> {
    %W delete insert [tkEntryNextWord %W insert]
}
bind Entry <Meta-f> {
    tkEntrySetCursor %W [tkEntryNextWord %W insert]
}
bind Entry <Meta-BackSpace> {
    %W delete [tkEntryPreviousWord %W insert] insert
}
bind Entry <Meta-Delete> {
    %W delete [tkEntryPreviousWord %W insert] insert
}


proc tkEntryKeySelect {w new} {
    if {![$w selection present]} {
	$w selection from insert
	$w selection to $new
    } else {
	$w selection adjust $new
    }
    $w icursor $new
}

# blt::HiertableInsert --
# Insert a string into an entry at the point of the insertion cursor.
# If there is a selection in the entry, and it covers the point of the
# insertion cursor, then delete the selection before inserting.
#
# Arguments:
# w -		The entry window in which to insert the string
# s -		The string to insert (usually just a single character)

proc blt::HiertableInsert {w s} {
    if {![string compare $s ""]} {
	return
    }
    $w text insert insert $s
}

proc blt::xHiertableInsert {w s} {
    if {![string compare $s ""]} {
	return
    }
    catch {
	set insert [$w text index insert]
	if {([$w text index sel.first] <= $insert)
		&& ([$w text index sel.last] >= $insert)} {
	    $w delete sel.first sel.last
	}
    }
    $w insert insert $s
    tkEntrySeeInsert $w
}

# tkEntryBackspace --
# Backspace over the character just before the insertion cursor.
# If backspacing would move the cursor off the left edge of the
# window, reposition the cursor at about the middle of the window.
#
# Arguments:
# w -		The entry window in which to backspace.

proc blt::HiertableEditorBackspace w {
    if {[$w text selection present]} {
	$w text delete sel.first sel.last
    } else {
	set index [expr [$w text index insert] - 1]
	if { $index >= 0 } {
	    $w text delete $index $index
	}
    }
}

proc tkEntryBackspace w {
    if {[$w selection present]} {
	$w delete sel.first sel.last
    } else {
	set x [expr {[$w index insert] - 1}]
	if {$x >= 0} {$w delete $x}
	if {[$w index @0] >= [$w index insert]} {
	    set range [$w xview]
	    set left [lindex $range 0]
	    set right [lindex $range 1]
	    $w xview moveto [expr {$left - ($right - $left)/2.0}]
	}
    }
}

# tkEntrySeeInsert --
# Make sure that the insertion cursor is visible in the entry window.
# If not, adjust the view so that it is.
#
# Arguments:
# w -		The entry window.

proc tkEntrySeeInsert w {
    set c [$w index insert]
    set left [$w index @0]
    if {$left > $c} {
	$w xview $c
	return
    }
    set x [winfo width $w]
    if {$c > [$w index @[winfo width $w]]} {
	$w xview insert
    }
}

# tkEntrySetCursor -
# Move the insertion cursor to a given position in an entry.  Also
# clears the selection, if there is one in the entry, and makes sure
# that the insertion cursor is visible.
#
# Arguments:
# w -		The entry window.
# pos -		The desired new position for the cursor in the window.

proc tkEntrySetCursor {w pos} {
    $w icursor $pos
    $w selection clear
    tkEntrySeeInsert $w
}

# tkEntryTranspose -
# This procedure implements the "transpose" function for entry widgets.
# It tranposes the characters on either side of the insertion cursor,
# unless the cursor is at the end of the line.  In this case it
# transposes the two characters to the left of the cursor.  In either
# case, the cursor ends up to the right of the transposed characters.
#
# Arguments:
# w -		The entry window.

proc tkEntryTranspose w {
    set i [$w index insert]
    if {$i < [$w index end]} {
	incr i
    }
    set first [expr {$i-2}]
    if {$first < 0} {
	return
    }
    set new [string index [$w get] [expr {$i-1}]][string index [$w get] $first]
    $w delete $first $i
    $w insert insert $new
    tkEntrySeeInsert $w
}

# tkEntryNextWord --
# Returns the index of the next word position after a given position in the
# entry.  The next word is platform dependent and may be either the next
# end-of-word position or the next start-of-word position after the next
# end-of-word position.
#
# Arguments:
# w -		The entry window in which the cursor is to move.
# start -	Position at which to start search.

if {![string compare $tcl_platform(platform) "windows"]}  {
    proc tkEntryNextWord {w start} {
	set pos [tcl_endOfWord [$w get] [$w index $start]]
	if {$pos >= 0} {
	    set pos [tcl_startOfNextWord [$w get] $pos]
	}
	if {$pos < 0} {
	    return end
	}
	return $pos
    }
} else {
    proc tkEntryNextWord {w start} {
	set pos [tcl_endOfWord [$w get] [$w index $start]]
	if {$pos < 0} {
	    return end
	}
	return $pos
    }
}

# tkEntryPreviousWord --
#
# Returns the index of the previous word position before a given
# position in the entry.
#
# Arguments:
# w -		The entry window in which the cursor is to move.
# start -	Position at which to start search.

proc tkEntryPreviousWord {w start} {
    set pos [tcl_startOfPreviousWord [$w get] [$w index $start]]
    if {$pos < 0} {
	return 0
    }
    return $pos
}
# tkEntryGetSelection --
#
# Returns the selected text of the entry with respect to the -show option.
#
# Arguments:
# w -         The entry window from which the text to get

proc tkEntryGetSelection {w} {
    set entryString [string range [$w get] [$w index sel.first] \
                       [expr [$w index sel.last] - 1]]
    if {[$w cget -show] != ""} {
      regsub -all . $entryString [string index [$w cget -show] 0] entryString
    }
    return $entryString
}
