
/*
 * bltColor.c --
 *
 *	This module contains routines for color allocation strategies
 *	used with color images in the BLT toolkit.
 *
 * Copyright 1997-1998 Lucent Technologies, Inc.
 *
 * Permission to use, copy, modify, and distribute this software and
 * its documentation for any purpose and without fee is hereby
 * granted, provided that the above copyright notice appear in all
 * copies and that both that the copyright notice and warranty
 * disclaimer appear in supporting documentation, and that the names
 * of Lucent Technologies any of their entities not be used in
 * advertising or publicity pertaining to distribution of the software
 * without specific, written prior permission.
 *
 * Lucent Technologies disclaims all warranties with regard to this
 * software, including all implied warranties of merchantability and
 * fitness.  In no event shall Lucent Technologies be liable for any
 * special, indirect or consequential damages or any damages
 * whatsoever resulting from loss of use, data or profits, whether in
 * an action of contract, negligence or other tortuous action, arising
 * out of or in connection with the use or performance of this
 * software.
 */

/*
 * Color strategies of 8-bit visuals:
 *
 * Try to "best" represent an N-color image into 8-bit (256 color)
 * colormap.  The simplest method is use the high bits of each RGB
 * value (3 bits for red and green, 2 bits for blue).  But this
 * produces lots of contouring since the distribution of colors tends
 * to be clustered.  Other problems: probably can't allocate even 256
 * colors. Other applications will have already taken some color
 * slots.  Furthermore, we might be displaying several images, and we
 * can't assume that all images are representative of the colors used.
 *
 * If we use a private colormap, we may want to allocate some number
 * of colors from the default colormap to prevent flashing when
 * colormaps are switched.
 *
 * Switches:
 *
 *	-exact boolean		Try to match the colors of the image rather
 *				then generating a "best" color ramp.
 *
 *	-threshold value	Maximum average error. Indicates how far
 *				to reduce the quantized color palette.
 *
 *	-tolerance value	Allow colors within this distance to match.
 *				This will weight allocation towards harder
 *				to match colors, rather than the most
 *				frequent.
 *
 *	-mincolors number	Minimum number of reduced quantized colors.
 *				or color ramp.
 *
 *	-dither	boolean		Turn on/off Floyd-Steinberg dithering.
 *
 *	-keep number		Hint to keep the first N colors in the
 *				in the default colormap.  This only applies to
 *				private colormaps.
 *
 *	-ramp number		Number of colors to use in a linear ramp.
 *
 */

#include "bltInt.h"

#ifndef WIN32

#include "bltImage.h"

#define NCOLORS		256

#define	RED	0
#define	GREEN	1
#define BLUE	2

#define R0	(cubePtr->r0)
#define R1	(cubePtr->r1)
#define G0	(cubePtr->g0)
#define G1	(cubePtr->g1)
#define B0	(cubePtr->b0)
#define B1	(cubePtr->b1)

typedef struct box {
    int r0, r1;			/* min, max values: min exclusive max inclusive */
    int g0, g1;
    int b0, b1;
    int vol;
} Cube;

typedef struct RGB {
    unsigned short int red;
    unsigned short int green;
    unsigned short int blue;
} RGB;

/*
 *----------------------------------------------------------------------
 *
 * Histogram is in elements 1..HISTSIZE along each axis,
 * element 0 is for base or marginal value
 * NB: these must start out 0!
 *----------------------------------------------------------------------
 */

typedef struct ColorStats {
    float gm2[33][33][33];
    long int wt[33][33][33];
    long int mR[33][33][33];
    long int mG[33][33][33];
    long int mB[33][33][33];
} ColorStats;

typedef struct ColorControl {
    XColor usedColors[256];	/* Colors already use, queried from the
				 * current colormap. */
    int numUsedColors;		/* Number of color slots used. */
    int numFreeColors;		/* Number of free slots in the color map */

    int freqArr[NCOLORS];	/* Frequency of quantized colors used */

    int acceptError;		/* Threshold error value, to accept or not
				 * accept a colormap.  The error is the
				 * sum of the distances */

    int useExact;		/* If non-zero, indicates to use the colors
				 * from the image, not a linear color ramp */

    int numKeepColors;		/* When using a private colormap, these are
				 * the number of colors to "keep" from the
				 * default colormap, to limit flashing. */
    int useBest;

    int numWorstColors;		/* Number of the worst matching colors to
				 * allocate before all others. */
} ColorControl;


static int
BuildColorRamp(palettePtr, rBits, gBits, bBits)
    register RGB *palettePtr;
    unsigned int rBits, gBits, bBits;
{
    register unsigned int r, g, b;
    register int count;
    unsigned int short rVal, gVal, bVal;

    count = 0;
    for (r = 0; r < rBits; r++) {
	rVal = (r * USHRT_MAX) / (rBits - 1);
	for (g = 0; g < gBits; g++) {
	    gVal = (g * USHRT_MAX) / (gBits - 1);
	    for (b = 0; b < bBits; b++) {
		bVal = (b * USHRT_MAX) / (bBits - 1);
		palettePtr->red = (unsigned short int)rVal;
		palettePtr->green = (unsigned short int)gVal;
		palettePtr->blue = (unsigned short int)bVal;
		palettePtr++, count++;
	    }
	}
    }
    return count;
}

/*
 *----------------------------------------------------------------------
 *
 * QueryColormap --
 *
 *	Fills an array or XColors with the color values (RGB and pixel)
 *	currently allocated in the colormap.
 *
 * Results:
 *	The number of colors allocated is returned. The array "colorArr"
 *	will contain the XColor values of each color in the colormap.
 *
 *----------------------------------------------------------------------
 */

static int
QueryColormap(display, colorMap, mapColors, numMapColorsPtr)
    Display *display;
    Colormap colorMap;
    XColor mapColors[];
    int *numMapColorsPtr;
{
    unsigned long int pixelValues[NCOLORS];
    int numAvail, numMapColors;
    register int i;
    register XColor *colorPtr;
    register unsigned long *indexPtr;
    int inUse[NCOLORS];

    /* Initially, we assume all color cells are allocated. */
    memset((char *)inUse, 0, sizeof(int) * NCOLORS);

    /*
     * Start allocating color cells.  This will tell us which color cells
     * haven't already been allocated in the colormap.  We'll release the
     * cells as soon as we find out how many there are.
     */
    numAvail = 0;
    for (indexPtr = pixelValues, i = 0; i < NCOLORS; i++, indexPtr++) {
	if (!XAllocColorCells(display, colorMap, False, NULL, 0, indexPtr, 1)) {
	    break;
	}
	inUse[*indexPtr] = TRUE;/* Indicate the cell is unallocated */
	numAvail++;
    }
    XFreeColors(display, colorMap, pixelValues, numAvail, 0);

    /*
     * Put the indices of the cells already allocated into a color array.
     * We'll use the array to query the RGB values of the allocated colors.
     */
    numMapColors = 0;
    colorPtr = mapColors;
    for (i = 0; i < NCOLORS; i++) {
	if (!inUse[i]) {
	    colorPtr->pixel = i;
	    colorPtr->flags = (DoRed | DoGreen | DoBlue);
	    colorPtr++, numMapColors++;
	}
    }
    XQueryColors(display, colorMap, mapColors, numMapColors);
    *numMapColorsPtr = numMapColors;
#ifndef notdef
    fprintf(stderr, "Number of colors (allocated/free) %d/%d\n", numMapColors,
	numAvail);
#endif
    return numAvail;
}

/*
 * Build 3-D color histogram of counts, R/G/B, c^2
 */
static void
Hist3d(statsPtr, image)
    ColorStats *statsPtr;
    Colorimage image;
{
    register int r, g, b;
    float table[NCOLORS];
    int numPixels;
    Pix32 *pixelPtr;
    register int i;

    /* Precompute table of squares. */
    for (i = 0; i < NCOLORS; i++) {
	table[i] = i * i;
    }
    numPixels = ColorimageWidth(image) * ColorimageHeight(image);
    pixelPtr = ColorimageData(image);
    for (i = 0; i < numPixels; i++, pixelPtr++) {
	/*
	 * Reduce the number of bits (5) per color component. This
	 * will keep the table size (2^15) reasonable without perceptually
	 * affecting the final image.
	 */
	r = (pixelPtr->Red >> 3) + 1;
	g = (pixelPtr->Green >> 3) + 1;
	b = (pixelPtr->Blue >> 3) + 1;
	statsPtr->wt[r][g][b] += 1;
	statsPtr->mR[r][g][b] += pixelPtr->Red;
	statsPtr->mG[r][g][b] += pixelPtr->Green;
	statsPtr->mB[r][g][b] += pixelPtr->Blue;
	statsPtr->gm2[r][g][b] +=
	    table[pixelPtr->Red] + table[pixelPtr->Green] + table[pixelPtr->Blue];
    }
}

/*
 *----------------------------------------------------------------------
 * At conclusion of the histogram step, we can interpret
 *   wt[r][g][b] = sum over voxel of P(c)
 *   mR[r][g][b] = sum over voxel of r*P(c)  ,  similarly for mG, mB
 *   m2[r][g][b] = sum over voxel of c^2*P(c)
 * Actually each of these should be divided by 'size' to give the usual
 * interpretation of P() as ranging from 0 to 1, but we needn't do that here.
 *----------------------------------------------------------------------
 */

/*
 *----------------------------------------------------------------------
 We now convert histogram into moments so that we can rapidly calculate
 * the sums of the above quantities over any desired box.
 *----------------------------------------------------------------------
 */

static void
M3d(wt, mR, mG, mB, gm2)	/* compute cumulative moments. */
    long int wt[33][33][33];	/* # pixels in voxel */
    long int mR[33][33][33];	/* Sum over voxel of red pixel values */
    long int mG[33][33][33];	/* Sum over voxel of green pixel values */
    long int mB[33][33][33];	/* Sum over voxel of blue pixel values */
    float gm2[33][33][33];	/* Variance */
{
    register unsigned char i, r, g, b;
    long int line, rLine, gLine, bLine;
    long int area[33], rArea[33], gArea[33], bArea[33];
    float line2, area2[33];

    for (r = 1; r <= 32; r++) {
	for (i = 0; i <= 32; ++i) {
	    area2[i] = area[i] = rArea[i] = gArea[i] = bArea[i] = 0;
	}
	for (g = 1; g <= 32; g++) {
	    line2 = line = rLine = gLine = bLine = 0;
	    for (b = 1; b <= 32; b++) {
		/* ind1 = RGBIndex(r, g, b); */

		line += wt[r][g][b];
		rLine += mR[r][g][b];
		gLine += mG[r][g][b];
		bLine += mB[r][g][b];
		line2 += gm2[r][g][b];

		area[b] += line;
		rArea[b] += rLine;
		gArea[b] += gLine;
		bArea[b] += bLine;
		area2[b] += line2;

		/* ind2 = ind1 - 1089; [r-1][g][b] */
		wt[r][g][b] = wt[r - 1][g][b] + area[b];
		mR[r][g][b] = mR[r - 1][g][b] + rArea[b];
		mG[r][g][b] = mG[r - 1][g][b] + gArea[b];
		mB[r][g][b] = mB[r - 1][g][b] + bArea[b];
		gm2[r][g][b] = gm2[r - 1][g][b] + area2[b];
	    }
	}
    }
}

/*
 *----------------------------------------------------------------------
 *
 *	Compute sum over a box of any given statistic
 *
 *----------------------------------------------------------------------
 */
static INLINE long int
Vol(cubePtr, mmt)
    struct box *cubePtr;
    long int mmt[33][33][33];
{
    return (mmt[R1][G1][B1] - mmt[R1][G1][B0] -
	mmt[R1][G0][B1] + mmt[R1][G0][B0] -
	mmt[R0][G1][B1] + mmt[R0][G1][B0] +
	mmt[R0][G0][B1] - mmt[R0][G0][B0]);
}

/*
 *----------------------------------------------------------------------
 *
 *	The next two routines allow a slightly more efficient
 *	calculation of Vol() for a proposed subbox of a given box.
 *	The sum of Top() and Bottom() is the Vol() of a subbox split
 *	in the given direction and with the specified new upper
 *	bound.
 *
 *----------------------------------------------------------------------
 */

/* Compute part of Vol(cubePtr, mmt) that doesn't depend on r1, g1, or b1 */
/* (depending on dir) */
static long int
Bottom(cubePtr, dir, mmt)
    struct box *cubePtr;
    unsigned char dir;
    long int mmt[33][33][33];
{
    switch (dir) {
    case RED:
	return (-mmt[R0][G1][B1] + mmt[R0][G1][B0] + mmt[R0][G0][B1] -
	    mmt[R0][G0][B0]);

    case GREEN:
	return (-mmt[R1][G0][B1] + mmt[R1][G0][B0] + mmt[R0][G0][B1] -
	    mmt[R0][G0][B0]);

    case BLUE:
	return (-mmt[R1][G1][B0] + mmt[R1][G0][B0] + mmt[R0][G1][B0] -
	    mmt[R0][G0][B0]);
    }
    return 0;
}

/*
 *----------------------------------------------------------------------
 *
 * Compute remainder of Vol(cubePtr, mmt), substituting pos for
 * r1, g1, or b1 (depending on dir)
 *
 *----------------------------------------------------------------------
 */
static long int
Top(cubePtr, dir, pos, mmt)
    struct box *cubePtr;
    unsigned char dir;
    int pos;
    long int mmt[33][33][33];
{
    switch (dir) {
    case RED:
	return (mmt[pos][G1][B1] - mmt[pos][G1][B0] -
	    mmt[pos][G0][B1] + mmt[pos][G0][B0]);

    case GREEN:
	return (mmt[R1][pos][B1] - mmt[R1][pos][B0] -
	    mmt[R0][pos][B1] + mmt[R0][pos][B0]);

    case BLUE:
	return (mmt[R1][G1][pos] - mmt[R1][G0][pos] -
	    mmt[R0][G1][pos] + mmt[R0][G0][pos]);
    }
    return 0;
}

/*
 *----------------------------------------------------------------------
 *
 *	Compute the weighted variance of a box NB: as with the raw
 *	statistics, this is really the (variance * size)
 *
 *----------------------------------------------------------------------
 */
static float
Var(cubePtr, statsPtr)
    struct box *cubePtr;
    ColorStats *statsPtr;
{
    float dR, dG, dB, xx;

    dR = Vol(cubePtr, statsPtr->mR);
    dG = Vol(cubePtr, statsPtr->mG);
    dB = Vol(cubePtr, statsPtr->mB);
    xx = (statsPtr->gm2[R1][G1][B1] - statsPtr->gm2[R1][G1][B0] -
	statsPtr->gm2[R1][G0][B1] + statsPtr->gm2[R1][G0][B0] -
	statsPtr->gm2[R0][G1][B1] + statsPtr->gm2[R0][G1][B0] +
	statsPtr->gm2[R0][G0][B1] - statsPtr->gm2[R0][G0][B0]);

    return (xx - (dR * dR + dG * dG + dB * dB) / (float)Vol(cubePtr, statsPtr->wt));
}

/*
 *----------------------------------------------------------------------
 *
 *	We want to minimize the sum of the variances of two subboxes.
 *	The sum(c^2) terms can be ignored since their sum over both
 *	subboxes is the same (the sum for the whole box) no matter
 *	where we split.  The remaining terms have a minus sign in
 *	the variance formula, so we drop the minus sign and MAXIMIZE
 *	the sum of the two terms.
 *
 *----------------------------------------------------------------------
 */
static float
Maximize(cubePtr, dir, first, last, cut, rWhole, gWhole, bWhole, wWhole, statsPtr)
    struct box *cubePtr;
    unsigned char dir;
    int first, last, *cut;
    long int rWhole, gWhole, bWhole, wWhole;
    ColorStats *statsPtr;
{
    register long int rHalf, gHalf, bHalf, wHalf;
    long int rBase, gBase, bBase, wBase;
    register int i;
    register float temp, max;

    rBase = Bottom(cubePtr, dir, statsPtr->mR);
    gBase = Bottom(cubePtr, dir, statsPtr->mG);
    bBase = Bottom(cubePtr, dir, statsPtr->mB);
    wBase = Bottom(cubePtr, dir, statsPtr->wt);
    max = 0.0;
    *cut = -1;
    for (i = first; i < last; i++) {
	rHalf = rBase + Top(cubePtr, dir, i, statsPtr->mR);
	gHalf = gBase + Top(cubePtr, dir, i, statsPtr->mG);
	bHalf = bBase + Top(cubePtr, dir, i, statsPtr->mB);
	wHalf = wBase + Top(cubePtr, dir, i, statsPtr->wt);

	/* Now half_x is sum over lower half of box, if split at i */
	if (wHalf == 0) {	/* subbox could be empty of pixels! */
	    continue;		/* never split into an empty box */
	} else {
	    temp = ((float)rHalf * rHalf + (float)gHalf * gHalf +
		(float)bHalf * bHalf) / wHalf;
	}
	rHalf = rWhole - rHalf;
	gHalf = gWhole - gHalf;
	bHalf = bWhole - bHalf;
	wHalf = wWhole - wHalf;
	if (wHalf == 0) {	/* Subbox could be empty of pixels! */
	    continue;		/* never split into an empty box */
	} else {
	    temp += ((float)rHalf * rHalf + (float)gHalf * gHalf +
		(float)bHalf * bHalf) / wHalf;
	}
	if (temp > max) {
	    max = temp;
	    *cut = i;
	}
    }
    return max;
}

/*
 *----------------------------------------------------------------------
 *----------------------------------------------------------------------
 */
static int
Cut(set1, set2, statsPtr)
    struct box *set1, *set2;
    ColorStats *statsPtr;
{
    unsigned char dir;
    int rCut, gCut, bCut;
    float rMax, gMax, bMax;
    long int rWhole, gWhole, bWhole, wWhole;

    rWhole = Vol(set1, statsPtr->mR);
    gWhole = Vol(set1, statsPtr->mG);
    bWhole = Vol(set1, statsPtr->mB);
    wWhole = Vol(set1, statsPtr->wt);

    rMax = Maximize(set1, RED, set1->r0 + 1, set1->r1, &rCut,
	rWhole, gWhole, bWhole, wWhole, statsPtr);
    gMax = Maximize(set1, GREEN, set1->g0 + 1, set1->g1, &gCut,
	rWhole, gWhole, bWhole, wWhole, statsPtr);
    bMax = Maximize(set1, BLUE, set1->b0 + 1, set1->b1, &bCut,
	rWhole, gWhole, bWhole, wWhole, statsPtr);

    if ((rMax >= gMax) && (rMax >= bMax)) {
	dir = RED;
	if (rCut < 0) {
	    return 0;		/* can't split the box */
	}
    } else {
	dir = ((gMax >= rMax) && (gMax >= bMax)) ? GREEN : BLUE;
    }
    set2->r1 = set1->r1;
    set2->g1 = set1->g1;
    set2->b1 = set1->b1;

    switch (dir) {
    case RED:
	set2->r0 = set1->r1 = rCut;
	set2->g0 = set1->g0;
	set2->b0 = set1->b0;
	break;

    case GREEN:
	set2->g0 = set1->g1 = gCut;
	set2->r0 = set1->r0;
	set2->b0 = set1->b0;
	break;

    case BLUE:
	set2->b0 = set1->b1 = bCut;
	set2->r0 = set1->r0;
	set2->g0 = set1->g0;
	break;
    }
    set1->vol = (set1->r1 - set1->r0) * (set1->g1 - set1->g0) *
	(set1->b1 - set1->b0);
    set2->vol = (set2->r1 - set2->r0) * (set2->g1 - set2->g0) *
	(set2->b1 - set2->b0);
    return 1;
}

/*
 *----------------------------------------------------------------------
 *--------------------------------------------------------------------
 */
static void
Mark(cubePtr, label, tag)
    struct box *cubePtr;
    int label;
    unsigned int tag[33][33][33];
{
    register int r, g, b;

    for (r = R0 + 1; r <= R1; r++) {
	for (g = G0 + 1; g <= G1; g++) {
	    for (b = B0 + 1; b <= B1; b++) {
		tag[r][g][b] = label;
	    }
	}
    }
}

static void
FindClosestColor(colorPtr, mapColors, numMapColors)
    ColorInfo *colorPtr;
    XColor mapColors[];
    int numMapColors;
{
    double r, g, b;
    register int i;
    double dist, min;
    XColor *lastMatch;
    register XColor *mapColorPtr;
    extern double bltPosInfinity;

    min = bltPosInfinity;	/* Any color is closer. */
    lastMatch = NULL;

    /* Linear search of color */

    mapColorPtr = mapColors;
    for (i = 0; i < numMapColors; i++, mapColorPtr++) {
	r = (double)mapColorPtr->red - (double)colorPtr->exact.red;
	g = (double)mapColorPtr->green - (double)colorPtr->exact.green;
	b = (double)mapColorPtr->blue - (double)colorPtr->exact.blue;

#ifdef notdef
	dist = (30 * r * r) + (59 * g * g) + (11 * b * b);
	dist = 212 * r * r + 715 * g * g + 72 * b * b;
	dist = 2.0 * (r * r) + (b * b) + 3.0 * (g * g);
#endif
	dist = (r * r) + (b * b) + (g * g);
	if (dist < min) {
	    min = dist;
	    lastMatch = mapColorPtr;
	}
    }
    colorPtr->best = *lastMatch;
    colorPtr->best.flags = (DoRed | DoGreen | DoBlue);
    colorPtr->error = (float)sqrt(min);
}

static int
CompareColors(a, b)
    void *a, *b;
{
    int diff;
    ColorInfo *i1Ptr, *i2Ptr;

    i1Ptr = *(ColorInfo **) a;
    i2Ptr = *(ColorInfo **) b;
#ifndef notdef
    /* Compare colors based upon frequency */
    diff = i2Ptr->freq - i1Ptr->freq;
    if (ABS(diff) > 100) {
	return diff;
    }
#endif
    if (i2Ptr->error > i1Ptr->error) {
	return 1;
    } else if (i2Ptr->error < i1Ptr->error) {
	return -1;
    }
    return 0;
}

static float
MatchColors(colorTabPtr, rgbPtr, numColors, numAvailColors, numMapColors,
    mapColors)
    struct ColorTable *colorTabPtr;
    register RGB *rgbPtr;
    int numColors;
    int numAvailColors;
    int numMapColors;
    XColor mapColors[NCOLORS];
{
    int numMatched;
    float sum;
    register int i;
    register ColorInfo *colorPtr;

    /*
     * For each quantized color, compute and store the error (i.e
     * the distance from a color that's already been allocated).
     * We'll use this information to sort the colors based upon how
     * badly they match and their frequency to the color image.
     */
    colorPtr = colorTabPtr->colorInfo;
    for (i = 0; i < numColors; i++, colorPtr++, rgbPtr++) {
	colorPtr->index = i;
	colorTabPtr->sortedColors[i] = colorPtr;
	colorPtr->exact.red = rgbPtr->red;
	colorPtr->exact.green = rgbPtr->green;
	colorPtr->exact.blue = rgbPtr->blue;
	colorPtr->exact.flags = (DoRed | DoGreen | DoBlue);
	FindClosestColor(colorPtr, mapColors, numMapColors);
    }

    /* Sort the colors, first by frequency (most to least), then by
     * matching error (worst to best).
     */
    qsort(colorTabPtr->sortedColors, numColors, sizeof(ColorInfo *),
	(QSortCompareProc *)CompareColors);

    for (i = 0; i < numColors; i++) {
	colorPtr = colorTabPtr->sortedColors[i];
	fprintf(stderr, "%d. %04x%04x%04x / %04x%04x%04x = %f (%d)\n", i,
	    colorPtr->exact.red, colorPtr->exact.green, colorPtr->exact.blue,
	    colorPtr->best.red, colorPtr->best.green, colorPtr->best.blue,
	    colorPtr->error, colorPtr->freq);
    }
    sum = 0.0;
    numMatched = 0;
    for (i = numAvailColors; i < numColors; i++) {
	colorPtr = colorTabPtr->sortedColors[i];
	sum += colorPtr->error;
	numMatched++;
    }
    if (numMatched > 0) {
	sum /= numMatched;
    }
    return sum;
}

static int
AllocateColors(nImageColors, colorTabPtr, matchOnly)
    int nImageColors;
    struct ColorTable *colorTabPtr;
    int matchOnly;
{
    register int i;
    register ColorInfo *colorPtr;
    unsigned long int pixelValue;

#ifndef notdef
    if (colorTabPtr->nPixels > 0) {
	fprintf(stderr, "freeing %d pixels\n", colorTabPtr->nPixels);
	XFreeColors(colorTabPtr->display, colorTabPtr->colorMap,
	    colorTabPtr->pixelValues, colorTabPtr->nPixels, 0);
    }
#endif
    for (i = 0; i < nImageColors; i++) {
	colorPtr = colorTabPtr->sortedColors[i];
	if (matchOnly) {
	    XAllocColor(colorTabPtr->display, colorTabPtr->colorMap,
		&colorPtr->best);
	    pixelValue = colorPtr->best.pixel;
	} else {
	    colorPtr->allocated = XAllocColor(colorTabPtr->display,
		colorTabPtr->colorMap, &colorPtr->exact);

	    if (colorPtr->allocated) {
		pixelValue = colorPtr->exact.pixel;
	    } else {
		XAllocColor(colorTabPtr->display, colorTabPtr->colorMap,
		    &colorPtr->best);
		pixelValue = colorPtr->best.pixel;
	    }
	}
	colorTabPtr->pixelValues[colorPtr->index] = pixelValue;
    }
    colorTabPtr->nPixels = nImageColors;
    return 1;
}

/*
 *----------------------------------------------------------------------
 *
 * QuantizeColorimage --
 *
 *	C Implementation of Wu's Color Quantizer (v. 2) (see Graphics Gems
 *	vol. II, pp. 126-133)
 *
 *	Author: Xiaolin Wu
 *		Dept. of Computer Science Univ. of Western
 *		Ontario London, Ontario
 *		N6A 5B7
 *		wu@csd.uwo.ca
 *
 *	Algorithm:
 *		Greedy orthogonal bipartition of RGB space for variance
 *		minimization aided by inclusion-exclusion tricks.  For
 *		speed no nearest neighbor search is done. Slightly
 *		better performance can be expected by more
 *		sophisticated but more expensive versions.
 *
 *	The author thanks Tom Lane at Tom_Lane@G.GP.CS.CMU.EDU for much of
 *	additional documentation and a cure to a previous bug.
 *
 *	Free to distribute, comments and suggestions are appreciated.
 *
 *----------------------------------------------------------------------
 */
static int
QuantizeColorimage(image, colorTabPtr)
    Colorimage image;		/* Source image */
    struct ColorTable *colorTabPtr;
{
    float error;
    int best = 1;
    struct box *cubePtr;
    struct box cubeArr[NCOLORS];
    int nColors, nFreeColors, nQuantized;
    int nBestColors;
    int next;
    register long int i, weight;
    register int k;
    float vv[NCOLORS], temp;
    ColorStats *statsPtr;
    float maxError;
    RGB palette[NCOLORS];
    RGB *rgbPtr;
    int nMapColors, nAvail;
    int matchOnly;
    XColor mapColors[NCOLORS];

    /*
     * Allocated a structure to hold color statistics.
    */
    statsPtr = (ColorStats *) calloc(1, sizeof(ColorStats));
    assert(statsPtr);

    Hist3d(statsPtr, image);
    M3d(statsPtr->wt, statsPtr->mR, statsPtr->mG, statsPtr->mB,
	statsPtr->gm2);

    switch (colorTabPtr->visualInfo.class) {
    case StaticGray:
    case StaticColor:
    case GrayScale:
	maxError = 320000.0;
	break;
    default:
	maxError = 121.0;
	break;
    }


  retry:
    nColors = NCOLORS;
    matchOnly = TRUE;
    if (best) {
	int nReds, nGreens, nBlues;
	static int colorVar[7][3] =
	{
	    {8, 8, 4, /*  #red, #green, #blue */ },
	    {7, 7, 4, /* 8 bits, 198 colors */ },
	    {5, 6, 4, /* 7 bits, 120 colors */ },
	    {4, 5, 3, /* 6 bits, 60 colors */ },
	    {3, 4, 2, /* 5 bits, 24 colors */ },
	    {2, 3, 2, /* 4 bits, 12 colors */ },
	    {2, 2, 2, /* 3 bits, 8 colors */ },
	};

	nAvail = QueryColormap(colorTabPtr->display, colorTabPtr->colorMap,
	    mapColors, &nMapColors);

	nReds = colorVar[1][RED];
	nGreens = colorVar[1][GREEN];
	nBlues = colorVar[1][BLUE];

	for (i = 0; i < 7; i++) {
	    nColors = BuildColorRamp(palette, nReds, nGreens, nBlues);
	    error = MatchColors(colorTabPtr, palette, nColors, nAvail,
		nMapColors, mapColors);
	    fprintf(stderr, "nColors=%d, error=%f\n", nColors, error);
	    if (error < 1.0) {
		break;
	    }
	    /*
	     * Reduce the number of shades of each primary to about
	     * 3/4 of the previous value.  This should reduce the
	     * total number of colors required to about half the
	     * previous value for PseudoColor displays.
	     * (Huh?. Try doing the math again).
	     */

	    nReds = (nReds * 3 + 2) / 4;
	    nGreens = (nGreens * 3 + 2) / 4;
	    nBlues = (nBlues * 3 + 2) / 4;
	}
	AllocateColors(nColors, colorTabPtr, FALSE);
	nBestColors = nColors;
	/* nColors = NCOLORS; */
	matchOnly = TRUE;
	maxError = 320000.0;
	nFreeColors = 0;
    }
    nAvail = QueryColormap(colorTabPtr->display, colorTabPtr->colorMap,
	mapColors, &nMapColors);

    for (;;) {
	cubeArr[0].r0 = cubeArr[0].g0 = cubeArr[0].b0 = 0;
	cubeArr[0].r1 = cubeArr[0].g1 = cubeArr[0].b1 = 32;

	next = 0;

	for (i = 1; i < nColors; i++) {
	    if (Cut(cubeArr + next, cubeArr + i, statsPtr)) {
		/*
		 * Volume test ensures we won't try to cut one-cell box
		 */
		vv[next] = (cubeArr[next].vol > 1) ?
		    Var(cubeArr + next, statsPtr) : 0.0;
		vv[i] = (cubeArr[i].vol > 1) ?
		    Var(cubeArr + i, statsPtr) : 0.0;
	    } else {
		vv[next] = 0.0;	/* don't try to split this box again */
		i--;		/* didn't create box i */
	    }

	    next = 0;
	    temp = vv[0];
	    for (k = 1; k <= i; k++) {
		if (vv[k] > temp) {
		    temp = vv[k];
		    next = k;
		}
	    }
	    if (temp <= 0.0) {
		nColors = i + 1;
		fprintf(stderr, "Only got %d boxes\n", nColors);
		break;
	    }
	}
	rgbPtr = palette;
	for (cubePtr = cubeArr, k = 0; k < nColors; k++, cubePtr++, rgbPtr++) {
	    weight = Vol(cubePtr, statsPtr->wt);
	    colorTabPtr->colorInfo[k].freq = weight;
	    if (weight) {
		rgbPtr->red =
		    (Vol(cubePtr, statsPtr->mR) / weight) * (NCOLORS + 1);
		rgbPtr->green =
		    (Vol(cubePtr, statsPtr->mG) / weight) * (NCOLORS + 1);
		rgbPtr->blue =
		    (Vol(cubePtr, statsPtr->mB) / weight) * (NCOLORS + 1);
	    } else {
		fprintf(stderr, "bogus box %d\n", k);
		rgbPtr->red = rgbPtr->green = rgbPtr->blue = 0;
	    }
	}
	error = MatchColors(colorTabPtr, palette, nColors, nAvail,
	    nMapColors, mapColors);
	fprintf(stderr, "!!nColors=%d, error=%f\n", nColors, error);
	if ((error > maxError) && (nColors > 32)) {
	    nColors /= 2;
	    continue;
	}
	nQuantized = nColors;
	break;			/* Break out of loop */
    }
    if (!AllocateColors(nQuantized, colorTabPtr, matchOnly)) {
	goto retry;		/* Color map changed */
    }
    for (cubePtr = cubeArr, k = 0; k < nQuantized; k++, cubePtr++) {
	Mark(cubePtr, (int)colorTabPtr->pixelValues[k], colorTabPtr->lut);
    }
    free((char *)statsPtr);
    return nQuantized;
}

/*
 *----------------------------------------------------------------------
 *
 * AllocateBestColors --
 *
 *	C Implementation of Wu's Color Quantizer (v. 2) (see Graphics Gems
 *	vol. II, pp. 126-133)
 *
 *	Author: Xiaolin Wu
 *		Dept. of Computer Science Univ. of Western
 *		Ontario London, Ontario
 *		N6A 5B7
 *		wu@csd.uwo.ca
 *
 *	Algorithm:
 *		Greedy orthogonal bipartition of RGB space for variance
 *		minimization aided by inclusion-exclusion tricks.  For
 *		speed no nearest neighbor search is done. Slightly
 *		better performance can be expected by more
 *		sophisticated but more expensive versions.
 *
 *	The author thanks Tom Lane at Tom_Lane@G.GP.CS.CMU.EDU for much of
 *	additional documentation and a cure to a previous bug.
 *
 *	Free to distribute, comments and suggestions are appreciated.
 *
 *----------------------------------------------------------------------
 */
static int
AllocateBestColors(image, colorTabPtr)
    Colorimage image;		/* Source image */
    struct ColorTable *colorTabPtr;
{
    float error;
    struct box *cubePtr;
    struct box cubeArr[NCOLORS];
    int nColors, nFreeColors;
    int nBestColors;
    int next;
    register long int i, weight;
    register int k;
    float vv[NCOLORS], temp;
    ColorStats *statsPtr;
    float maxError;
    RGB palette[NCOLORS];
    RGB *rgbPtr;
    int nMapColors, nAvail;
    int matchOnly;
    XColor mapColors[NCOLORS];
    unsigned int pixelValues[NCOLORS];
    int nReds, nGreens, nBlues;
    static int paletteChoice[7][3] =
    {
	{8, 8, 4, /*  #red, #green, #blue */ },
	{7, 7, 4, /* 8 bits, 198 colors */ },
	{5, 6, 4, /* 7 bits, 120 colors */ },
	{4, 5, 3, /* 6 bits, 60 colors */ },
	{3, 4, 2, /* 5 bits, 24 colors */ },
	{2, 3, 2, /* 4 bits, 12 colors */ },
	{2, 2, 2, /* 3 bits, 8 colors */ },
    };

    nColors = NCOLORS;
    matchOnly = TRUE;


    nAvail = QueryColormap(colorTabPtr->display, colorTabPtr->colorMap,
	mapColors, &nMapColors);

    nReds = paletteChoice[1][RED];
    nGreens = paletteChoice[1][GREEN];

    nBlues = paletteChoice[1][BLUE];

    for (i = 0; i < 7; i++) {
	nColors = BuildColorRamp(palette, nReds, nGreens, nBlues);
	error = MatchColors(colorTabPtr, palette, nColors, nAvail,
	    nMapColors, mapColors);
	fprintf(stderr, "nColors=%d, error=%f\n", nColors, error);
	if (error < 1.0) {
	    break;
	}
	/*
	 * Reduce the number of shades of each primary to about
	 * 3/4 of the previous value.  This should reduce the
	 * total number of colors required to about half the
	 * previous value for PseudoColor displays.
	 * (Huh?. Try doing the math again).
	 */
	nReds = (nReds * 3 + 2) / 4;
	nGreens = (nGreens * 3 + 2) / 4;
	nBlues = (nBlues * 3 + 2) / 4;
    }

    AllocateColors(nColors, colorTabPtr, FALSE);

    nBestColors = nColors;
    matchOnly = TRUE;
    maxError = 320000.0;
    nFreeColors = 0;

    /*
     * Allocated a structure to hold color statistics.
     */
    statsPtr = (ColorStats *) calloc(1, sizeof(ColorStats));
    assert(statsPtr);

    Hist3d(statsPtr, image);
    M3d(statsPtr->wt, statsPtr->mR, statsPtr->mG, statsPtr->mB,
	statsPtr->gm2);

    switch (colorTabPtr->visualInfo.class) {
    case StaticGray:
    case StaticColor:
    case GrayScale:
	maxError = 320000.0;
	break;
    default:
	maxError = 1.0;
	break;
    }

    nAvail = QueryColormap(colorTabPtr->display, colorTabPtr->colorMap,
	mapColors, &nMapColors);

    cubeArr[0].r0 = cubeArr[0].g0 = cubeArr[0].b0 = 0;
    cubeArr[0].r1 = cubeArr[0].g1 = cubeArr[0].b1 = 32;

    next = 0;

    nColors = NCOLORS;
    for (i = 1; i < nColors; i++) {
	if (Cut(cubeArr + next, cubeArr + i, statsPtr)) {
	    /*
	     * Volume test ensures we won't try to cut one-cell box
	     */
	    vv[next] = (cubeArr[next].vol > 1) ?
		Var(cubeArr + next, statsPtr) : 0.0;
	    vv[i] = (cubeArr[i].vol > 1) ?
		Var(cubeArr + i, statsPtr) : 0.0;
	} else {
	    vv[next] = 0.0;	/* don't try to split this box again */
	    i--;		/* didn't create box i */
	}

	next = 0;
	temp = vv[0];
	for (k = 1; k <= i; k++) {
	    if (vv[k] > temp) {
		temp = vv[k];
		next = k;
	    }
	}
	if (temp <= 0.0) {
	    nColors = i + 1;
	    fprintf(stderr, "Only got %d boxes\n", nColors);
	    break;
	}
    }
    rgbPtr = palette;
    for (cubePtr = cubeArr, k = 0; k < nColors; k++, cubePtr++, rgbPtr++) {
	weight = Vol(cubePtr, statsPtr->wt);
	colorTabPtr->colorInfo[k].freq = weight;
	if (weight) {
	    rgbPtr->red = (Vol(cubePtr, statsPtr->mR) / weight) * (NCOLORS + 1);
	    rgbPtr->green = (Vol(cubePtr, statsPtr->mG) / weight) * (NCOLORS + 1);
	    rgbPtr->blue = (Vol(cubePtr, statsPtr->mB) / weight) * (NCOLORS + 1);
	} else {
	    fprintf(stderr, "bogus box %d\n", k);
	    rgbPtr->red = rgbPtr->green = rgbPtr->blue = 0;
	}
    }
    error = MatchColors(colorTabPtr, palette, nColors, nAvail,
	nMapColors, mapColors);
    fprintf(stderr, "!!nColors=%d, error=%f\n", nColors, error);

    for (k = 0; k < nColors; k++) {
	pixelValues[k] = colorTabPtr->colorInfo[k].best.pixel;
    }
    for (cubePtr = cubeArr, k = 0; k < nColors; k++, cubePtr++) {
	Mark(cubePtr, pixelValues[k], colorTabPtr->lut);
    }
    free((char *)statsPtr);
    return nColors;
}

ColorTable
Blt_CreateColorTable(tkwin)
    Tk_Window tkwin;
{
    XVisualInfo visualInfo, *visualInfoPtr;
    int nVisuals;
    Visual *visualPtr;
    Display *display;
    struct ColorTable *colorTabPtr;

    display = Tk_Display(tkwin);
    visualPtr = Tk_Visual(tkwin);

    colorTabPtr = (ColorTable)calloc(1, sizeof(struct ColorTable));
    assert(colorTabPtr);
    colorTabPtr->display = Tk_Display(tkwin);
    colorTabPtr->colorMap = Tk_Colormap(tkwin);

    visualInfo.screen = Tk_ScreenNumber(tkwin);
    visualInfo.visualid = XVisualIDFromVisual(visualPtr);
    visualInfoPtr = XGetVisualInfo(display, VisualScreenMask | VisualIDMask,
	&visualInfo, &nVisuals);

    colorTabPtr->visualInfo = *visualInfoPtr;
    XFree(visualInfoPtr);

    return colorTabPtr;
}

void
Blt_FreeColorTable(colorTabPtr)
    struct ColorTable *colorTabPtr;
{
    if (colorTabPtr == NULL) {
	return;
    }
    if (colorTabPtr->nPixels > 0) {
	XFreeColors(colorTabPtr->display, colorTabPtr->colorMap,
	    colorTabPtr->pixelValues, colorTabPtr->nPixels, 0);
    }
    free((char *)colorTabPtr);
}

extern int redAdjust, greenAdjust, blueAdjust;
extern int redMaskShift, greenMaskShift, blueMaskShift;

/*
 *----------------------------------------------------------------------
 *
 * Blt_DirectColorTable --
 *
 *	Creates a color table using the DirectColor visual.  We try
 *	to allocate colors across the color spectrum.
 *
 * Results:
 *
 *
 *----------------------------------------------------------------------
 */
ColorTable
Blt_DirectColorTable(interp, tkwin, image)
    Tcl_Interp *interp;
    Tk_Window tkwin;
    Colorimage image;
{
    struct ColorTable *colorTabPtr;
    Visual *visualPtr;
    Display *display;
    XColor color;
    int nr, ng, nb;
    int rBand, gBand, bBand;
    int rLast, gLast, bLast;
    unsigned int r, g, b;
    unsigned int value;
    register int i;

    display = Tk_Display(tkwin);
    visualPtr = Tk_Visual(tkwin);

    colorTabPtr = Blt_CreateColorTable(tkwin);
    /*
     * Compute the number of distinct colors in each band
     */
    nr = ((unsigned int)visualPtr->red_mask >> redMaskShift) + 1;
    ng = ((unsigned int)visualPtr->green_mask >> greenMaskShift) + 1;
    nb = ((unsigned int)visualPtr->blue_mask >> blueMaskShift) + 1;

#ifdef notdef
    assert((nr <= visualPtr->map_entries) && (ng <= visualPtr->map_entries) &&
	(nb <= visualPtr->map_entries));
#endif
    rBand = NCOLORS / nr;
    gBand = NCOLORS / ng;
    bBand = NCOLORS / nb;

  retry:
    color.flags = (DoRed | DoGreen | DoBlue);
    rLast = gLast = bLast = 0;
    r = g = b = 0;
    for (i = 0; i < visualPtr->map_entries; i++) {
	if (rLast < NCOLORS) {
	    r = rLast + rBand;
	    if (r > NCOLORS) {
		r = NCOLORS;
	    }
	}
	if (gLast < NCOLORS) {
	    g = gLast + gBand;
	    if (g > NCOLORS) {
		g = NCOLORS;
	    }
	}
	if (bLast < NCOLORS) {
	    b = bLast + bBand;
	    if (b > NCOLORS) {
		b = NCOLORS;
	    }
	}
	color.red = (r - 1) * (NCOLORS + 1);
	color.green = (g - 1) * (NCOLORS + 1);
	color.blue = (b - 1) * (NCOLORS + 1);

	if (!XAllocColor(display, colorTabPtr->colorMap, &color)) {
	    XFreeColors(display, colorTabPtr->colorMap,
		colorTabPtr->pixelValues, i, 0);
	    if ((colorTabPtr->flags & PRIVATE_COLORMAP) == 0) {
		/*
		 * If we can't allocate a color in the default
		 * colormap, try again, this time with a private
		 * colormap.
		 */
		fprintf(stderr, "Need to allocate private colormap\n");
		colorTabPtr->colorMap = Tk_GetColormap(interp, tkwin, ".");

		XSetWindowColormap(display, Tk_WindowId(tkwin),
		    colorTabPtr->colorMap);
		colorTabPtr->flags |= PRIVATE_COLORMAP;
		goto retry;
	    }
	    fprintf(stderr, "Failed to allocate after %d colors\n", i);
	    free((char *)colorTabPtr);
	    return NULL;	/* Ran out of colors in private map? */
	}
	colorTabPtr->pixelValues[i] = color.pixel;
	/*
	 * Fill in pixel values for each band at this intensity
	 */
	value = color.pixel & visualPtr->red_mask;
	while (rLast < r) {
	    colorTabPtr->red[rLast++] = value;
	}
	value = color.pixel & visualPtr->green_mask;
	while (gLast < g) {
	    colorTabPtr->green[gLast++] = value;
	}
	value = color.pixel & visualPtr->blue_mask;
	while (bLast < b) {
	    colorTabPtr->blue[bLast++] = value;
	}
    }
    colorTabPtr->nPixels = i;
    return colorTabPtr;
}

/*
 * First attempt:
 *	Allocate colors all the colors in the image (up to NCOLORS). Bail out
 *	on the first failure or if we need more than NCOLORS.
 */
static int
GetUniqueColors(image)
    Colorimage image;
{
    register int i, nColors;
    register Pix32 *pixelPtr;
    Pix32 color;
    Tcl_HashEntry *hPtr;
    int isNew, nPixels;
    int refCount;
    Tcl_HashTable colorTable;

    Tcl_InitHashTable(&colorTable, TCL_ONE_WORD_KEYS);

    nPixels = ColorimageWidth(image) * ColorimageHeight(image);
    nColors = 0;
    pixelPtr = ColorimageData(image);
    for (i = 0; i < nPixels; i++, pixelPtr++) {
	color.value = pixelPtr->value;
	color.Alpha = 0xFF;	/* Ignore alpha-channel values */
	hPtr = Tcl_CreateHashEntry(&colorTable, (char *)color.value, &isNew);
	if (isNew) {
	    refCount = 1;
	    nColors++;
	} else {
	    refCount = (int)Tcl_GetHashValue(hPtr);
	    refCount++;
	}
	Tcl_SetHashValue(hPtr, (ClientData)refCount);
    }
    Tcl_DeleteHashTable(&colorTable);
    return nColors;
}

#define Blt_DefaultColormap(tkwin)  \
	DefaultColormap(Tk_Display(tkwin), Tk_ScreenNumber(tkwin))


static void
PrivateColormap(interp, colorTabPtr, image, tkwin)
    Tcl_Interp *interp;
    struct ColorTable *colorTabPtr;
    Colorimage image;
    Tk_Window tkwin;
{
    int keepColors = 0;
    int best = 1;
    register int i;
    XColor usedColors[NCOLORS];
    int nFreeColors, nUsedColors;
    Colormap colorMap;
    int inUse[NCOLORS];
    XColor *colorPtr;
    XColor *imageColors;

    /*
     * Create a private colormap if one doesn't already exist for the
     * window.
     */

    colorTabPtr->colorMap = colorMap = Tk_Colormap(tkwin);

    nUsedColors = 0;		/* Number of colors allocated */

    if (colorTabPtr->nPixels > 0) {
	XFreeColors(colorTabPtr->display, colorTabPtr->colorMap,
	    colorTabPtr->pixelValues, colorTabPtr->nPixels, 0);
    }
    nFreeColors = QueryColormap(colorTabPtr->display, colorMap, usedColors,
	&nUsedColors);
    memset((char *)inUse, 0, sizeof(int) * NCOLORS);
    if ((nUsedColors == 0) && (keepColors > 0)) {

	/*
	 * We're starting with a clean colormap so find out what colors
	 * have been used in the default colormap.
	 */

	nFreeColors = QueryColormap(colorTabPtr->display,
	    Blt_DefaultColormap(tkwin), usedColors, &nUsedColors);

	/*
	 * Copy a number of colors from the default colormap into the private
	 * colormap.  We can assume that this is the working set from most
	 * (non-image related) applications. While this doesn't stop our
	 * image from flashing and looking dumb when colormaps are swapped
	 * in and out, at least everything else should remain unaffected.
	 */

	if (nUsedColors > keepColors) {
	    nUsedColors = keepColors;
	}
	/*
	 * We want to allocate colors in the same ordering as the old colormap,
	 * and we can't assume that the colors in the old map were contiguous.
	 * So mark the colormap locations (i.e. pixels) that we find in use.
	 */

    }
    for (colorPtr = usedColors, i = 0; i < nUsedColors; i++, colorPtr++) {
	inUse[colorPtr->pixel] = TRUE;
    }

    /*
     * In an "exact" colormap, we try to allocate as many of colors from the
     * image as we can fit.  If necessary, we'll cheat and reduce the number
     * of colors by quantizing.
     */
    imageColors = usedColors + nUsedColors;

    if (best) {
	AllocateBestColors(image, colorTabPtr);
    } else {
	QuantizeColorimage(image, colorTabPtr);
    }
    Tk_SetWindowColormap(tkwin, colorMap);
}

ColorTable
Blt_PseudoColorTable(interp, tkwin, image)
    Tcl_Interp *interp;
    Tk_Window tkwin;
    Colorimage image;
{
    struct ColorTable *colorTabPtr;
    Colormap defColorMap;
    int usePrivate;

    colorTabPtr = Blt_CreateColorTable(tkwin);
    defColorMap = DefaultColormap(colorTabPtr->display, Tk_ScreenNumber(tkwin));
    if (colorTabPtr->colorMap == defColorMap) {
	fprintf(stderr, "Using default colormap\n");
    }
    /* All other visuals use an 8-bit colormap */
    colorTabPtr->lut = (unsigned int *)malloc(sizeof(unsigned int) * 33 * 33 * 33);
    assert(colorTabPtr->lut);

    usePrivate = TRUE;
    if (usePrivate) {
	PrivateColormap(interp, colorTabPtr, image, tkwin);
    } else {
#ifdef notdef
	ReadOnlyColormap(colorTabPtr, image, tkwin);
#endif
    }
    return colorTabPtr;
}

#ifdef notdef

static void
ConvoleColorimage(srcImage, destImage, kernelPtr)
    Colorimage srcImage, destImage;
    ConvoleKernel *kernelPtr;
{
    Pix32 *srcPtr, *destPtr;
    Pix32 *src[MAXROWS];
    register int x, y, i, j;
    int red, green, blue;

    /* i = 0 case, ignore left column of pixels */

    srcPtr = ColorimageData(srcImage);
    destPtr = ColorimageData(destImage);

    width = ColorimageWidth(srcImage);
    height = ColorimageHeight(srcImage);

    yOffset = kernelPtr->height / 2;
    xOffset = kernelPtr->width / 2;
    for (y = yOffset; y < (height - yOffset); y++) {
	/* Set up pointers to individual rows */
	for (i = 0; i < kernelPtr->height; i++) {
	    src[i] = srcPtr + (i * width);
	}
	for (x = xOffset; x < (width - xOffset); x++) {
	    red = green = blue = 0;
	    kernPtr = kernelPtr->values;
	    for (i = 0; i < kernelPtr->height; i++) {
		for (j = 0; j < kernelPtr->width; j++) {
		    red += *valuePtr * src[i][j].Red;
		    green += *valuePtr * src[i][j].Green;
		    blue += *valuePtr * src[i][j].Blue;
		    valuePtr++;
		}
	    }
	    destPtr->Red = red / kernelPtr->sum;
	    destPtr->Green = green / kernelPtr->sum;
	    destPtr->Blue = blue / kernelPtr->sum;
	    destPtr++;
	}
	srcPtr += width;
    }
    sum = bot[0].Red +
	red = bot[0].Red + bot[1].Red + mid[1].Red + top[0].Red + top[1].Red;
    green = bot[0].Green + bot[1].Green + mid[1].Green + top[0].Green +
	top[1].Green;
    blue = bot[0].Blue + bot[1].Blue + mid[1].Blue + top[0].Blue + top[1].Blue;
    error = (red / 5) - mid[0].Red;
    redVal = mid[0].Red - (error * blend / blend_divisor);
    error = (green / 5) - mid[0].Green;
    greenVal = mid[0].Green - (error * blend / blend_divisor);
    error = (blue / 5) - mid[0].Blue;
    blueVal = mid[0].Blue - (error * blend / blend_divisor);

    out[0].Red = CLAMP(redVal);
    out[0].Green = CLAMP(greenVal);
    out[0].Blue = CLAMP(blueVal);

    for (i = 1; i < (width - 1); i++) {
	for (chan = 0; chan < 3; chan++) {
	    total = bot[chan][i - 1] + bot[chan][i] + bot[chan][i + 1] +
		mid[chan][i - 1] + mid[chan][i + 1] +
		top[chan][i - 1] + top[chan][i] + top[chan][i + 1];
	    avg = total >> 3;	/* divide by 8 */
	    diff = avg - mid[chan][i];
	    result = mid[chan][i] - (diff * blend / blend_divisor);
	    out[chan][i] = CLAMP(result);
	}
    }
    /* i = in_hdr.xmax case, ignore right column of pixels */
    for (chan = 0; chan < 3; chan++) {
	total = bot[chan][i - 1] + bot[chan][i] +
	    mid[chan][i - 1] +
	    top[chan][i - 1] + top[chan][i];
	avg = total / 5;
	diff = avg - mid[chan][i];
	result = mid[chan][i] - (diff * blend / blend_divisor);
	out[chan][i] = CLAMP(result);
    }
}

static void
DitherRow(srcImage, destImage, lastRow, curRow)
    Colorimage srcImage;
    Colorimage destImage;
    int width, height;
    int bottom, top;
{
    int width, height;

    width = ColorimageWidth(srcImage);
    topPtr = ColorimageData(destPtr) + (width * row);
    rowPtr = topPtr + width;
    botPtr = rowPtr + width;

    for (x = 0; x < width; x++) {

	/* Clamp current error entry */

	midPtr->red = CLAMP(midPtr->red);
	midPtr->blue = CLAMP(midPtr->blue);
	midPtr->green = CLAMP(midPtr->green);

	r = (midPtr->red >> 3) + 1;
	g = (midPtr->green >> 3) + 1;
	b = (midPtr->blue >> 3) + 1;
	index = colorTabPtr->lut[r][g][b];

	redVal = midPtr->red * (NCOLORS + 1);
	greenVal = midPtr->green * (NCOLORS + 1);
	blueVal = midPtr->blue * (NCOLORS + 1);

	error = colorVal - colorMap[index].red;
	if (x < 511) {
	    currRow[x + 1].Red = currRow[x + 1].Red + 7 * error / 16;
	    nextRow[x + 1].Red = nextRow[x + 1].Red + error / 16;
	}
	nextRow[x].Red = nextRow[x].Red + 5 * error / 16;
	if (x > 0) {
	    nextRow[x - 1].Red = nextRow[x - 1].Red + 3 * error / 16;
	}
	error = row[x][c] - colormap[index][c];

	value = srcPtr->channel[i] * error[i];
	value = CLAMP(value);
	destPtr->channel[i] = value;

	/* Closest pixel */
	pixel = PsuedoColorPixel();
	error[RED] = colorPtr->Red - srcPtr->Red * (NCOLORS + 1);

	/* translate pixel to colorInfoPtr to get error */
	colorTabPtr->lut[r][g][b];
	colorPtr = PixelToColorInfo(pixel);
	error = colorPtr->error;

	register rle_pixel *optr;
	register int j;
	register short *thisptr, *nextptr = NULL;
	int chan;
	static int nchan = 0;
	int lastline = 0, lastpixel;
	static int *cval = 0;
	static rle_pixel *pixel = 0;

	if (nchan != in_hdr->ncolors)
	    if (cval) {
		free(cval);
		free(pixel);
	    }
	nchan = in_hdr->ncolors;
	if (!cval) {
	    if ((cval = (int *)malloc(nchan * sizeof(int))) == 0)
		    MALLOC_ERR;
	    if ((pixel = (rle_pixel *) malloc(nchan * sizeof(rle_pixel))) == 0)
		MALLOC_ERR;
	}
	optr = outrow[RLE_RED];

	thisptr = row_top;
	if (row_bottom)
	    nextptr = row_bottom;
	else
	    lastline = 1;

	for (x = 0; x < width; x++) {
	    int cmap_index = 0;

	    lastpixel = (x == (width - 1));
	    val = srcPtr->Red;

	    for (chan = 0; chan < 3; chan++) {
		cval[chan] = *thisptr++;

		/*
		 * Current channel value has been accumulating error,
		 * it could be out of range.
		 */
		if (cval[chan] < 0)
		    cval[chan] = 0;
		else if (cval[chan] > 255)
		    cval[chan] = 255;

		pixel[chan] = cval[chan];
	    }

	    /* find closest color */
	    find_closest(map, nchan, maplen, pixel, &cmap_index);
	    *optr++ = cmap_index;

	    /* thisptr is now looking at pixel to the right of current pixel
	     * nextptr is looking at pixel below current pixel
	     * So, increment thisptr as stuff gets stored.  nextptr gets moved
	     * by one, and indexing is done +/- nchan.
	     */
	    for (chan = 0; chan < nchan; chan++) {
		cval[chan] -= map[chan][cmap_index];

		if (!lastpixel) {
		    thisptr[chan] += cval[chan] * 7 / 16;
		}
		if (!lastline) {
		    if (j != 0) {
			nextptr[-nchan] += cval[chan] * 3 / 16;
		    }
		    nextptr[0] += cval[chan] * 5 / 16;
		    if (!lastpixel) {
			nextptr[nchan] += cval[chan] / 16;
		    }
		    nextptr++;
		}
	    }
	}
    }
}

/********************************************/
static Colorimage
DoColorDither(pic24, pic8, w, h, rmap, gmap, bmap, rdisp, gdisp, bdisp, maplen)
    byte *pic24, *pic8, *rmap, *gmap, *bmap, *rdisp, *gdisp, *bdisp;
    int w, h, maplen;
{
    /* takes a 24 bit picture, of size w*h, dithers with the colors in
       rdisp, gdisp, bdisp (which have already been allocated),
       and generates an 8-bit w*h image, which it returns.
       ignores input value 'pic8'
       returns NULL on error

       note: the rdisp,gdisp,bdisp arrays should be the 'displayed' colors,
       not the 'desired' colors

       if pic24 is NULL, uses the passed-in pic8 (an 8-bit image) as
       the source, and the rmap,gmap,bmap arrays as the desired colors */

    byte *np, *ep, *newpic;
    short *cache;
    int r2, g2, b2;
    int *thisline, *nextline, *thisptr, *nextptr, *tmpptr;
    int i, j, rerr, gerr, berr, pwide3;
    int imax, jmax;
    int key;
    long cnt1, cnt2;
    int error[512];		/* -255 .. 0 .. +255 */

    /* compute somewhat non-linear floyd-steinberg error mapping table */
    for (i = j = 0; i <= 0x40; i++, j++) {
	error[256 + i] = j;
	error[256 - i] = -j;
    }
    for ( /*empty*/ ; i < 0x80; i++, j += !(i & 1) ? 1 : 0) {
	error[256 + i] = j;
	error[256 - i] = -j;
    }
    for ( /*empty*/ ; i <= 0xff; i++) {
	error[256 + i] = j;
	error[256 - i] = -j;
    }

    cnt1 = cnt2 = 0;
    pwide3 = w * 3;
    imax = h - 1;
    jmax = w - 1;
    ep = (pic24) ? pic24 : pic8;

    /* attempt to malloc things */
    newpic = (byte *) malloc((size_t) (w * h));
    cache = (short *)calloc((size_t) (2 << 14), sizeof(short));
    thisline = (int *)malloc(pwide3 * sizeof(int));
    nextline = (int *)malloc(pwide3 * sizeof(int));
    if (!cache || !newpic || !thisline || !nextline) {
	if (newpic)
	    free(newpic);
	if (cache)
	    free(cache);
	if (thisline)
	    free(thisline);
	if (nextline)
	    free(nextline);
	return (byte *) NULL;
    }
    np = newpic;

    /* Get first line of picture in reverse order. */

    srcPtr = ColorimageData(image), tempPtr = tempArr;
    for (x = 0; x < width; x++, tempPtr++, srcPtr--) {
	*tempPtr = *srcPtr;
    }

    for (y = 0; y < height; y++) {
	tempPtr = curRowPtr, curRowPtr = nextRowPtr, nextRowPtr = tempPtr;

	if (y != (height - 1)) {/* get next line */
	    for (x = 0; x < width; x++, tempPtr++, srcPtr--)
		*tempPtr = *srcPtr;
	}
    }


    free(thisline);
    free(nextline);
    free(cache);

    return newpic;
}


static void
DitherImage(image)
    Colorimage image;
{
    int width, height;



}

#endif

#endif /* WIN32 */
