
/*
 * bltTree.h --
 *
 * Copyright 1998-1999 Lucent Technologies, Inc.
 *
 * Permission to use, copy, modify, and distribute this software and
 * its documentation for any purpose and without fee is hereby
 * granted, provided that the above copyright notice appear in all
 * copies and that both that the copyright notice and warranty
 * disclaimer appear in supporting documentation, and that the names
 * of Lucent Technologies or any of their entities not be used in
 * advertising or publicity pertaining to distribution of the software
 * without specific, written prior permission.
 *
 * Lucent Technologies disclaims all warranties with regard to this
 * software, including all implied warranties of merchantability and
 * fitness.  In no event shall Lucent Technologies be liable for any
 * special, indirect or consequential damages or any damages
 * whatsoever resulting from loss of use, data or profits, whether in
 * an action of contract, negligence or other tortuous action, arising
 * out of or in connection with the use or performance of this
 * software.
 *
 *	The "tree" data object was created by George A. Howlett.
 */

#include "bltInt.h"

#ifndef NO_TREE

#include "bltChain.h"
#include "bltTree.h"

#ifdef __STDC__
static Tcl_IdleProc NotifyClients;
static Tcl_InterpDeleteProc TreeInterpDeleteProc;
#endif

#define TREE_THREAD_KEY		"BLT Tree Data"
#define TREE_MAGIC		((unsigned int) 0x46170277)

typedef struct Blt_TreeNode_ Node;
typedef struct Blt_TreeToken_ TreeToken;
typedef struct Blt_TreeObject_ TreeObject;

typedef struct ThreadData {
    Tcl_HashTable treeTable;	/* Table of trees. */
    unsigned int nextId;
    Tcl_Interp *interp;
} ThreadData;


/*
 * Datum --
 *
 *	Tree nodes contain heterogeneous data fields, represented
 *	as a chain of these structures.  Each field contains the
 *	name of the field (atom) and the Tcl_Obj containing the
 *	actual data representations.
 *
 */
typedef struct Datum {
    Blt_TreeAtom fieldAtom;	/* String identifying the data field */
    Tcl_Obj *objPtr;		/* Data representation. */
} Datum;

#define NOTIFY_UPDATED		(1<<4)
#define NOTIFY_DESTROYED	(1<<5)

#define NOTIFY_NEVER		(1<<6)	/* Never notify clients of updates to
					 * the vector */
#define NOTIFY_ALWAYS		(1<<7)	/* Notify clients after each update
					 * of the vector is made */
#define NOTIFY_WHENIDLE		(1<<8)	/* Notify clients at the next idle point
					 * that the vector has been updated. */

#define NOTIFY_PENDING		(1<<9)	/* A do-when-idle notification of the
					 * vector's clients is pending. */
#define NOTIFY_NOW		(1<<10)	/* Notify clients of changes once
					 * immediately */
/*
 * TreeKey --
 *
 *	Tree objects are identified by their name and their namespace.
 *	We can have two different trees each with the same name, but 
 *	created in different namespaces.
 *
 *	Warning: Don't delete a namespace without first deleting the
 *	trees associated with it.  Right now (Tcl 8.0.4/8.2.2), there
 *	is no fool-proof mechanism to detect when a namespace is
 *	destroyed and invoke a callback to clean up.  
 */
typedef struct TreeKey {
    Tk_Uid id;
    Tcl_Namespace *nsPtr;
} TreeKey;

/*
How does data get represented?

	string or procedure to translate to string

What's at each node?

	Name and data

Is there a string representation for data? name / data

	Create / Delete /
	
add a node to the tree.
via tree
must let browser refresh itself?
create entry
*/

/*
 * ----------------------------------------------------------------------
 *
 * NotifyClients --
 *
 *	Notifies each client of the vector that the vector has changed
 *	(updated or destroyed) by calling the provided function back.
 *	The function pointer may be NULL, in that case the client is
 *	not notified.
 *
 * Results:
 *	None.
 *
 * Side effects:
 *	The results depend upon what actions the client callbacks
 *	take.
 *
 * ----------------------------------------------------------------------
 */
static void
NotifyClients(clientData)
    ClientData clientData;
{
    TreeToken *tokenPtr = (TreeToken *)clientData;
    TreeObject *treePtr = tokenPtr->tree;
    Blt_ChainLink *linkPtr;
    TreeToken *clientPtr;
    unsigned int flags;

    flags = (treePtr->notifyFlags & NOTIFY_DESTROYED)
	? NOTIFY_DESTROYED : NOTIFY_UPDATED;
    treePtr->notifyFlags &= 
	~(NOTIFY_UPDATED | NOTIFY_DESTROYED | NOTIFY_PENDING);

    for (linkPtr = Blt_ChainFirstLink(treePtr->chainPtr); linkPtr != NULL;
	linkPtr = Blt_ChainNextLink(linkPtr)) {
	clientPtr = (TreeToken *)Blt_ChainGetValue(linkPtr);
	if ((clientPtr->notifyFlags & NOTIFY_UPDATED) &&
	    (clientPtr->procsPtr->notifyUpdateProc != NULL)) {
	    (*clientPtr->procsPtr->notifyUpdateProc) 
		(clientPtr, clientPtr->clientData, flags);
	}
    }
    /*
     * Some clients may not handle the "destroy" callback properly
     * (they should call Blt_FreeVectorId to release the client
     * identifier), so mark any remaining clients to indicate that
     * vector's server has gone away.
     */
    if (flags == NOTIFY_DESTROYED) {
	for (linkPtr = Blt_ChainFirstLink(treePtr->chainPtr); linkPtr != NULL;
	    linkPtr = Blt_ChainNextLink(linkPtr)) {
	    clientPtr = (TreeToken *)Blt_ChainGetValue(linkPtr);
	    clientPtr->tree = NULL;
	}
    }
}

/*
 *----------------------------------------------------------------------
 *
 * GetAtom --
 *
 *	Gets or creates a unique string identifier.  Strings are
 *	reference counted.  The string is placed into a hashed table
 *	local to the hierbox.
 *
 * Results:
 *	Returns the pointer to the hashed string.
 *
 *---------------------------------------------------------------------- 
 */
static Blt_TreeAtom
GetAtom(treePtr, string)
    TreeObject *treePtr;
    char *string;
{
    Tcl_HashEntry *hPtr;
    int refCount;
    int isNew;

    hPtr = Tcl_CreateHashEntry(&(treePtr->atomTable), string, &isNew);
    if (isNew) {
	refCount = 1;
    } else {
	refCount = (int)Tcl_GetHashValue(hPtr);
	refCount++;
    }
    Tcl_SetHashValue(hPtr, (ClientData)refCount);
    return Tcl_GetHashKey(&(treePtr->atomTable), hPtr);
}

/*
 *----------------------------------------------------------------------
 *
 * FreeAtom --
 *
 *	Releases the atom.  Atoms are reference counted, so only when
 *	the reference count is zero (i.e. no one else is using the
 *	string) is the entry removed from the hash table.
 *
 * Results:
 *	None.
 *
 *---------------------------------------------------------------------- 
 */
static void
FreeAtom(treePtr, atom)
    TreeObject *treePtr;
    Blt_TreeAtom atom;
{
    Tcl_HashEntry *hPtr;
    int refCount;

    hPtr = Tcl_FindHashEntry(&(treePtr->atomTable), atom);
    assert(hPtr != NULL);
    refCount = (int)Tcl_GetHashValue(hPtr);
    refCount--;
    if (refCount > 0) {
	Tcl_SetHashValue(hPtr, (ClientData)refCount);
    } else {
	Tcl_DeleteHashEntry(hPtr);
    }
}


static Node *
NewNode(treePtr, name)
    TreeObject *treePtr;
    char *name;
{
    Tcl_HashEntry *hPtr;
    Node *nodePtr;
    int isNew;
    int inode;

    /* Generate an unique serial number for this node.  */
    do {
	inode = treePtr->nextNode++;
	hPtr = Tcl_CreateHashEntry(&(treePtr->nodeTable),(char *)inode, 
				   &isNew);
    } while (!isNew);

    /* Create the node structure */
    nodePtr = (Node *) calloc(1, sizeof(Node));
    assert(nodePtr);
    nodePtr->inode = inode;
    if (name == NULL) {
	name = "";
    }
    nodePtr->atom = GetAtom(treePtr, name);
    nodePtr->treePtr = treePtr;
    nodePtr->dataPtr = Blt_ChainCreate();
    Tcl_SetHashValue(hPtr, (ClientData)nodePtr);
    treePtr->nNodes++;
    return nodePtr;
}

static void
DeleteNode(nodePtr)
    Node *nodePtr;
{
    Tcl_HashEntry *hPtr;
    Blt_ChainLink *linkPtr;
    Datum *datumPtr;

    /*
     * Run through the list of data field, deleting any entries
     * associated with this node.
     */
    for (linkPtr = Blt_ChainFirstLink(nodePtr->dataPtr);
	linkPtr != NULL; linkPtr = Blt_ChainNextLink(linkPtr)) {
	datumPtr = (Datum *) Blt_ChainGetValue(linkPtr);
	if (datumPtr->objPtr != NULL) {
	    Tcl_DecrRefCount(datumPtr->objPtr);
	}
	FreeAtom(nodePtr->treePtr, datumPtr->fieldAtom);
	free((char *)datumPtr);
    }
    Blt_ChainDestroy(nodePtr->dataPtr);

    /* Unlink the node from parent's list of siblings. */
    if (nodePtr->linkPtr != NULL) {
	Blt_ChainDeleteLink(nodePtr->parent->chainPtr, nodePtr->linkPtr);
    }
    nodePtr->treePtr->nNodes--;
    hPtr = Tcl_FindHashEntry(&(nodePtr->treePtr->nodeTable), 
	(char *)nodePtr->inode);
    assert(hPtr);
    FreeAtom(nodePtr->treePtr, nodePtr->atom);
    Tcl_DeleteHashEntry(hPtr);
    free((char *)nodePtr);
}

static TreeObject *
NewTreeObject(interp, nameUid, nsPtr)
    Tcl_Interp *interp;
    Tk_Uid nameUid;
    Tcl_Namespace *nsPtr;
{
    TreeObject *treePtr;

    treePtr = (TreeObject *)calloc(1, sizeof(TreeObject));
    if (treePtr == NULL) {
	Tcl_SetResult(interp, "can't allocate tree", TCL_STATIC);
	return NULL;
    }
    treePtr->interp = interp;
    treePtr->nameUid = nameUid;
    treePtr->nsPtr = nsPtr;
    Tcl_InitHashTable(&(treePtr->atomTable), TCL_STRING_KEYS);
    Tcl_InitHashTable(&(treePtr->nodeTable), TCL_ONE_WORD_KEYS);
    treePtr->depth = 1;
    treePtr->notifyFlags = 0;
    treePtr->root = NewNode(treePtr, "");
    treePtr->chainPtr = Blt_ChainCreate();
    return treePtr;
}

/*
 * ----------------------------------------------------------------------
 *
 * GetTreeObject --
 *
 *	Searches for the tree object associated by the name given.
 *
 * Results:
 *	Returns a pointer to the tree if found, otherwise NULL.
 *	If the name is not associated with a tree object and the
 *	TCL_LEAVE_ERR_MSG flag is set, and interp->result will
 *	contain an error message.
 *
 * ----------------------------------------------------------------------
 */
static TreeObject *
GetTreeObject(dataPtr, name)
    ThreadData *dataPtr;	/* Thread-specific data. */
    char *name;
{
    char *treeName;
    Tcl_Namespace *nsPtr;
    Tk_Uid nameUid;

    nsPtr = NULL;
    treeName = name;
    if (Blt_ParseQualifiedName(dataPtr->interp, name, &nsPtr, &treeName) 
	!= TCL_OK) {
	return NULL;		/* Can't find namespace. */
    }
    nameUid = Blt_FindUid(treeName);
    if (nameUid != NULL) {
	Tcl_HashEntry *hPtr;
	TreeKey key;

	key.id = nameUid;
	if (nsPtr != NULL) {	
	    /* Search only in designated namespace. */
	    key.nsPtr = nsPtr;
	    hPtr = Tcl_FindHashEntry(&(dataPtr->treeTable), (char *)&key);
	    if (hPtr == NULL) {
		return NULL;
	    }
	    return (TreeObject *)Tcl_GetHashValue(hPtr);
	}
	/* Look first in the current namespace. */
	key.nsPtr = Tcl_GetCurrentNamespace(dataPtr->interp);
	hPtr = Tcl_FindHashEntry(&(dataPtr->treeTable), (char *)&key);
	if (hPtr != NULL) {
	    return (TreeObject *)Tcl_GetHashValue(hPtr);
	}
	/* And then in the global namespace. [Is this a good or bad thing?] */
	key.nsPtr = Tcl_GetGlobalNamespace(dataPtr->interp);
	hPtr = Tcl_FindHashEntry(&(dataPtr->treeTable), (char *)&key);
	if (hPtr != NULL) {
	    return (TreeObject *)Tcl_GetHashValue(hPtr);
	}
    }
    return NULL;
}

static void
ResetDepths(nodePtr, depth)
    Node *nodePtr;
    int depth;
{
    Blt_ChainLink *linkPtr;

    nodePtr->depth = depth;
    for (linkPtr = Blt_ChainFirstLink(nodePtr->chainPtr); linkPtr != NULL; 
	linkPtr = Blt_ChainNextLink(linkPtr)) {
	nodePtr = (Node *)Blt_ChainGetValue(linkPtr);
	ResetDepths(nodePtr, depth + 1);
    }
}

static void
DeleteBranch(nodePtr)
    Node *nodePtr;
{
    if (nodePtr->chainPtr != NULL) {
	Blt_ChainLink *linkPtr, *nextPtr;
	Node *childPtr;

	for (linkPtr = Blt_ChainFirstLink(nodePtr->chainPtr); 
	     linkPtr != NULL; linkPtr = nextPtr) {
	    nextPtr = Blt_ChainNextLink(linkPtr);
	    childPtr = (Node *)Blt_ChainGetValue(linkPtr);
	    childPtr->linkPtr = NULL;
	    DeleteBranch(childPtr);
	}
	Blt_ChainDestroy(nodePtr->chainPtr);
	nodePtr->chainPtr = NULL;
    }
    /* Remove the top-most node. */
    DeleteNode(nodePtr);
}

static void
DestroyTreeObject(treePtr)
    TreeObject *treePtr;
{
    Blt_ChainLink *linkPtr;
    TreeToken *tokenPtr;

    treePtr->nNodes = 0;
    if (treePtr->notifyFlags & NOTIFY_PENDING) {
	treePtr->notifyFlags &= ~NOTIFY_PENDING;
	Tcl_CancelIdleCall(NotifyClients, (ClientData)treePtr);
    }
    /* Remove the remaining clients. */
    for (linkPtr = Blt_ChainFirstLink(treePtr->chainPtr); linkPtr != NULL;
	linkPtr = Blt_ChainNextLink(linkPtr)) {
	tokenPtr = (TreeToken *)Blt_ChainGetValue(linkPtr);
	free((char *)tokenPtr);
    }
    Blt_ChainDestroy(treePtr->chainPtr);

    DeleteBranch(treePtr->root);

    Tcl_DeleteHashTable(&(treePtr->atomTable));
    Tcl_DeleteHashTable(&(treePtr->nodeTable));
    if (treePtr->hashPtr != NULL) {
	/* Remove the entry from the global tree table. */
	Tcl_DeleteHashEntry(treePtr->hashPtr); 
    }
#ifdef NAMESPACE_DELETE_NOTIFY
    if (treePtr->nsPtr != NULL) {
	Blt_DestroyNsDeleteNotify(treePtr->interp, treePtr->nsPtr, 
		  (ClientData)treePtr);
    }
#endif /* NAMESPACE_DELETE_NOTIFY */
    if (treePtr->nameUid != NULL) {
	Blt_FreeUid(treePtr->nameUid);
    }
    if (treePtr->fullName != NULL) {
	free((char *)treePtr->fullName);
    }
    free((char *)treePtr);
}

/*
 * -----------------------------------------------------------------------
 *
 * TreeInterpDeleteProc --
 *
 *	This is called when the interpreter hosting the tree object
 *	is deleted from the interpreter.  
 *
 * Results:
 *	None.
 *
 * Side effects:
 *	Destroys all remaining trees.  In addition removes
 *	the hash table managing all vector names.
 *
 * ------------------------------------------------------------------------
 */
/* ARGSUSED */
static void
TreeInterpDeleteProc(clientData, interp)
    ClientData clientData;	/* Thread-specific data. */
    Tcl_Interp *interp;
{
    ThreadData *dataPtr = (ThreadData *)clientData;
    Tcl_HashEntry *hPtr;
    Tcl_HashSearch cursor;
    TreeObject *treePtr;
    
    for (hPtr = Tcl_FirstHashEntry(&(dataPtr->treeTable), &cursor);
	 hPtr != NULL; hPtr = Tcl_NextHashEntry(&cursor)) {
	treePtr = (TreeObject *)Tcl_GetHashValue(hPtr);
	treePtr->hashPtr = NULL;
	DestroyTreeObject(treePtr);
    }
    Tcl_DeleteHashTable(&(dataPtr->treeTable));
    Tcl_DeleteAssocData(interp, TREE_THREAD_KEY);
    free((char *)dataPtr);
}

static ThreadData *
GetTreeThreadData(interp)
    Tcl_Interp *interp;
{
    ThreadData *dataPtr;
    Tcl_InterpDeleteProc *proc;

    dataPtr = (ThreadData *)Tcl_GetAssocData(interp, TREE_THREAD_KEY, &proc);
    if (dataPtr == NULL) {
	dataPtr = (ThreadData *)malloc(sizeof(ThreadData));
	assert(dataPtr);
	dataPtr->interp = interp;
	Tcl_SetAssocData(interp, TREE_THREAD_KEY, TreeInterpDeleteProc,
		 (ClientData)dataPtr);
	Tcl_InitHashTable(&(dataPtr->treeTable), sizeof(TreeKey)/sizeof(int));
    }
    return dataPtr;
}


/* Public Routines */

Blt_TreeNode
Blt_TreeCreateNode(tokenPtr, parentPtr, name, pos)
    TreeToken *tokenPtr;
    Node *parentPtr;		/* Parent node where the new node will
				 * be inserted. */
    char *name;			/* Name of node. */
    int pos;			/* Position in the parent's list of children
				 * where to insert the new node. */
{
    TreeObject *treePtr = parentPtr->treePtr;
    Node *nodePtr;	/* Node to be inserted. */
    Blt_ChainLink *linkPtr;
    TreeToken *clientPtr;

    if (parentPtr->chainPtr == NULL) {
	/* 
	 * Create chains to hold children only as necessary.  We don't
	 * want to allocate extra storage for leaves (that have no
	 * children) when there can be many.  
	 */
	parentPtr->chainPtr = Blt_ChainCreate();
    }
    linkPtr = Blt_ChainNewLink();
    nodePtr = NewNode(treePtr, name);
    Blt_ChainSetValue(linkPtr, (ClientData)nodePtr);

    if ((pos == -1) || (pos >= Blt_ChainGetLength(parentPtr->chainPtr))) {
	Blt_ChainAppendLink(parentPtr->chainPtr, linkPtr);
    } else {
	Blt_ChainLink *beforePtr;

	beforePtr = Blt_ChainGetNthLink(parentPtr->chainPtr, pos);
	Blt_ChainLinkBefore(parentPtr->chainPtr, linkPtr, beforePtr);
    }
    nodePtr->depth = parentPtr->depth + 1;
    nodePtr->parent = parentPtr;
    nodePtr->linkPtr = linkPtr;
    /* 
     * Issue callbacks to each client indicating that a new node has
     * been created.
     */
    for (linkPtr = Blt_ChainFirstLink(treePtr->chainPtr);
	linkPtr != NULL; linkPtr = Blt_ChainNextLink(linkPtr)) {
	clientPtr = (TreeToken *)Blt_ChainGetValue(linkPtr);
	if ((clientPtr->procsPtr->notifyCreateProc == NULL) ||
	    (clientPtr == tokenPtr)) {
	    continue;
	}
	if ((*clientPtr->procsPtr->notifyCreateProc) (clientPtr, nodePtr, 
		clientPtr->clientData) != TCL_OK) {
	    Tcl_BackgroundError(treePtr->interp);
	}
    }
    return nodePtr;
}

/*
 *----------------------------------------------------------------------
 *
 * Blt_TreeMoveNode --
 *
 *	Move an entry into a new location in the hierarchy.
 *
 *
 *----------------------------------------------------------------------
 */
/*ARGSUSED*/
int
Blt_TreeMoveNode(tokenPtr, nodePtr, parentPtr, beforePtr)
    TreeToken *tokenPtr;
    Node *nodePtr, *parentPtr, *beforePtr;
{
    TreeObject *treePtr = nodePtr->treePtr;
    int depth;

    /* Verify they aren't ancestors. */
    if (Blt_TreeIsAncestor(nodePtr, parentPtr)) {
	return TCL_ERROR;
    }
    Blt_ChainUnlinkLink(nodePtr->parent->chainPtr, nodePtr->linkPtr);
    if (beforePtr == NULL) {
	Blt_ChainLinkAfter(parentPtr->chainPtr, nodePtr->linkPtr, NULL);
    } else {
	Blt_ChainLinkBefore(parentPtr->chainPtr, nodePtr->linkPtr, 
		beforePtr->linkPtr);
    }
    nodePtr->parent = parentPtr;
    depth = parentPtr->depth + 1;
    if (nodePtr->depth != depth) {
	/* Descend the branch resetting the depths. */
	ResetDepths(nodePtr, depth);
    }
    if (!(treePtr->flags & NOTIFY_PENDING)) {
	treePtr->flags |= NOTIFY_PENDING;
	Tcl_DoWhenIdle(NotifyClients, (ClientData)tokenPtr);
    }
    return TCL_OK;
}

int
Blt_TreeDeleteNode(tokenPtr, nodePtr)
    TreeToken *tokenPtr;
    Node *nodePtr;
{
    TreeObject *treePtr = nodePtr->treePtr;
    Blt_ChainLink *linkPtr;
    TreeToken *clientPtr;

    if (nodePtr->chainPtr != NULL) {
	Node *childPtr;

	/* In depth-first order, delete each descendant node. */
	for (linkPtr = Blt_ChainFirstLink(nodePtr->chainPtr); 
	     linkPtr != NULL; linkPtr = Blt_ChainNextLink(linkPtr)) {
	    childPtr = (Node *)Blt_ChainGetValue(linkPtr);
	    Blt_TreeDeleteNode(tokenPtr, childPtr);
	}
	Blt_ChainDestroy(nodePtr->chainPtr);
	nodePtr->chainPtr = NULL;
    }

    /* 
     * Issue callbacks to each client indicating that the node can
     * no longer be used.  
     */
    for (linkPtr = Blt_ChainFirstLink(treePtr->chainPtr);
	linkPtr != NULL; linkPtr = Blt_ChainNextLink(linkPtr)) {
	clientPtr = (TreeToken *)Blt_ChainGetValue(linkPtr);
	if ((clientPtr->procsPtr->notifyDeleteProc == NULL) ||
	    (clientPtr == tokenPtr)) {
	    continue;
	}
	if ((*clientPtr->procsPtr->notifyDeleteProc) (clientPtr, nodePtr, 
	     clientPtr->clientData) != TCL_OK) {
	    Tcl_BackgroundError(treePtr->interp);
	}
    }
    /* Now remove the actual node. */
    DeleteNode(nodePtr);
    return TCL_OK;
}

Blt_TreeNode
Blt_TreeGetNode(tokenPtr, inode)
    TreeToken *tokenPtr;
    unsigned int inode;
{
    TreeObject *treePtr = tokenPtr->tree;
    Tcl_HashEntry *hPtr;

    hPtr = Tcl_FindHashEntry(&(treePtr->nodeTable), (char *)inode);
    if (hPtr != NULL) {
	return (Blt_TreeNode)Tcl_GetHashValue(hPtr);
    }
    return NULL;
}

/*
 *----------------------------------------------------------------------
 *
 * Blt_TreeFindChild --
 *
 *	Searches for the named node in a parent's chain of siblings.  
 *
 *
 * Results:
 *	If found, the child node is returned, otherwise NULL.
 *
 *----------------------------------------------------------------------
 */
Blt_TreeNode
Blt_TreeFindChild(parentPtr, name)
    Node *parentPtr;
    char *name;
{
    Blt_TreeAtom nameAtom;
    
    nameAtom = GetAtom(parentPtr->treePtr, name);
    if (nameAtom != NULL) {
	Node *nodePtr;
	register Blt_ChainLink *linkPtr;

	for (linkPtr = Blt_ChainFirstLink(parentPtr->chainPtr); 
	     linkPtr != NULL; linkPtr = Blt_ChainNextLink(linkPtr)) {
	    nodePtr = (Node *) Blt_ChainGetValue(linkPtr);
	    if (nameAtom == nodePtr->atom) {
		return nodePtr;
	    }
	}
    }
    return NULL;
}

/*
 *----------------------------------------------------------------------
 *
 * Blt_TreePrevNode --
 *
 *	Returns the "previous" node in the tree.  This node (in 
 *	depth-first order) is its parent, if the node has no siblings
 *	that are previous to it.  Otherwise it is the last descendant 
 *	of the last sibling.  In this case, descend the sibling's
 *	hierarchy, using the last child at any ancestor, with we
 *	we find a leaf.
 *
 *----------------------------------------------------------------------
 */
Blt_TreeNode
Blt_TreePrevNode(rootPtr, nodePtr, nodeFlags)
    Node *rootPtr, *nodePtr;
    unsigned int nodeFlags;
{
    Blt_ChainLink *linkPtr;

    if (nodePtr == rootPtr) {
	return NULL;		/* The root is the first node. */
    }
    linkPtr = Blt_ChainPrevLink(nodePtr->linkPtr);
    if (linkPtr == NULL) {
	/* There are no siblings previous to this one, so pick the parent. */
	return nodePtr->parent;
    }
    /*
     * Traverse down the right-most thread, in order to select the
     * next entry.  Stop if we find a "closed" entry or reach a leaf.
     */
    nodePtr = (Node *)Blt_ChainGetValue(linkPtr);
    while ((nodePtr->flags & nodeFlags) == nodeFlags) {
	linkPtr = Blt_ChainLastLink(nodePtr->chainPtr);
	if (linkPtr == NULL) {
	    break;		/* Found a leaf. */
	}
	nodePtr = (Node *)Blt_ChainGetValue(linkPtr);
    }
    return nodePtr;
}

/*
 *----------------------------------------------------------------------
 *
 * Blt_TreeNextNode --
 *
 *	Returns the "next" node in relation to the given node.  
 *	The next node (in depth-first order) is either the first 
 *	child of the given node the next sibling if the node has
 *	no children (the node is a leaf).  If the given node is the 
 *	last sibling, then try it's parent next sibling.  Continue
 *	until we either find a next sibling for some ancestor or 
 *	we reach the root node.  In this case the current node is 
 *	the last node in the tree.
 *
 *----------------------------------------------------------------------
 */
Blt_TreeNode
Blt_TreeNextNode(rootPtr, nodePtr, nodeFlags)
    Node *rootPtr, *nodePtr;
    unsigned int nodeFlags;
{
    Blt_ChainLink *linkPtr;

    if ((nodePtr->flags & nodeFlags) == nodeFlags) {
	/* Pick the first sub-node. */
	linkPtr = Blt_ChainFirstLink(nodePtr->chainPtr);
	if (linkPtr != NULL) {
	    return (Blt_TreeNode)Blt_ChainGetValue(linkPtr);
	}
    }
    /* 
     * Back up until we can find a level where we can pick a 
     * "next sibling".  For the last entry we'll thread our 
     * way back to the root.  
     */
    while (nodePtr != rootPtr) {
	linkPtr = Blt_ChainNextLink(nodePtr->linkPtr);
	if (linkPtr != NULL) {
	    return (Blt_TreeNode)Blt_ChainGetValue(linkPtr);
	}
	nodePtr = nodePtr->parent;
    }
    return NULL;		/* At root, no next node. */
}


int
Blt_TreeIsBefore(n1Ptr, n2Ptr)
    Node *n1Ptr, *n2Ptr;
{
    int depth;
    register int i;
    Blt_ChainLink *linkPtr;
    Node *nodePtr;

    depth = MIN(n1Ptr->depth, n2Ptr->depth);
    if (depth == 0) {		/* One of the nodes is root. */
	return (n1Ptr->parent == NULL);
    }
    /* 
     * Traverse back from the deepest node, until the both nodes are
     * at the same depth.  Check if the ancestor node found is the
     * other node.  
     */
    for (i = n1Ptr->depth; i > depth; i--) {
	n1Ptr = n1Ptr->parent;
    }
    if (n1Ptr == n2Ptr) {
	return FALSE;
    }
    for (i = n2Ptr->depth; i > depth; i--) {
	n2Ptr = n2Ptr->parent;
    }
    if (n2Ptr == n1Ptr) {
	return TRUE;
    }

    /* 
     * First find the mutual ancestor of both nodes.  Look at each
     * preceding ancestor level-by-level for both nodes.  Eventually
     * we'll find a node that's the parent of both ancestors.  Then
     * find the first ancestor in the parent's list of subnodes.  
     */
    for (i = depth; i > 0; i--) {
	if (n1Ptr->parent == n2Ptr->parent) {
	    break;
	}
	n1Ptr = n1Ptr->parent;
	n2Ptr = n2Ptr->parent;
    }
    for (linkPtr = Blt_ChainFirstLink(n1Ptr->parent->chainPtr); 
	 linkPtr != NULL; linkPtr = Blt_ChainNextLink(linkPtr)) {
	nodePtr = (Node *)Blt_ChainGetValue(linkPtr);
	if (nodePtr == n1Ptr) {
	    return TRUE;
	} else if (nodePtr == n2Ptr) {
	    return FALSE;
	}
    }
    assert(linkPtr != NULL);
    return FALSE;
}

int
Blt_TreeGetValue(nodePtr, fieldAtom, objPtrPtr)
    Node *nodePtr;
    Blt_TreeAtom fieldAtom;
    Tcl_Obj **objPtrPtr;
{
    register Blt_ChainLink *linkPtr;
    register Datum *datumPtr;

    for (linkPtr = Blt_ChainFirstLink(nodePtr->dataPtr); linkPtr != NULL;
	 linkPtr = Blt_ChainNextLink(linkPtr)) {
	datumPtr = (Datum *)Blt_ChainGetValue(linkPtr);
	if (datumPtr->fieldAtom == fieldAtom) {
	    *objPtrPtr = datumPtr->objPtr;
	    return TCL_OK;
	}
    }
    return TCL_ERROR;
}

int
Blt_TreeSetValue(tokenPtr, nodePtr, fieldAtom, objPtr)
    TreeToken *tokenPtr;
    Node *nodePtr;		/* Node to be updated. */
    Blt_TreeAtom fieldAtom;	/* Atom identifying the field in node. */
    Tcl_Obj *objPtr;		/* New value of field. If NULL, field
				 * is deleted. */
{
    TreeObject *treePtr = nodePtr->treePtr;
    register Blt_ChainLink *linkPtr;
    Datum *datumPtr;
    TreeToken *clientPtr;

    datumPtr = NULL;		/* Suppress compiler warning. */

    /* See if a data field by the given name exists. */
    for (linkPtr = Blt_ChainFirstLink(nodePtr->dataPtr); linkPtr != NULL;
	 linkPtr = Blt_ChainNextLink(linkPtr)) {
	datumPtr = (Datum *)Blt_ChainGetValue(linkPtr);
	if (datumPtr->fieldAtom == fieldAtom) {
	    break;
	}
    }
    if (objPtr == NULL) {	/* NULL Tcl_Obj: Delete field. */
	if (linkPtr != NULL) {
	    Tcl_DecrRefCount(datumPtr->objPtr);
	    FreeAtom(treePtr, datumPtr->fieldAtom);
	    Blt_ChainDeleteLink(nodePtr->dataPtr, linkPtr);
	    free((char *)datumPtr);
	}
    } else {
	if (linkPtr == NULL) {
	    /* Create a new one and append it to the node's chain. */
	    datumPtr = (Datum *)malloc(sizeof(Datum));
	    datumPtr->fieldAtom = GetAtom(treePtr, fieldAtom);
	    Tcl_IncrRefCount(objPtr);
	    Blt_ChainAppend(nodePtr->dataPtr, (ClientData)datumPtr); 
	} else {
	    Tcl_IncrRefCount(objPtr);
	    if (datumPtr->objPtr != objPtr) {
		Tcl_DecrRefCount(datumPtr->objPtr);
	    }
	}
	datumPtr->objPtr = objPtr;
    }
    for(linkPtr = Blt_ChainFirstLink(treePtr->chainPtr); 
	linkPtr != NULL; linkPtr = Blt_ChainNextLink(linkPtr)) {
	clientPtr = (TreeToken *)Blt_ChainGetValue(linkPtr);
	if ((clientPtr->procsPtr->notifySetProc == NULL) ||
	    (clientPtr == tokenPtr)) {
	    continue;
	}
	if ((*clientPtr->procsPtr->notifySetProc) (clientPtr, nodePtr, 
		fieldAtom, clientPtr->clientData) != TCL_OK) {
	    Tcl_BackgroundError(treePtr->interp);
	}
    }
    return TCL_OK;
}

int
Blt_TreeUnsetValue(tokenPtr, nodePtr, fieldAtom)
    TreeToken *tokenPtr;
    Node *nodePtr;		/* Node to be updated. */
    Blt_TreeAtom fieldAtom;	/* Name of field in node. */
{
    TreeObject *treePtr = nodePtr->treePtr;
    register Blt_ChainLink *linkPtr;
    Datum *datumPtr;
    TreeToken *clientPtr;

    for (linkPtr = Blt_ChainFirstLink(nodePtr->dataPtr); linkPtr != NULL;
	 linkPtr = Blt_ChainNextLink(linkPtr)) {
	datumPtr = (Datum *)Blt_ChainGetValue(linkPtr);
	if (datumPtr->fieldAtom == fieldAtom) {
	    break;
	}
    }
    if (linkPtr == NULL) {
	return TCL_OK;		/* No value was already set. */
    }
    FreeAtom(treePtr, datumPtr->fieldAtom);
    Tcl_DecrRefCount(datumPtr->objPtr);
    free((char *)datumPtr);
    Blt_ChainDeleteLink(nodePtr->dataPtr, linkPtr);

    for(linkPtr = Blt_ChainFirstLink(treePtr->chainPtr); 
	linkPtr != NULL; linkPtr = Blt_ChainNextLink(linkPtr)) {
	clientPtr = (TreeToken *)Blt_ChainGetValue(linkPtr);
	if ((clientPtr->procsPtr->notifyUnsetProc == NULL) ||
	    (clientPtr == tokenPtr)) {
	    continue;
	}
	if ((*clientPtr->procsPtr->notifyUnsetProc) (clientPtr, nodePtr, 
		fieldAtom, clientPtr->clientData) != TCL_OK) {
	    Tcl_BackgroundError(treePtr->interp);
	}
    }
    return TCL_OK;
}

int
Blt_TreeEnumFields(nodePtr, proc)
    Node *nodePtr;
    Blt_TreeEnumProc *proc;
{
    register Blt_ChainLink *linkPtr;
    Datum *datumPtr;
    int result;

    for (linkPtr = Blt_ChainFirstLink(nodePtr->dataPtr); linkPtr != NULL;
	 linkPtr = Blt_ChainNextLink(linkPtr)) {
	datumPtr = (Datum *)Blt_ChainGetValue(linkPtr);
	result = (*proc)(nodePtr, datumPtr->fieldAtom, datumPtr->objPtr);
	if (result != TCL_OK) {
	    return result;
	}
    }
    return TCL_OK;
}

Blt_TreeNode
Blt_TreeFirstChild(parentPtr)
    Node *parentPtr;
{
    if (parentPtr->chainPtr != NULL) {
	Blt_ChainLink *linkPtr;

	linkPtr = Blt_ChainFirstLink(parentPtr->chainPtr);
	if (linkPtr != NULL) {
	    return (Blt_TreeNode)Blt_ChainGetValue(linkPtr);
	}
    }
    return NULL;
}

Blt_TreeNode
Blt_TreeNextSibling(nodePtr)
    Node *nodePtr;
{
    Blt_ChainLink *linkPtr;

    linkPtr = Blt_ChainNextLink(nodePtr->linkPtr);
    if (linkPtr != NULL) {
	return (Blt_TreeNode)Blt_ChainGetValue(linkPtr);
    }
    return NULL;
}

Blt_TreeNode
Blt_TreeLastChild(parentPtr)
    Node *parentPtr;
{
    if (parentPtr->chainPtr != NULL) {
	Blt_ChainLink *linkPtr;

	linkPtr = Blt_ChainLastLink(parentPtr->chainPtr);
	if (linkPtr != NULL) {
	    return (Blt_TreeNode) Blt_ChainGetValue(linkPtr);
	}
    }
    return NULL;
}

Blt_TreeNode
Blt_TreePrevSibling(nodePtr)
    Node *nodePtr;
{
    Blt_ChainLink *linkPtr;

    linkPtr = Blt_ChainPrevLink(nodePtr->linkPtr);
    if (linkPtr != NULL) {
	return (Blt_TreeNode)Blt_ChainGetValue(linkPtr);
    }
    return NULL;
}

int
Blt_TreeIsAncestor(rootPtr, nodePtr)
    Node *rootPtr, *nodePtr;
{
    if (nodePtr != NULL) {
	nodePtr = nodePtr->parent;
	while (nodePtr != NULL) {
	    if (nodePtr == rootPtr) {
		return TRUE;
	    }
	    nodePtr = nodePtr->parent;
	}
    }
    return FALSE;
}

/*
 *----------------------------------------------------------------------
 *
 * Blt_TreeSortNode --
 *
 *	Sorts the subnodes at a given node.
 *
 * Results:
 *	Always returns TCL_OK.
 *
 *----------------------------------------------------------------------
 */
int
Blt_TreeSortNode(nodePtr, proc)
    Node *nodePtr;
    Blt_TreeCompareNodesProc *proc;
{
    Node **nodeArr;
    register Blt_ChainLink *linkPtr;
    int nNodes;
    register int i;

    nNodes = Blt_ChainGetLength(nodePtr->chainPtr);
    if (nNodes < 2) {
	return TCL_ERROR;
    }
    nodeArr = (Node **)malloc(sizeof(Node *) * (nNodes + 1));
    if (nodeArr == NULL) {
	return TCL_ERROR;	/* Out of memory. */
    }
    i = 0;
    for (linkPtr = Blt_ChainFirstLink(nodePtr->chainPtr); linkPtr != NULL; 
	linkPtr = Blt_ChainNextLink(linkPtr)) {
	nodeArr[i++] = (Node *)Blt_ChainGetValue(linkPtr);
    }
    qsort((char *)nodeArr, nNodes, sizeof(Node *),
	(QSortCompareProc *)proc);

    i = 0;
    for (linkPtr = Blt_ChainFirstLink(nodePtr->chainPtr); linkPtr != NULL; 
	linkPtr = Blt_ChainNextLink(linkPtr)) {
	Blt_ChainSetValue(linkPtr, nodeArr[i++]);
    }
    free((char *)nodeArr);
    return TCL_OK;
}


int
Blt_TreeApply(nodePtr, proc, clientData)
    Node *nodePtr;		/* Root node of subtree. */
    Blt_TreeApplyProc *proc;	/* Procedure to call for each node. */
    ClientData clientData;	/* One-word of data passed when calling
				 * proc. */
{
    Blt_ChainLink *linkPtr, *nextPtr;
    Node *childPtr;

    nextPtr = NULL;
    for (linkPtr = Blt_ChainFirstLink(nodePtr->chainPtr); linkPtr != NULL; 
	linkPtr = nextPtr) {
	/* 
	 * Get the next link in the chain before calling
	 * Blt_TreeApply recursively.  This is because the 
	 * apply callback may delete the node and its link.  
	 */
	nextPtr = Blt_ChainNextLink(linkPtr);
	childPtr = (Node *)Blt_ChainGetValue(linkPtr);
	if (Blt_TreeApply(childPtr, proc, clientData) != TCL_OK) {
	    return TCL_ERROR;
	}
    }
    if ((*proc) (nodePtr, clientData) != TCL_OK) {
	return TCL_ERROR;
    }
    return TCL_OK;
}

int
Blt_TreeCreate(interp, pathName)
    Tcl_Interp *interp;		/* Interpreter to report errors back to. */
    char *pathName;		/* Namespace qualified pathname of tree.
				 * Tree must not already exist. */
{
    TreeObject *treePtr;
    ThreadData *dataPtr;
    TreeKey key;
    char *name;
    Tcl_Namespace *nsPtr;
    Tcl_HashEntry *hPtr;
    char string[200];
    int isNew;
    int length;

    dataPtr = GetTreeThreadData(interp);
    nsPtr = NULL;
    name = NULL;
    if (pathName != NULL) {
	if (Blt_ParseQualifiedName(interp, pathName, &nsPtr, &name) != TCL_OK) {
	    Tcl_AppendResult(interp, "can't find namespace in \"", pathName, 
		     "\"", (char *)NULL);
	    return TCL_ERROR;
	}
    }
    if (nsPtr == NULL) {
	nsPtr = Tcl_GetCurrentNamespace(interp);
    }
    key.nsPtr = nsPtr;
    if (name != NULL) {
	/* Check if this tree already exists the current namespace. */
	treePtr = GetTreeObject(dataPtr, name);
	if (treePtr != NULL) {
	    Tcl_AppendResult(interp, "a tree object \"", name,
			     "\" already exists", (char *)NULL);
	    return TCL_ERROR;
	}
	key.id = Blt_GetUid(name);
    } else {
	/* Generate a unique tree name in the current namespace. */
	do {
	    sprintf(string, "tree%d", dataPtr->nextId++);
	    key.id = Blt_GetUid(string);
	    hPtr = Tcl_FindHashEntry(&(dataPtr->treeTable), (char *)&key);
	} while (hPtr != NULL);
	name = string;
    } 
    hPtr = Tcl_CreateHashEntry(&(dataPtr->treeTable),(char *)&key, &isNew);
    treePtr = NewTreeObject(interp, key.id, nsPtr);
    if (treePtr == NULL) {
	Tcl_SetResult(interp, "can't allocate tree", TCL_STATIC);
	return TCL_ERROR;
    }
    treePtr->hashPtr = hPtr;
    length = strlen(treePtr->nameUid) + strlen(nsPtr->fullName) + 3;
    treePtr->fullName = (char *)calloc(length, sizeof(char));
    if (Tcl_GetGlobalNamespace(interp) != nsPtr) {
	strcpy(treePtr->fullName, nsPtr->fullName);
    }
    strcat(treePtr->fullName, "::");
    strcat(treePtr->fullName, treePtr->nameUid);
    
#ifdef NAMESPACE_DELETE_NOTIFY
    Blt_CreateNsDeleteNotify(interp, nsPtr, (ClientData)treePtr,
		TreeInstDeleteProc);
#endif /* NAMESPACE_DELETE_NOTIFY */
    Tcl_SetHashValue(hPtr, (char *)treePtr);
    return TCL_OK;
}

Blt_Tree
Blt_TreeGetToken(interp, pathName, procsPtr, clientData)
    Tcl_Interp *interp;		/* Interpreter to report errors back to. */
    char *pathName;		/* Namespace qualified pathname of tree. */
    Blt_TreeNotifyProcs *procsPtr; /* Set of notifier callbacks  */
    ClientData clientData;	/* One-word of data to be passed when calling
				 * the notifier routines. */
{
    TreeToken *tokenPtr;
    ThreadData *dataPtr;
    TreeObject *treePtr;

    dataPtr = GetTreeThreadData(interp);
    treePtr = GetTreeObject(dataPtr, pathName);
    if (treePtr == NULL) {
	Tcl_AppendResult(interp, "can't find a tree object \"", pathName, "\"",
			 (char *)NULL);
	return NULL;
    }
    tokenPtr = (TreeToken *)calloc(1, sizeof(TreeToken));
    if (tokenPtr == NULL) {
	Tcl_SetResult(interp, "can't allocate tree token", TCL_STATIC);
	return NULL;
    }
    tokenPtr->magic = TREE_MAGIC;
    tokenPtr->procsPtr = procsPtr;
    tokenPtr->clientData = clientData;
    tokenPtr->notifyFlags = NOTIFY_ALL;
    tokenPtr->linkPtr = Blt_ChainAppend(treePtr->chainPtr, 
	(ClientData)tokenPtr);
    tokenPtr->tree = treePtr;
    return tokenPtr;
}

unsigned int
Blt_TreeGetNotifyFlags(tokenPtr)
    TreeToken *tokenPtr;
{
    if (tokenPtr->magic != TREE_MAGIC) {
	fprintf(stderr, "invalid tree object token 0x%lx\n", 
		(unsigned long)tokenPtr);
	return 0;
    }
    return tokenPtr->notifyFlags;
}

void
Blt_TreeSetNotifyFlags(tokenPtr, mask)
    TreeToken *tokenPtr;
    unsigned int mask;
{
    if (tokenPtr->magic != TREE_MAGIC) {
	fprintf(stderr, "invalid tree object token 0x%lx\n", 
		(unsigned long)tokenPtr);
	return;
    }
    tokenPtr->notifyFlags = mask;
}

void
Blt_TreeReleaseToken(tokenPtr)
    TreeToken *tokenPtr;
{
    TreeObject *treePtr;

    if (tokenPtr->magic != TREE_MAGIC) {
	fprintf(stderr, "invalid tree object token 0x%lx\n", 
		(unsigned long)tokenPtr);
	return;
    }
    treePtr = tokenPtr->tree;
    if (treePtr != NULL) {
	/* Remove the client from the server's list */
	Blt_ChainDeleteLink(treePtr->chainPtr, tokenPtr->linkPtr);
	if (Blt_ChainGetLength(treePtr->chainPtr) == 0) {
	    DestroyTreeObject(treePtr);
	}
    }
    tokenPtr->magic = 0;
    free((char *)tokenPtr);
}

/*
 *----------------------------------------------------------------------
 *
 * Blt_TreeGetAtom --
 *
 *	Gets or creates a unique string identifier.  Strings are
 *	reference counted.  The string is placed into a hashed table
 *	local to the hierbox.
 *
 * Results:
 *	Returns the pointer to the hashed string.
 *
 *---------------------------------------------------------------------- 
 */
Blt_TreeAtom
Blt_TreeGetAtom(tokenPtr, string)
    TreeToken *tokenPtr;
    char *string;
{
    TreeObject *treePtr = tokenPtr->tree;
    return GetAtom(treePtr, string);
}

/*
 *----------------------------------------------------------------------
 *
 * Blt_TreeFreeAtom --
 *
 *	Releases the atom.  Atoms are reference counted, so only when
 *	the reference count is zero (i.e. no one else is using the
 *	string) is the entry removed from the hash table.
 *
 * Results:
 *	None.
 *
 *---------------------------------------------------------------------- 
 */
void
Blt_TreeFreeAtom(tokenPtr, atom)
    TreeToken *tokenPtr;
    Blt_TreeAtom atom;
{
    TreeObject *treePtr = tokenPtr->tree;
    FreeAtom(treePtr, atom);
}

#endif /* NO_TREE */
