// Swarm library. Copyright  1996-1999 Santa Fe Institute.
// This library is distributed without any warranty; without even the
// implied warranty of merchantability or fitness for a particular purpose.
// See file LICENSE for details and terms of copying.

#import <defobj/internal.h>
#import <defobj.h> // raiseEvent, DSIZE, SSTRDUP

#include <misc.h> // strtoul, isDigit
#include <objc/objc-api.h>

#define TYPE_SHORT "short"
#define TYPE_UNSIGNED_SHORT "unsigned short"
#define TYPE_INT "int"
#define TYPE_UNSIGNED "unsigned"
#define TYPE_LONG "long"
#define TYPE_UNSIGNED_LONG "unsigned long"
#define TYPE_LONG_LONG "long long"
#define TYPE_UNSIGNED_LONG_LONG "unsigned long long"
#define TYPE_FLOAT "float"
#define TYPE_DOUBLE "double"
#define TYPE_LONG_DOUBLE "long double"
#define TYPE_STRING "string"
#define TYPE_OBJECT "object"

size_t
alignsizeto (size_t pos, size_t alignment)
{
  size_t mask = (alignment - 1);

  if ((pos & mask) == 0)
    return pos;
  else
    return (pos + alignment) & ~mask;
}

void *
alignptrto (void *ptr, size_t alignment)
{
  PTRUINT mask = (alignment - 1);
  PTRUINT pos = (PTRUINT) ptr;

  if ((pos & mask) == 0)
    return ptr;
  else
    return (void *) ((pos + alignment) & ~mask);
}

size_t
alignment_for_objc_type (const char *varType)
{
  size_t alignment = 0;

  switch (*varType)
    {
    case _C_SHT: case _C_USHT:
      alignment = __alignof__ (short);
      break;
    case _C_INT: case _C_UINT:
      alignment = __alignof__ (int);
      break;
    case _C_LNG: case _C_ULNG:
      alignment = __alignof__ (long);
      break;
    case _C_LNG_LNG: case _C_ULNG_LNG:
      alignment = __alignof__ (long long);
      break;
    case _C_FLT:
      alignment = __alignof__ (float);
      break;
    case _C_DBL:
      alignment = __alignof__ (double);
      break;
    case _C_LNG_DBL:
      alignment = __alignof__ (long double);
      break;
    case _C_CHR: case _C_UCHR:
      alignment = __alignof__ (char);
      break;
    case _C_CHARPTR:
      alignment = __alignof__ (const char *);
      break;
    case _C_ID:
      alignment = __alignof__ (id);
      break;
    case _C_CLASS:
      alignment = __alignof__ (Class);
      break;
    case _C_ARY_B:
      varType++;
      while (isDigit (*varType))
        varType++;
      
      alignment = alignment_for_objc_type (varType);
      break;

    default:
      abort ();
    }
  return alignment;
}

size_t
size_for_objc_type (const char *varType)
{
  size_t size;

  switch (*varType)
    {
    case _C_INT: case _C_UINT:
      size = sizeof (int);
      break;
    case _C_SHT: case _C_USHT:
      size = sizeof (short);
      break;
    case _C_LNG: case _C_ULNG:
      size = sizeof (long);
      break;
    case _C_LNG_LNG: case _C_ULNG_LNG:
      size = sizeof (long long);
      break;
    case _C_CHR: case _C_UCHR:
      size = sizeof (char);
      break;
    case _C_FLT:
      size = sizeof (float);
      break;
    case _C_DBL:
      size = sizeof (double);
      break;
    case _C_LNG_DBL:
      size = sizeof (long double);
      break;
    case _C_CHARPTR:
      size = sizeof (const char *);
      break;
    case _C_ID:
      size = sizeof (id);
      break;
    case _C_CLASS:
      size = sizeof (Class);
      break;
    case _C_ARY_B:
      {
        char *tail;
        unsigned count = strtoul (varType + 1, &tail, 10);
       
        size = count * size_for_objc_type (tail);
      }
      break;
    default:
      abort ();
    }
  return size;
}

void
process_array (const char *type,
               void (*setup_array) (unsigned rank, unsigned *dims,
                                    const char *baseType),
               void (*start_dim) (unsigned dimnum),
               void (*end_dim) (void),
               void (*start_element) (void),
               void (*end_element) (void),
               void (*output_type) (const char *type,
                                    unsigned offset,
                                    void *data),
               const void *ptr,
               void *data)
{
  const char *baseType;
  char *tail;

  unsigned get_rank (const char *type)
    {
      unsigned rank = 0;
      
      do {
        
        errno = 0;
        strtoul (type + 1, &tail, 10);
        
        if (errno != 0)
          raiseEvent (InvalidArgument, "Value out of range [%s]", type + 1);
        
        rank++;
        type = tail;
      } while (*tail == _C_ARY_B);
      return rank;
    }
  {
    unsigned rank = get_rank (type);
    unsigned dims[rank];
    
    void fill_dims (const char *type)
      {
        unsigned dimnum = 0;

        do {
          
          errno = 0;
          dims[dimnum] = strtoul (type + 1, &tail, 10);
          if (errno != 0)
            raiseEvent (InvalidArgument, "Value out of range [%s]", type + 1);
          
          dimnum++;
          type = tail;
        } while (*tail == _C_ARY_B);
      }
    
    fill_dims (type);
    baseType = tail;

    if (setup_array)
      setup_array (rank, dims, baseType);

    if (output_type)
      {
        unsigned coord[rank];
        
        void permute (unsigned dim)
          {
            unsigned i;
            
            if (dim < rank)
              {
                if (start_dim)
                  start_dim (dim);
                for (i = 0; i < dims[dim]; i++)
                  {
                    coord[dim] = i;
                    permute (dim + 1);
                  }
                if (end_dim)
                  end_dim ();
              }
            else
              {
                unsigned offset = 0;
                unsigned mult = 1;
                
                offset = coord[rank - 1];
                for (i = rank - 1; i > 0; i--)
                  {
                    mult *= dims[i];
                    offset += coord[i - 1] * mult;
                  }
                if (start_element)
                  start_element ();
                output_type (baseType, offset, NULL);
                if (end_element)
                  end_element ();
              }
          }
        permute (0);
      }
  }
}

void
map_ivars (Class class,
           void (*process_object) (struct objc_ivar *ivar))
{
  struct objc_ivar_list *ivars = class->ivars;
  
  if (class->super_class)
    {
      if (strcmp (class->super_class->name, "Object_s") != 0)
        map_ivars (class->super_class, process_object);
    }
  
  if (ivars)
    {
      unsigned i, ivar_count = ivars->ivar_count;
      struct objc_ivar *ivar_list = ivars->ivar_list;
      
      for (i = 0; i < ivar_count; i++)
        {
          // Special case to allow member_t for setIndexFromMemberLoc: lists.
          if (strcmp (ivar_list[i].ivar_type, "{?=\"memberData\"[2^v]}") == 0)
            continue;
          else if (*ivar_list[i].ivar_type == _C_PTR)
            continue;
          process_object (&ivar_list[i]);
        }
    }
}

struct objc_ivar *
find_ivar (Class class, const char *name)
{
  struct objc_ivar_list *ivars = class->ivars;

  if (class->super_class)
    {
      if (strcmp (class->super_class->name, "Object_s") != 0)
        {
          struct objc_ivar *ret = find_ivar (class->super_class, name);
          
          if (ret)
            return ret;
        }
    }
  
  if (ivars)
    {
      unsigned i, ivar_count = ivars->ivar_count;
      struct objc_ivar *ivar_list = ivars->ivar_list;

      for (i = 0; i < ivar_count; i++)
        {
          if (strcmp (ivar_list[i].ivar_name, name) == 0)
            return &ivars->ivar_list[i];
        }
      return NULL;
    }
  return NULL;
}

void *
ivar_ptr (id obj, const char *name)
{
  struct objc_ivar *ivar = find_ivar ([obj class], name);

  if (ivar)
    return (void *) obj + ivar->ivar_offset;
  return NULL;
}

const char *
objc_type_for_array (const char *baseType, unsigned rank, unsigned *dims)
{
  unsigned i;
  char nbuf[DSIZE (unsigned) + 1];
  char buf[rank * (2 + DSIZE (unsigned)) + strlen (baseType) + 1], *p = buf;

  for (i = 0; i < rank ; i++)
    {
      *p++ = '[';
      sprintf (nbuf, "%u", dims[i]);
      p = stpcpy (p, nbuf);
    }
  p = stpcpy (p, baseType);
  for (i = 0; i < rank ; i++)
    *p++ = ']';
  *p = '\0';
  return SSTRDUP (buf);
}

void
lisp_process_array (const char *type,
                    const void *ptr, void *data,
                    id <OutputStream> stream,
                    BOOL deepFlag)
{
  BOOL firstElement;
  
  void lisp_setup_array (unsigned rank, unsigned *dims, const char *baseType)
    {
      [stream catArrayRank: rank];
    }
  void lisp_start_dim (unsigned dim)
    {
      [stream catStartExpr];
      firstElement = YES;
    }
  void lisp_end_dim (void)
    {
      [stream catEndExpr];
    }
  void lisp_start_element (void)
    {
      if (!firstElement)
        [stream catSeparator];
    }
  void lisp_end_element (void)
    {
      firstElement = NO;
    }
  void lisp_array_output_type (const char *type,
                               unsigned offset,
                               void *data)
    {
      lisp_output_type (type, ptr, offset, data, stream, deepFlag);
    }
    
  process_array (type,
                 lisp_setup_array,
                 lisp_start_dim,
                 lisp_end_dim,
                 lisp_start_element,
                 lisp_end_element,
                 lisp_array_output_type,
                 ptr,
                 data);
  [stream catEndArray];
}

void
lisp_output_type (const char *type,
                  const void *ptr,
                  unsigned offset,
                  void *data,
                  id <OutputStream> stream,
                  BOOL deepFlag)
{
  switch (*type)
    {
    case _C_ID:
      {
        id obj = ((id *) ptr)[offset];

        if (obj == nil || !deepFlag)
          [stream catBoolean: NO];
        else
          [obj lispOutDeep: stream];
        break;
      }
    case _C_CLASS:
      [stream catClass: *(Class *) ptr];
      break;
    case _C_SEL:
      raiseEvent (NotImplemented, "Selectors not supported");
      break;
    case _C_CHR: 
    case _C_UCHR: 
      [stream catChar: ((unsigned char *) ptr)[offset]];
      break;
    case _C_SHT: 
      [stream catShort: ((short *) ptr)[offset]];
      break;
    case _C_USHT: 
      [stream catUnsignedShort: ((unsigned short *) ptr)[offset]];
      break;
    case _C_INT:
      [stream catInt: ((int *) ptr)[offset]];
      break;
    case _C_UINT:
      [stream catUnsigned: ((unsigned *) ptr)[offset]];
      break;
    case _C_LNG:
      [stream catLong: ((long *) ptr)[offset]];
      break;
    case _C_ULNG:
      [stream catUnsignedLong: ((unsigned long *) ptr)[offset]];
      break;
    case _C_LNG_LNG:
      [stream catLongLong: ((long long *) ptr)[offset]];
      break;
    case _C_ULNG_LNG:
      [stream catUnsignedLongLong: ((unsigned long long *) ptr)[offset]];
      break;
    case _C_FLT:
      [stream catFloat:((float *) ptr)[offset]];
      break;
    case _C_DBL:
      [stream catDouble: ((double *) ptr)[offset]];
      break;
    case _C_LNG_DBL:
      [stream catLongDouble: ((long double *) ptr)[offset]];
      break;
    case _C_BFLD:
      raiseEvent (NotImplemented, "Bit fields not supported [%s]", type);
      break;
    case _C_VOID:
      abort ();
      break;
    case _C_UNDEF: 
      abort ();
      break;
    case _C_PTR:
      raiseEvent (NotImplemented, "Pointers not supported [%s]", type);
      break;
    case _C_CHARPTR:
      [stream catString: ((const char **) ptr)[offset]];
      break;
    case _C_ATOM:
      raiseEvent (NotImplemented, "Atoms not supported");
      break;
    case _C_ARY_B:
      lisp_process_array (type, ptr, data, stream, deepFlag);
      break;
    case _C_ARY_E:
      abort ();
      break;
    case _C_UNION_B:
      raiseEvent (NotImplemented, "Unions not supported [%s]", type);
      break;
    case _C_UNION_E:
      abort ();
      break;
    case _C_STRUCT_B:
      raiseEvent (NotImplemented, "Structures not supported [%s]", type);
      break;
    case _C_STRUCT_E:
      abort ();
      break;
    default:
      abort ();
      break;
    }
}

  
const char *
objc_type_for_lisp_type (const char *lispTypeString)
{
  if (strcmp (lispTypeString, TYPE_SHORT) == 0)
    return @encode (short);
  else if (strcmp (lispTypeString, TYPE_UNSIGNED_SHORT) == 0)
    return @encode (unsigned short);
  else if (strcmp (lispTypeString, TYPE_INT) == 0)
    return @encode (int);
  else if (strcmp (lispTypeString, TYPE_UNSIGNED) == 0)
    return @encode (unsigned);
  else if (strcmp (lispTypeString, TYPE_LONG) == 0)
    return @encode (long);
  else if (strcmp (lispTypeString, TYPE_UNSIGNED_LONG) == 0)
    return @encode (unsigned long);
  else if (strcmp (lispTypeString, TYPE_LONG_LONG) == 0)
    return @encode (long long);
  else if (strcmp (lispTypeString, TYPE_UNSIGNED_LONG_LONG) == 0)
    return @encode (unsigned long long);
  else if (strcmp (lispTypeString, TYPE_FLOAT) == 0)
    return @encode (float);
  else if (strcmp (lispTypeString, TYPE_DOUBLE) == 0)
    return @encode (double);
  else if (strcmp (lispTypeString, TYPE_LONG_DOUBLE) == 0)
    return @encode (long double);
  else if (strcmp (lispTypeString, TYPE_STRING) == 0)
    return @encode (const char *);
  else if (strcmp (lispTypeString, TYPE_OBJECT) == 0)
    return @encode (id);
  else
    abort ();
}

const char *
lisp_type_for_objc_type (const char *varType,
                         void (*func) (unsigned dim, unsigned count))
{
  const char *baseType;
  unsigned dimnum;

  void expand_type (const char *type)
    {
      switch (*type)
        {
        case _C_SHT:
          baseType = TYPE_SHORT;
          break;
        case _C_USHT:
          baseType = TYPE_UNSIGNED_SHORT;
          break;
        case _C_INT:
          baseType = TYPE_INT;
          break;
        case _C_UINT:
          baseType = TYPE_UNSIGNED;
          break;
        case _C_LNG:
          baseType = TYPE_LONG;
          break;
        case _C_ULNG:
          baseType = TYPE_UNSIGNED_LONG;
          break;
        case _C_LNG_LNG:
          baseType = TYPE_LONG_LONG;
          break;
        case _C_ULNG_LNG:
          baseType = TYPE_UNSIGNED_LONG_LONG;
          break;
        case _C_FLT:
          baseType = TYPE_FLOAT;
          break;
        case _C_DBL:
          baseType = TYPE_DOUBLE;
          break;
        case _C_LNG_DBL:
          baseType = TYPE_LONG_DOUBLE;
          break;
        case _C_CHARPTR:
          baseType = TYPE_STRING;
          break;
        case _C_ID:
          baseType = TYPE_OBJECT;
          break;
        case _C_ARY_B:
          type++;
          {
            char *tail;
            unsigned count = strtoul (type, &tail, 10);
            
            if (func)
              func (dimnum, count);
            dimnum++;
            expand_type (tail);
          }
          break;
        default:
          abort ();
        }
    }
  dimnum = 0;
  expand_type (varType);
  return baseType;
}

#if ((__GNUC__ == 2) && (__GNUC_MINOR__ >= 8)) || (__GNUC__ > 2)
id
nil_method (id receiver, SEL op, ...)
{
  raiseEvent (InvalidArgument,  "The message `%s' was sent to nil.\n",
              sel_get_name (op));
  return nil;
}
#endif

char *
zstrdup (id aZone, const char *str)
{
  size_t len = strlen (str); 
  char *ptr = [(aZone) alloc: len + 1];

  strcpy (ptr, str); 
  return ptr;
}
