/*  LibKiazma 0.2
 *  Copyright (C) 2006/2007 Roberto -MadBob- Guido <m4db0b@users.sourceforge.net>
 *
 *  This program is free software; you can redistribute it and/or modify
 *  it under the terms of the GNU General Public License as published by
 *  the Free Software Foundation; either version 2 of the License, or
 *  (at your option) any later version.
 *
 *  This program is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU Library General Public License for more details.
 *
 *  You should have received a copy of the GNU General Public License
 *  along with this program; if not, write to the Free Software
 *  Foundation, Inc., 51 Franklin St, Fifth Floor, Boston, MA  02110-1301  USA
 */

/**
	@file		kiazma_buttoned_box.c
	@brief		A GtkContainer with a button

	The KiazmaButtoned is a container with an attached button on the selection of which the
	application can performs operations on the sets of elements contained

	@todo		Made this a GtkBin instead a GtkHBox
*/

#include "libkiazma.h"
#include "kiazma_buttoned_box.h"

/**
	@internal

	List of signals handled by the widget
*/
enum {
	KIAZMA_BB_CLICKED,
	KIAZMA_BB_END_SIGNALS
};

/**
	@internal

	Array of identifiers for the signals registered into the widget class
	initializazion routine
*/
static guint KiazmaButtonedBoxSignals [ KIAZMA_BB_END_SIGNALS ];

/**
	@internal

	Init the class for KiazmaButtonedBox, and registers dedicated signals

	@param	klass	Instance of the class to init
*/
static void kiazma_buttoned_box_class_init ( KiazmaButtonedBoxClass *klass ) {
	GObjectClass *gobject_class;

	gobject_class = ( GObjectClass* ) klass;

	KiazmaButtonedBoxSignals [ KIAZMA_BB_CLICKED ] = g_signal_new ( g_intern_static_string ( "clicked" ),
						G_TYPE_FROM_CLASS ( gobject_class ), G_SIGNAL_RUN_FIRST, 0, NULL, NULL,
						g_cclosure_marshal_VOID__VOID, G_TYPE_NONE, 0 );
}

/**
	Init components for a KiazmaButtonedBox

	@param	ttt		The KiazmaButtoned to init
*/
static void kiazma_buttoned_box_init ( KiazmaButtonedBox *ttt ) {
	gtk_container_set_border_width ( GTK_CONTAINER ( ttt ), 0 );
	gtk_box_set_homogeneous ( GTK_BOX ( ttt ), FALSE );
}

/**
	Registers a new type for the KiazmaButtonedBox class

	@return			The registered type for the widget
*/
GType kiazma_buttoned_box_get_type () {
	static GType ttt_type = 0;

	if ( !ttt_type ) {
		static const GTypeInfo ttt_info = {
			sizeof ( KiazmaButtonedBoxClass ),
			NULL,
			NULL,
			( GClassInitFunc ) kiazma_buttoned_box_class_init,
			NULL,
			NULL,
			sizeof ( KiazmaButtonedBox ),
			0,
			( GInstanceInitFunc ) kiazma_buttoned_box_init,
		};

		ttt_type = g_type_register_static ( GTK_TYPE_HBOX, "KiazmaButtonedBox", &ttt_info, 0 );
	}

	return ttt_type;
}

/**
	@internal

	Callback assigned to the "clicked" signal of the GtkButton embedded
	into the KiazmaButtonedBox

	@param	button		Instance of the GtkButton clicked
	@param	box		KiazmaButtonedBox into which the button is
				included

	@return			FALSE
*/
static gboolean clicked ( GtkButton *button, KiazmaButtonedBox *box ) {
	g_signal_emit ( box, KiazmaButtonedBoxSignals [ KIAZMA_BB_CLICKED ], 0 );
	return FALSE;
}

/**
	Builds a new KiazmaButtonedBox

	@param	item		Item placed into the KiazmaButtonedBox
	@param	stock		Stock to use for the embedded button

	@return			A new KiazmaButtonedBox
*/
GtkWidget* kiazma_buttoned_box_new ( GtkWidget *item, const gchar *stock ) {
	GtkWidget *new_item;
	GtkWidget *button;

	new_item = GTK_WIDGET ( g_object_new ( kiazma_buttoned_box_get_type (), NULL ) );
	gtk_box_pack_start ( GTK_BOX ( new_item ), item, TRUE, TRUE, 0 );

	button = kiazma_button_stock_new ( stock );
	g_signal_connect ( G_OBJECT ( button ), "clicked", G_CALLBACK ( clicked ), new_item );
	gtk_box_pack_start ( GTK_BOX ( new_item ), button, FALSE, FALSE, 0 );

	return new_item;
}

/**
	Retrive the widget contained in the KiazmaButtonedBox

	@param	box		The KiazmaButtonedBox

	@return			Reference to the widget here contained
*/
GtkWidget* kiazma_buttoned_box_get_item ( KiazmaButtonedBox *box ) {
	GList *list;
	GtkWidget *item;

	list = gtk_container_get_children ( GTK_CONTAINER ( box ) );
	list = g_list_first ( list );
	item = list->data;
	g_list_free ( list );
	return item;
}
