/*  LibKiazma 0.2
 *  Copyright (C) 2006/2007 Roberto -MadBob- Guido <m4db0b@users.sourceforge.net>
 *
 *  This program is free software; you can redistribute it and/or modify
 *  it under the terms of the GNU General Public License as published by
 *  the Free Software Foundation; either version 2 of the License, or
 *  (at your option) any later version.
 *
 *  This program is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU Library General Public License for more details.
 *
 *  You should have received a copy of the GNU General Public License
 *  along with this program; if not, write to the Free Software
 *  Foundation, Inc., 51 Franklin St, Fifth Floor, Boston, MA  02110-1301  USA
 */

/**
	@file		kiazma_metadata_chooser.c
	@brief		Permits selection of a metadata and of his assigned value

	The KiazmaMetadataChooser include a KiazmaMetadataSelector, which permits selection of a
	metadata, and a KiazmaMetadataWrapper, which permits specification of a value. This last
	widget is automatically changed on the selection of a different metadata, so to offer the
	right way to input the value
*/

#include "libkiazma.h"
#include "kiazma_metadata_chooser.h"

/**
	@internal

	Return the widget which store the value component of the chooser

	@param	chooser		A KiazmaMetadataChooser

	@return			The widget which rappresent the value
*/
static inline GtkWidget* get_value_widget ( KiazmaMetadataChooser *chooser ) {
	return gtk_bin_get_child ( GTK_BIN ( chooser->value ) );
}

/**
	@internal

	Init the class for KiazmaMetadataChooser

	@param	klass	Instance of the class to init
*/
static void kiazma_metadata_chooser_class_init ( KiazmaMetadataChooserClass *klass ) {
	return;
}

/**
	@internal

	Sets the widget which rappresents the value inside the
	KiazmaMetadataChooser

	@param	ttt		KiazmaMetadataChooser to manage
	@param	meta		Metadata to rappresent into the chooser
*/
static void set_value_widget ( KiazmaMetadataChooser *ttt, UINT64 meta ) {
	GtkWidget *value;

	if ( meta == METADATA_INVALID )
		value = gtk_label_new ( _( "Choose a metadata" ) );
	else
		value = kiazma_metadata_wrapper_new ( meta );

	gtk_container_add ( GTK_CONTAINER ( ttt->value ), value );
	gtk_widget_show_all ( GTK_WIDGET ( ttt ) );
}

/**
	@internal

	Callback invoked when the user selects an item from the
	KiazmaMetadataSelector embedded into the KiazmaMetadataChooser

	@param	combo		The KiazmaMetadataSelector which value changed
	@param	ttt		Main KiazmaMetadataChooser
*/
static gboolean choosed ( GtkComboBox *combo, KiazmaMetadataChooser *ttt ) {
	UINT64 meta;

	meta = kiazma_metadata_selector_get ( KIAZMA_METADATA_SELECTOR ( ttt->meta ) );
	gtk_widget_destroy ( get_value_widget ( ttt ) );
	set_value_widget ( ttt, meta );
	return FALSE;
}

/**
	@internal

	Init all components into a KiazmaMetadataChooser

	@param	ttt		The KiazmaMetadataChooser to init
*/
static void kiazma_metadata_chooser_init ( KiazmaMetadataChooser *ttt ) {
	gtk_container_set_border_width ( GTK_CONTAINER ( ttt ), 2 );
	gtk_box_set_homogeneous ( GTK_BOX ( ttt ), FALSE );

	ttt->meta = kiazma_metadata_selector_new ();
	g_signal_connect ( G_OBJECT ( ttt->meta ), "changed", G_CALLBACK ( choosed ), ttt );

	ttt->value = gtk_frame_new ( NULL );
	gtk_container_set_border_width ( GTK_CONTAINER ( ttt->value ), 0 );
	gtk_frame_set_shadow_type ( GTK_FRAME ( ttt->value ), GTK_SHADOW_NONE );
	gtk_container_add ( GTK_CONTAINER ( ttt->value ), gtk_label_new ( _( "Choose a metadata" ) ) );

	gtk_box_pack_start ( GTK_BOX ( ttt ), ttt->meta, FALSE, FALSE, 0 );
	gtk_box_pack_start ( GTK_BOX ( ttt ), ttt->value, FALSE, FALSE, 0 );
}

/**
	Register a new type for KiazmaMetadataChooser

	@return			The type registered for this widget class
*/
GType kiazma_metadata_chooser_get_type () {
	static GType ttt_type = 0;

	if ( !ttt_type ) {
		static const GTypeInfo ttt_info = {
			sizeof ( KiazmaMetadataChooserClass ),
			NULL,
			NULL,
			( GClassInitFunc ) kiazma_metadata_chooser_class_init,
			NULL,
			NULL,
			sizeof ( KiazmaMetadataChooser ),
			0,
			( GInstanceInitFunc ) kiazma_metadata_chooser_init,
		};

		ttt_type = g_type_register_static ( GTK_TYPE_HBOX, "KiazmaMetadataChooser", &ttt_info, 0 );
	}

	return ttt_type;
}

/**
	Builds a new KiazmaMetadataChooser

	@return			A new KiazmaMetadataChooser allocated and
				inited
*/
GtkWidget* kiazma_metadata_chooser_new () {
	return GTK_WIDGET ( g_object_new ( kiazma_metadata_chooser_get_type (), NULL ) );
}

/**
	Adds a metadata to the list of values managed from the
	KiazmaMetadataChooser

	@param	chooser		The KiazmaMetadataChooser to enrich
	@param	meta		The metadata to add. See documentation
				about KiazmaMetadataSelector to know more on
				the behaviour
*/
void kiazma_metadata_chooser_add_meta ( KiazmaMetadataChooser *chooser, UINT64 meta ) {
	kiazma_metadata_selector_add_meta ( KIAZMA_METADATA_SELECTOR ( chooser->meta ), meta );
}

/**
	Shorthand to add all available metadatas in the list for the
	KiazmaMetadataChooser

	@param	chooser		KiazmaMetadataChooser to fill with all
				metadata
*/
void kiazma_metadata_chooser_fill_with_complete_range ( KiazmaMetadataChooser *chooser ) {
	UINT64 i;

	for ( i = 0; i < METADATA_END; i++ )
		kiazma_metadata_chooser_add_meta ( chooser, i );
}

/**
	Remove a value from the list of available metadata handled by the
	KiazmaMetadataChooser

	@param	chooser		The KiazmaMetadataChooser from which remove
				the metadata
	@param	meta		Metadata identifier to remove. See
				documentation about KiazmaMetadataSelector to
				know more on the behaviour
*/
void kiazma_metadata_chooser_remove_meta ( KiazmaMetadataChooser *chooser, UINT64 meta ) {
	kiazma_metadata_selector_remove_meta ( KIAZMA_METADATA_SELECTOR ( chooser->meta ), meta );
}

/**
	Sets the specified values to init the KiazmaMetadataChooser

	@param	chooser		The KiazmaMetadataChooser to init
	@param	meta		Assigned metadata. See documentation about
				KiazmaMetadataSelector to know more on the
				behaviour
	@param	value		Assigned value. See documentation about
				KiazmaMetadataWrapper to know more on the
				behaviour
*/
void kiazma_metadata_chooser_set ( KiazmaMetadataChooser *chooser, UINT64 meta, gchar *value ) {
	kiazma_metadata_selector_set ( KIAZMA_METADATA_SELECTOR ( chooser->meta ), meta );
	kiazma_metadata_wrapper_set_value ( KIAZMA_METADATA_WRAPPER ( get_value_widget ( chooser ) ), value );
}

/**
	Retrive values of user selected metadata and value

	@param	chooser		The KiazmaMetadataChooser from which retrive
				values
	@param	meta		Pointer filled with the selected metadata
	@param	value		Pointer filled with the choosed value. Have
				to be freed when no longer used

	@return			FALSE if no metadata have been selected, and
				passed pointers are not filled. TRUE otherwise
*/
gboolean kiazma_metadata_chooser_get ( KiazmaMetadataChooser *chooser, UINT64 *meta, gchar **value ) {
	UINT64 i_meta;

	i_meta = kiazma_metadata_selector_get ( KIAZMA_METADATA_SELECTOR ( chooser->meta ) );
	if ( i_meta == 0 )
		return FALSE;

	*value = kiazma_metadata_wrapper_get_value ( KIAZMA_METADATA_WRAPPER ( get_value_widget ( chooser ) ) );
	*meta = i_meta;
	return TRUE;
}

/**
	Resets selections into the KiazmaMetadataChooser, but leave untouched
	the list of available metadata

	@param	chooser		KiazmaMetadataChooser to reset
*/
void kiazma_metadata_chooser_reset ( KiazmaMetadataChooser *chooser ) {
	gtk_combo_box_set_active ( GTK_COMBO_BOX ( chooser->meta ), 0 );
	set_value_widget ( chooser, 0 );
}
