/*  LibKiazma 0.2
 *  Copyright (C) 2006/2007 Roberto -MadBob- Guido <m4db0b@users.sourceforge.net>
 *
 *  This program is free software; you can redistribute it and/or modify
 *  it under the terms of the GNU General Public License as published by
 *  the Free Software Foundation; either version 2 of the License, or
 *  (at your option) any later version.
 *
 *  This program is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU Library General Public License for more details.
 *
 *  You should have received a copy of the GNU General Public License
 *  along with this program; if not, write to the Free Software
 *  Foundation, Inc., 51 Franklin St, Fifth Floor, Boston, MA  02110-1301  USA
 */

/**
	@file	kiazma_shortcut_editor.c

	The KiazmaShortcutEditor permits to the user to define a keyboard
	combination just pressing the involved keys
*/

#include "libkiazma.h"
#include "kiazma_shortcut_editor.h"

#include <gdk/gdkkeysyms.h>

enum {
	SHORTCUT_CHANGED,
	SE_LAST_SIGNAL
};

/**
	Array of signals registrered from the KiazmaShortcutEditor
*/
guint		KiazmaShortcutEditorSignals [ SE_LAST_SIGNAL ];

/**
	@internal

	Init the GTK class rappresenting the KiazmaShortcutEditor

	@param	klass	Instance of the registering class
*/
static void kiazma_shortcut_editor_class_init ( KiazmaShortcutEditor *klass ) {
	KiazmaShortcutEditorSignals [ SHORTCUT_CHANGED ] = g_signal_new ( "shortcut-changed",
			G_TYPE_FROM_CLASS ( klass ), G_SIGNAL_RUN_FIRST | G_SIGNAL_ACTION,
			G_STRUCT_OFFSET ( KiazmaShortcutEditorClass, kiazma_shortcut_editor ), NULL, NULL,
			g_cclosure_marshal_VOID__VOID, G_TYPE_NONE, 0 );
}

/**
	Convert a key combination into a string human readable.\n
	This function is used for local purposes, but is available also
	externally the widget context so to provide a convenient way to
	rappresent a shortcut

	@param	mod	Code for the modifier of the combination
	@param	key	Code for the key of the combination

	@return		The textual rappresentation of the combination. The
			string have to be freed with g_free() when no longer
			used
*/
gchar* kiazma_shortcuteditor_convert_num_to_string ( GdkModifierType mod, guint key ) {
	gchar *ret;
	gchar *skey;

	if ( mod == 0 && key == 0 )
		return g_strdup ( "" );

	switch ( key ) {
		case GDK_Home:
			skey = g_strdup ( "Home" );
			break;

		case GDK_Escape:
			skey = g_strdup ( "Esc" );
			break;

		case GDK_Delete:
			skey = g_strdup ( "Canc" );
			break;

		case GDK_Left:
			skey = g_strdup ( "Left" );
			break;

		case GDK_Right:
			skey = g_strdup ( "Right" );
			break;

		case GDK_Up:
			skey = g_strdup ( "Up" );
			break;

		case GDK_Down:
			skey = g_strdup ( "Down" );
			break;

		case GDK_Page_Up:
			skey = g_strdup ( "PagUP" );
			break;

		case GDK_Page_Down:
			skey = g_strdup ( "PagDOWN" );
			break;

		default:
			if ( key > 128 )
				skey = g_strdup ( "" );
			else
				skey = g_strdup_printf ( "%c", key );
			break;
	}

	switch ( mod ) {
		case GDK_SHIFT_MASK:
			ret = g_strdup_printf ( "Shift + %s", skey );
			break;

		case GDK_CONTROL_MASK:
			ret = g_strdup_printf ( "Ctrl + %s", skey );
			break;

		case GDK_MOD1_MASK:
			ret = g_strdup_printf ( "Alt + %s", skey );
			break;

		default:
			ret = g_strdup ( skey );
			break;
	}

	g_free ( skey );
	return ret;
}

/**
	@internal

	Callback ionvoked when a new combinaton is immitted in the widget

	@param	editor	Reference to the input cell
	@param	ev	Datas for the button press event
	@param	data	Reference to the whole KiazmaShortcutEditor

	@return		Always FALSE
*/
gboolean kiazma_shortcuteditor_eval ( GtkWidget *editor, GdkEventKey *ev, gpointer data ) {
	KiazmaShortcutEditor *ttt;

	ttt = ( KiazmaShortcutEditor* ) data;
	if ( ttt->label )
		g_free ( ttt->label );

	ttt->label = kiazma_shortcuteditor_convert_num_to_string ( ev->state, ev->keyval );
	gtk_entry_set_text ( GTK_ENTRY ( ttt->text ), ttt->label );
	ttt->modifier = ev->state;
	ttt->key = ev->keyval;
	g_signal_emit ( G_OBJECT ( ttt ), KiazmaShortcutEditorSignals [ SHORTCUT_CHANGED ], 0 );
	return FALSE;
}

/**
	Callback invoked when pressed the "clear" button assigned to the
	KiazmaShortcutEditor

	@param	clear	Reference to the pressed button
	@param	data	Reference to the whole KiazmaShortcutEditor

	@return		Always FALSE
*/
gboolean kiazma_shortcuteditor_clear ( GtkButton *clear, gpointer data ) {
	KiazmaShortcutEditor *ttt;

	ttt = ( KiazmaShortcutEditor* ) data;
	if ( ttt->label )
		g_free ( ttt->label );

	ttt->label = g_strdup ( "" );
	gtk_entry_set_text ( GTK_ENTRY ( ttt->text ), ttt->label );
	ttt->modifier = 0;
	ttt->key = 0;
	return FALSE;
}

/**
	@internal

	Build and init a new instance of KiazmaShortcutEditor

	@param	ttt	Instance of KiazmaShortcutEditor to fill with the
			composing widget and init
*/
static void kiazma_shortcut_editor_init ( KiazmaShortcutEditor *ttt ) {
	GtkWidget *image;
	GtkWidget *clear;

	gtk_table_resize ( GTK_TABLE ( ttt ), 1, 2 );
	gtk_table_set_homogeneous ( GTK_TABLE ( ttt ), FALSE );

	image = gtk_image_new_from_stock ( GTK_STOCK_CLEAR, GTK_ICON_SIZE_BUTTON );
	clear = gtk_button_new ();
	gtk_container_add ( GTK_CONTAINER ( clear ), image );
	g_signal_connect ( G_OBJECT ( clear ), "button-press-event",
				G_CALLBACK ( kiazma_shortcuteditor_clear ), ( gpointer ) ttt );
	gtk_table_attach_defaults ( GTK_TABLE ( ttt ), clear, 0, 1, 0, 1 );

	ttt->text = gtk_entry_new ();
	gtk_entry_set_editable ( GTK_ENTRY ( ttt->text ), FALSE );
	g_signal_connect ( G_OBJECT ( ttt->text ), "key-press-event",
				G_CALLBACK ( kiazma_shortcuteditor_eval ), ( gpointer ) ttt );
	gtk_table_attach_defaults ( GTK_TABLE ( ttt ), ttt->text, 1, 2, 0, 1 );

	ttt->modifier = 0;
	ttt->key = 0;
	ttt->label = NULL;
}

/**
	Register the new GType for this kind of widget

	@return		The type for KiazmaShortcutEditor
*/
GType kiazma_shortcut_editor_get_type () {
	static GType ttt_type = 0;

	if ( !ttt_type ) {
		static const GTypeInfo ttt_info = {
			sizeof ( KiazmaShortcutEditorClass ),
			NULL,
			NULL,
			( GClassInitFunc ) kiazma_shortcut_editor_class_init,
			NULL,
			NULL,
			sizeof ( KiazmaShortcutEditor ),
			0,
			( GInstanceInitFunc ) kiazma_shortcut_editor_init,
		};

		ttt_type = g_type_register_static ( GTK_TYPE_TABLE, "KiazmaShortcutEditor", &ttt_info, 0 );
	}

	return ttt_type;
}

/**
	Create a new KiazmaShortcutEditor

	@return		A new KiazmaShortcutEditor allocated and inited
*/
GtkWidget* kiazma_shortcut_editor_new () {
	return GTK_WIDGET ( g_object_new ( kiazma_shortcut_editor_get_type (), NULL ) );
}

/**
	Used to get the value defined by the widget

	@param	synp	KiazmaShortcutEditor from which get the value
	@param	modifier	Will be setted to the modifier code defined
				into the widget
	@param	key	Will be setted to the key code defined into the widget
*/
void kiazma_shortcut_editor_get_value ( KiazmaShortcutEditor *synp, GdkModifierType *modifier, guint *key ) {
	*modifier = synp->modifier;
	*key = synp->key;
}

/**
	Used to set the value of the widget

	@param	synp	KiazmaShortcutEditor in which set the value
	@param	modifier	Modifier for the new keys combination
	@param	key	Key of the new keys combination
*/
void kiazma_shortcut_editor_set_value ( KiazmaShortcutEditor *synp, GdkModifierType modifier, guint key ) {
	synp->modifier = modifier;
	synp->key = key;

	if ( synp->label )
		g_free ( synp->label );

	synp->label = kiazma_shortcuteditor_convert_num_to_string ( modifier, key );
	gtk_entry_set_text ( GTK_ENTRY ( synp->text ), synp->label );
}
