/*  Synapse 0.1
 *  Copyright (C) 2006 Roberto -MadBob- Guido <m4db0b@users.sourceforge.net>
 *
 *  This program is free software; you can redistribute it and/or modify
 *  it under the terms of the GNU General Public License as published by
 *  the Free Software Foundation; either version 2 of the License, or
 *  (at your option) any later version.
 *
 *  This program is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU Library General Public License for more details.
 *
 *  You should have received a copy of the GNU General Public License
 *  along with this program; if not, write to the Free Software
 *  Foundation, Inc., 51 Franklin St, Fifth Floor, Boston, MA  02110-1301  USA
 */

#include "synapse.h"

/**
	Callback assegnata al pulsante per la cancellazione della query
	scritta nell'apposita barra

	@param	button	Il pulsante dedicato alla pulizia della barra per le
			query
	@param	data	Non usato

	@return		FALSE
*/
gboolean clear_query ( GtkButton *button, gpointer data ) {
	gtk_tree_store_clear ( Graph.running->icons );
	gtk_entry_set_text ( GTK_ENTRY ( gtk_bin_get_child ( GTK_BIN ( GTK_COMBO_BOX_ENTRY ( Graph.query_text ) ) ) ), "" );
	return FALSE;
}

/**
	Usata per inizializzare i valori su cui si basera' il calcolo del
	tempo di esecuzione della query sul filesystem

	@param	t	Puntatore che verra' settato con la data corrente, in
			secondi
	@param	tim	Puntatore che verra' settato con l'arrotondamento in
			microsecondi della data corrente
*/
void get_interval_init ( time_t *t, suseconds_t *tim ) {
	struct timeval t1;
	GdkCursor *cursor;

	gettimeofday ( &t1, NULL );
	*t = t1.tv_sec;
	*tim = t1.tv_usec;

	cursor = gdk_cursor_new ( GDK_WATCH );
	gdk_window_set_cursor ( Graph.main_win->window, cursor );
	gdk_cursor_unref ( cursor );
}

/**
	Da invocare al termine dell'esecuzione di una query, dati in ingresso
	i valori prelevati all'inizio con get_interval_init() calcola la
	differenza tra i due tempi in microsecondi

	@param	init_sec	Numero di secondi all'inizio dell'esecuzione
				della query
	@param	init_usec	Numero di microsecondi all'inizio
				dell'esecuzione della query
	@param	total	Puntatore che verra' settato alla differenza, in
			microsecondi, tra il tempo rappresentato dagli altri
			due parametri della funzione ed il tempo corrente
*/
void get_interval_final ( time_t init_sec, suseconds_t init_usec, suseconds_t *total ) {
	time_t t;
	suseconds_t now;

	get_interval_init ( &t, &now );
	*total = ( ( t - init_sec ) * 1000000 ) + ( now - init_usec );
}

/**
	Finalizza l'esecuzione di una query, aggiornando la barra di stato
	dell'applicazione e aggiungendo la query appena eseguita nella history

	@param	query	La query appena eseguita
	@param	n	Numero di elementi raccolti con la query
	@param	tims	Numero di secondi al momento dell'inizio
			dell'esecuzione della query, cosi' come vengono
			ottenuti con get_interval_init()
	@param	tim	Numero di microsecondi al momento dell'inizio
			dell'esecuzione della query, cosi' come vengono
			ottenuti con get_interval_init()
*/
inline void finalize_query ( gchar *query, int n, time_t tims, suseconds_t tim ) {
	suseconds_t final;
	gchar status [ DEFAULT_STRING_LENGHT ];

	get_interval_final ( tims, tim, &final );
	save_in_history ( query );
	( void ) snprintf ( status, DEFAULT_STRING_LENGHT, _( "%d items - Query performed in %ld microseconds" ), n, final );
	update_status_progress ( 0, 0 );
	update_status_bar ( status );

	gdk_window_set_cursor ( Graph.main_win->window, NULL );
}

/**
	Add an item to an icon view

	@param	file		Rappresentation of the file to add in the view
	@param	to_extract	0-terminated array of metadata selected by
				the user, to include in the rappresentation
				of the file
	@param	extract_n	Number of item in the "to_extract" array
	@param	parent		Reference to the parent item under the new
				file have to go. To rappresent a toplevel
				item, use NULL here
	@param	icons		Reference to the GtkTreeStore embedded into
				the SynapseIconStackedView widget
	@param	iter		Pointer here assign to the new GtkTreeIter
				created and inserted into "icons". If NULL,
				an internal structure is used
*/
void add_file_to_view ( HyppoVFSFile *file, UINT64 *to_extract, int extract_n,
			GtkTreeIter *parent, GtkTreeStore *icons, GtkTreeIter *iter ) {

	UINT64 id;
	gchar filename [ MAX_VALUE_SIZE + 1 ];
	GdkPixbuf *icon;
	GList *metas;
	GtkTreeIter my_iter;
	GtkWidget *ic;

	if ( !iter )
		iter = &my_iter;

	id = file->id;
	( void ) strncpy ( filename, file->name, MAX_VALUE_SIZE + 1 );
	metas = hyppo_vfs_listxattr_limited ( id, to_extract, extract_n );
	ic = kiazma_icon_new_list ( id, metas );
	icon = kiazma_icon_prerender ( KIAZMA_ICON ( ic ) );

	gtk_tree_store_append ( icons, iter, parent );
	gtk_tree_store_set ( icons, iter,
				ICON_VIEW_NAME_COL, filename,
				ICON_VIEW_ICON_COL, icon,
				ICON_VIEW_ID_COL, id, -1 );

	if ( icon )
		g_object_unref ( icon );
}

/**
	Explode a directory in the icon view, to create an icon pile

	@param	directory	Directory item to unfold
	@param	selected	0-terminated array of metadata selected by
				the user in the original query
	@param	icons		Model in which add the items inside the
				directory
*/
void unfold_dir_in_view ( HyppoVFSFile *directory, UINT64 *selected, int n_sel, GtkTreeStore *icons ) {
	GList *file;
	GtkTreeIter iter;

	if ( directory->children == NULL )
		return;

	file = g_list_first ( directory->children );
	add_file_to_view ( ( HyppoVFSFile* ) file->data, selected, n_sel, NULL, Graph.running->icons, &iter );
	file = g_list_next ( file );

	while ( file ) {
		add_file_to_view ( ( HyppoVFSFile* ) file->data, selected, n_sel, &iter, Graph.running->icons, NULL );
		file = g_list_next ( file );
	}
}

/**
	Esegue la query passata ed aggiorna la visualizzazione ad icone
	dell'applicazione

	@param	query	La query da eseguire, gia' trattata con
			hyppo_traslate_query()
*/
void run_query ( gchar *query ) {
	int n;
	int tot;
	int sels;
	time_t seconds;
	suseconds_t interval;
	UINT64 *selected;
	GList *files;
	GList *iter;
	HyppoVFSFile *file;

	get_interval_init ( &seconds, &interval );
	n = 0;

	files = hyppo_vfs_query ( query );
	sels = hyppo_query_get_selection ( query, &selected );

	if ( files == NULL ) {
		if ( errno != 0 )
			xfce_err ( _( "Unable to execute query\n%s\n%s (%d)" ), query, strerror ( errno ), errno );
	}

	else {
		files = iter = g_list_first ( files );
		tot = g_list_length ( iter );

		while ( iter ) {
			update_status_progress ( tot, n );
			file = ( HyppoVFSFile* ) iter->data;

			/**
				@warning	Due not so clear motivations, I can't access to the
						DTTOIF macro to translate a d_type in stat::st_mode
						format, so I have to manually shift bits to coincide
			*/
			if ( S_ISDIR ( file->type << 12 ) )
				unfold_dir_in_view ( file, selected, sels, Graph.running->icons );
			else
				add_file_to_view ( file, selected, sels, NULL, Graph.running->icons, NULL );

			errno = 0;
			n++;
			iter = g_list_next ( iter );
		}

		hyppo_vfs_free_query ( files );
	}

	finalize_query ( query, n, seconds, interval );
}

/**
	Callback assegnata al pulsante di esecuzione delle query scritte
	nell'apposita barra
*/
void start_query () {
	int n;
	int len;
	const gchar *query;
	gchar *true_query			= NULL;

	gtk_tree_store_clear ( Graph.running->icons );
	query = gtk_combo_box_get_active_text ( GTK_COMBO_BOX ( Graph.query_text ) );
	len = strlen ( query );

	/**
		@todo	Forse non e' una bella idea quella di mostrare *tutti*
			i files sul filesystem quando la query e' vuota...
			Correggere questo quando l'interfaccia SQL al
			filesystem sara' stata un minimo debuggata...
	*/
	if ( !len )
		true_query = g_strdup ( "" );

	else {
		switch ( hyppo_traslate_query ( query, &true_query, &n ) ) {
			case HYPPO_TRASLATION_OK:
				break;

			default:
				/**
					@todo	Correct visualization indicating point where
						query have been found invalid
				*/
				xfce_warn ( _( "Invalid query submitted" ) );
				return;
		}
	}

	run_query ( true_query );
	g_free ( true_query );
}

/**
	Callback assegnata ai pulsanti che, nel menu contestuale legato ai
	singoli files, permettono di rintracciare gli altri files che hanno
	un metadato in comune con quello selezionato. Questa funzione genera
	la query di ricerca e la esegue

	@param	item	Il pulsante selezionato nel menu contestuale del file
	@param	data	Riferimento al file selezionato da cui e' partita
			l'invocazione, espresso qui in forma di GtkTreePath

	@return		FALSE
*/
gboolean do_collective_query ( GtkMenuItem *item, gpointer data ) {
	UINT64 fileid;
	UINT64 metaid;
	gchar *query;
	gchar *original_meta_name;
	gchar value [ MAX_VALUE_SIZE ];
	gchar *true_value;
	GtkTreePath *path;

	original_meta_name = ( gchar* ) gtk_label_get_text ( GTK_LABEL ( GTK_BIN ( item )->child ) );
	metaid = reverse_translation ( original_meta_name );
	if ( !metaid ) {
		xfce_err ( _( "Unable to get metadata identifier\n%s" ), original_meta_name );
		return FALSE;
	}

	path = ( GtkTreePath* ) data;
	fileid = 0;
	fileid_by_path ( path, &fileid );

	memset ( value, 0, sizeof ( gchar ) * MAX_VALUE_SIZE );
	if ( hyppo_vfs_getxattr ( fileid, metaid, value, MAX_VALUE_SIZE ) < 0 ) {
		xfce_err ( _( "Unable to get metadata %llu for %llu" ), AllMetadata [ metaid ].id, fileid );
		return FALSE;
	}

	true_value = NULL;

	if ( AllMetadata [ metaid ].type & META_TYPE_STRING )
		true_value = g_strdup_printf ( "\'%s\'", value );

	if ( !true_value )
		true_value = g_strdup ( value );

	query = g_strdup_printf ( "SELECT %0*llu WHERE %0*llu = %s", HYPPO_MAX_REAL_NAMELEN,
	                          AllMetadata [ META_NAME ].id, HYPPO_MAX_REAL_NAMELEN, metaid, true_value );

	gtk_tree_store_clear ( Graph.running->icons );
	gtk_entry_set_text ( GTK_ENTRY ( gtk_bin_get_child ( GTK_BIN ( GTK_COMBO_BOX_ENTRY ( Graph.query_text ) ) ) ), query );
	run_query ( query );
	g_free ( query );
	g_free ( true_value );
	return FALSE;
}
