/**
 * @file add-shell.c
 * Add a shell to the list of valid shells
 *
 * Copyright (C) 2004 Free Software Foundation, Inc.
 *
 *  This file is part of GNU Sysutils
 *
 *  GNU Sysutils is free software; you can redistribute it and/or modify
 *  it under the terms of the GNU General Public License as published by
 *  the Free Software Foundation; either version 2 of the License, or
 *  (at your option) any later version.
 *
 *  GNU Sysutils is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU General Public License for more details.
 *
 *  You should have received a copy of the GNU General Public License
 *  along with this program; if not, write to the Free Software Foundation,
 *  Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
 */
#include <argp.h>
#include <errno.h>
#include <stdio.h>
#include <string.h>
#include <stdlib.h>
#include <unistd.h>

#include "misc.h"
#include "sysutils.h"

#define PRG_NAME "add-shell"	/**< Name shown by --help etc */

extern const char *progname;	/**< Used to store the name of the program */

/** Address to send bug-reports to */
const char *argp_program_bug_address = PACKAGE_BUGREPORT;

/** Usage information */
static char args_doc[] =
	N_("SHELL");

/** Program synopsis */
static char doc[] =
	N_("Add a shell to the list of valid shells.");

/** Structure with the available command line options */
static struct argp_option options[] = {
	{ "force", 'f', 0, 0,
	  N_("Allow non-existing shells to be added"), -2 },
	{ "verbose", 'v', 0, 0,
	  N_("Warn if the shell is already listed as a valid shell"), -2 },
	{ 0, 0, 0, 0, 0, 0 }
};

/** Structure to hold output from argument parser */
struct arguments {
	const char *shell;		/**< Shell to add */
	int force;			/**< Add non-existing shells */
	int verbose;			/**< Warn if shell is already listed */
};

/**
 * Parse a single option
 *
 * @param key The option
 * @param arg The argument for the option
 * @param state The state of argp
 * @return 0 on success,
 *         ARGP_ERR_UNKNOWN on failure
 */
static error_t parse_opt(int key, char *arg, struct argp_state *state)
{
	struct arguments *args = state->input;
	error_t status = 0;

	switch (key) {
	case 'f':
		args->force = 1;
		break;

	case 'v':
		args->verbose = 1;
		break;

	case ARGP_KEY_INIT:
		args->shell = NULL;
		args->force = 0;
		args->verbose = 0;
		break;

	case ARGP_KEY_ARG:
		if (args->shell)
			argp_usage(state);

		args->shell = arg;
		break;

	case ARGP_KEY_NO_ARGS:
		argp_usage(state);

	default:
		status = ARGP_ERR_UNKNOWN;
		break;
	}

	return status;
}

/**
 * The program's main-function
 *
 * @param argc The number of arguments
 * @param argv The arguments
 * @return 0 on success, errno on failure
 */
int main(int argc, char *argv[])
{
	FILE *afp = NULL;

	error_t status = 0;

	/* argp parser */
	struct argp argp = {
		.options	= options,
		.parser		= parse_opt,
		.args_doc	= args_doc,
		.doc		= doc,
	};

	struct arguments args;

	argp_program_version_hook = version;
	argp_err_exit_status = EINVAL;

	errno = 0;

	/* Initialise support for locales, and set the program-name */
	if ((status = init_locales(PRG_NAME)))
		goto EXIT;

	set_author_information(_("Written by David Weinehall.\n"));

	/* Parse command line */
	if ((status = argp_parse(&argp, argc, argv, 0, 0, &args))) {
		if (status != EINVAL)
			fprintf(stderr,
				_("%s: `%s' failed; %s\n"),
				progname, "argp_parse()", strerror(status));

		goto EXIT;
	}

	/* Make sure the caller has root privileges */
	if ((status = is_root())) {
		if (status == EPERM) {
			fprintf(stderr,
				_("%s: insufficient privileges\n"
				  "You must be root to %s.\n"),
				progname,
				_("add a shell to the list of valid shells"));
		} else {
			fprintf(stderr,
				_("%s: `%s' failed; %s\n"),
				progname, "is_root", strerror(errno));
		}

		goto EXIT;
	}

	if (!args.force && (status = is_valid_shell(args.shell))) {
		if (errno)
			fprintf(stderr,
				_("%s: invalid shell specified\n"),
				progname);

		goto EXIT;
	}

	if (!is_listed_shell(args.shell)) {
		if (args.verbose) {
			fprintf(stderr,
				_("%s: warning: `%s' already exists "
				  "in the list of valid shells\n"),
				progname, args.shell);
		}

		goto EXIT;
	}

	/* Open /etc/shells for append */
	if (!(afp = open_file(SHELLS_FILE, "a+"))) {
		status = errno;
		goto EXIT;
	}

	if ((status = fprintf(afp, "%s\n", args.shell)) < 0) {
		fprintf(stderr,
			_("%s: `%s' failed; %s\n"),
			progname, "fprintf()", strerror(errno));
		goto EXIT;
	}

	/* Close infile */
	if ((status = close_file(&afp)))
		goto EXIT;

EXIT:
	return status;
}
