/**
 * @file chage.c
 * Change password expiry information the specified users
 *
 * Copyright (C) 2002, 2003, 2004 David Weinehall
 * Copyright (C) 2004, 2006 Free Software Foundation, Inc.
 *
 *  This file is part of GNU Sysutils
 *
 *  GNU Sysutils is free software; you can redistribute it and/or modify
 *  it under the terms of the GNU General Public License as published by
 *  the Free Software Foundation; either version 2 of the License, or
 *  (at your option) any later version.
 *
 *  GNU Sysutils is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU General Public License for more details.
 *
 *  You should have received a copy of the GNU General Public License
 *  along with this program; if not, write to the Free Software Foundation,
 *  Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
 */
#include <argp.h>
#include <errno.h>
#include <stdio.h>
#include <shadow.h>
#include <stdlib.h>
#include <string.h>
#include <sys/stat.h>
#include <sys/types.h>

#include "misc.h"
#include "sysutils.h"

#define PRG_NAME "chage"	/**< Name shown by --help etc */

extern const char *progname;	/**< Used to store the name of the program */

/** Address to send bug-reports to */
const char *argp_program_bug_address = PACKAGE_BUGREPORT;

/** Usage information */
static char args_doc[] =
	N_("USERS");

/** Program synopsis */
static char doc[] =
	N_("Change password aging information for the specified users.\n"
	   "\n"
	   "USERS should be a comma-separated list of users, "
	   "or the word ALL to change expiry-information for all users.");

/** Structure with the available command line options */
static struct argp_option options[] = {
	{ "expire", 'E', N_("DATE"), OPTION_ARG_OPTIONAL,
	  N_("Date of expiration for the account (YYYY-MM-DD); "
	     "empty == never"), 0 },
	{ "inact", 'I', N_("DAYS"), 0,
	  N_("Max number of days of inactivity after password expiry "
	     "before the account is locked"), 0 },
	{ "lstchg", 'd', N_("DATE"), OPTION_ARG_OPTIONAL,
	  N_("Date when the password was last changed (YYYY-MM-DD); "
	     "empty == now"), 0 },
	{ "max", 'M', N_("DAYS"), 0,
	  N_("Maximum number of days before the password must be changed"), 0 },
	{ "min", 'm', N_("DAYS"), 0,
	  N_("Minimum number of days before the password can be changed"), 0 },
	{ "warn", 'W', N_("DAYS"), 0,
	  N_("Number of days before the password expires to warn"), 0 },
	{ "force", 'f', 0, 0,
	  N_("Force changes"), -2 },
	{ "verbose", 'v', 0, 0,
	  N_("Warn if the specified users does not exist"), -2 },
	{ 0, 0, 0, 0, 0, 0 }
};

/** Structure to hold output from argument parser */
struct arguments {
	const char *usrlist;	/**< Comma-separated list of users to modify */
	long lstchg;		/**< Latest change; seconds since epoch */
	long expire;		/**< Account expiry; seconds since epoch */
	long inact;		/**< Days after password expiry to inactivate */
	long min;		/**< Min days before password can be changed */
	long max;		/**< Max days before password must be changed */
	long warn;		/**< Days before password expiry to warn */
	long force;		/**< Allow changing ALL users */
	int verbose;		/**< Warn about non-existing users */
};

/**
 * Used to keep track of whether we're batch-editing
 * or doing interactive changes
 * 0 - interactive
 * 1 - batch
 */
static int batch = 0;

/**
 * Parse a single option
 *
 * @param key The option
 * @param arg The argument for the option
 * @param state The state of argp
 * @return 0 on success,
 *         ARGP_ERR_UNKNOWN on failure
 */
static error_t parse_opt(int key, char *arg, struct argp_state *state)
{
	struct arguments *args = state->input;
	error_t status = 0;

	switch (key) {
	case 'd':
		if ((status = is_valid_date(arg)))
			argp_error(state,
				   _("invalid value supplied to `-d'"));

		if ((status = string_to_date(arg,
					     (void **)&args->lstchg))) {
			fprintf(stderr,
				_("%s: `%s' failed; %s\n"),
				progname, "string_to_date", strerror(errno));
			goto EXIT;
		} else if (args->lstchg == -2) {
			if ((args->lstchg = get_current_date()) == -1) {
				status = errno;
				goto EXIT;
			}
		}

		if (!args->lstchg)
			args->lstchg = -1;

		batch = 1;
		break;

	case 'E':
		if ((status = is_valid_date(arg)))
			argp_error(state,
				   _("invalid value supplied to `-E'"));

		if ((status = string_to_date(arg,
					     (void **)&args->expire))) {
			fprintf(stderr,
				_("%s: `%s' failed; %s\n"),
				progname, "string_to_date", strerror(errno));
			goto EXIT;
		} else if (args->expire == -2) {
			args->expire = 0;
		}

		if (!args->expire)
			args->expire = -1;

		batch = 1;
		break;

	case 'I':
		if ((status = is_long(arg)))
			argp_error(state,
				   _("invalid value supplied to `-I'"));

		if ((status = string_to_long(arg,
					     (void **)&args->inact))) {
			fprintf(stderr,
				_("%s: `%s' failed; %s\n"),
				progname, "string_to_long", strerror(errno));
			goto EXIT;
		}

		batch = 1;
		break;

	case 'm':
		if ((status = is_long(arg)))
			argp_error(state,
				   _("invalid value supplied to `-m'"));

		if ((status = string_to_long(arg,
					     (void **)&args->min))) {
			fprintf(stderr,
				_("%s: `%s' failed; %s\n"),
				progname, "string_to_long", strerror(errno));
			goto EXIT;
		}

		batch = 1;
		break;

	case 'M':
		if ((status = is_long(arg)))
			argp_error(state,
				   _("invalid value supplied to `-M'"));

		if ((status = string_to_long(arg,
					     (void **)&args->max))) {
			fprintf(stderr,
				_("%s: `%s' failed; %s\n"),
				progname, "string_to_long", strerror(errno));
			goto EXIT;
		}

		batch = 1;
		break;

	case 'W':
		if ((status = is_long(arg)))
			argp_error(state,
				   _("invalid value supplied to `-W'"));

		if ((status = string_to_long(arg,
					     (void **)&args->warn))) {
			fprintf(stderr,
				_("%s: `%s' failed; %s\n"),
				progname, "string_to_long", strerror(errno));
			goto EXIT;
		}

		batch = 1;
		break;

	case 'f':
		args->force = 1;
		break;

	case 'v':
		args->verbose = 1;
		break;

	case ARGP_KEY_INIT:
		args->usrlist = NULL;
		args->lstchg = -2;
		args->expire = -2;
		args->inact = -2;
		args->min = -2;
		args->max = -2;
		args->warn = -2;
		args->force = -2;
		args->verbose = 0;
		break;

	case ARGP_KEY_ARG:
		if (args->usrlist)
			argp_usage(state);

		args->usrlist = arg;
		break;

	case ARGP_KEY_NO_ARGS:
		argp_usage(state);
		break;

	default:
		status = ARGP_ERR_UNKNOWN;
		break;
	}

EXIT:

	return status;
}

/**
 * The program's main-function
 *
 * @param argc The number of arguments
 * @param argv The arguments
 * @return 0 on success, errno on failure
 */
int main(int argc, char *argv[])
{
	FILE *sprfp = NULL;
	FILE *spwfp = NULL;
	struct spwd *sp;

	int empty = 1;
	int changed = 0;

	error_t status = 0;

	char *spwname = NULL;
	char *spbname = NULL;

	char *lstchg = NULL;
	char *min = NULL;
	char *max = NULL;
	char *warn = NULL;
	char *inact = NULL;
	char *expire = NULL;

	char **usrarray = NULL;

	mode_t oldmask;

	/* argp parser */
	struct argp argp = {
		.options	= options,
		.parser		= parse_opt,
		.args_doc	= args_doc,
		.doc		= doc,
	};

	struct arguments args;

	argp_program_version_hook = version;
	argp_err_exit_status = EINVAL;

	errno = 0;

	/* Initialise support for locales, and set the program-name */
	if ((status = init_locales(PRG_NAME)))
		goto EXIT;

	set_author_information(_("Written by David Weinehall.\n"));

	/* Parse command line */
	if ((status = argp_parse(&argp, argc, argv, 0, 0, &args))) {
		if (status != EINVAL)
			fprintf(stderr,
				_("%s: `%s' failed; %s\n"),
				progname, "argp_parse()", strerror(status));

		goto EXIT;
	}

	/* Make sure the caller has root privileges */
	if ((status = is_useradmin())) {
		if (status == EPERM) {
			fprintf(stderr,
				_("%s: insufficient privileges\n"
				  "You must be a user-administrator to "
				  "%s.\n"),
				progname,
				_("change password aging information"));
		} else {
			fprintf(stderr,
				_("%s: `%s' failed; %s\n"),
				progname, "is_useradmin", strerror(errno));
		}

		goto EXIT;
	}

	/* There are two alternatives here, neither of which are really
	 * pretty; either to read the entire passwd file once to get
	 * all usernames, then use them for the ALL list, or to
	 * have separate code for the ALL case and the case of separate
	 * user-entries.  Since the latter is probably the most common,
	 * the latter has been chosen.
	 */
	if (!strcmp(args.usrlist, "ALL")) {
		char *tmp = NULL;

		if (!args.force) {
			fprintf(stderr,
				_("%s: specifying ALL requires `--force' "
				  "to be specified\n"),
				progname);
			status = EINVAL;
			goto EXIT;
		}

		if (!(tmp = get_all_users())) {
			status = errno;
		} else if (!strlen(tmp)) {
			fprintf(stderr,
				_("%s: could not find any %s; the %s "
				  "might be corrupt\n"),
				progname, _("users"), _("user database"));
			status = ENOENT;
		} else if (!(usrarray = strsplit(tmp, ",", 0))) {
			fprintf(stderr,
				_("%s: `%s' failed; %s\n"),
				progname, "strsplit", strerror(errno));
			status = errno;
		}

		free(tmp);

		if (status)
			goto EXIT;
	} else {
		char *tmp = NULL;
		uid_t i = 0; /* We're scanning <= LASTUID, hence uid_t */

		if ((status = is_valid_namelist(args.usrlist))) {
			if (status == EINVAL) {
				fprintf(stderr,
					_("%s: the specified list of %s "
					  "contains invalid characters\n"),
					progname, _("users"));
			} else {
				fprintf(stderr,
					_("%s: `%s' failed; %s\n"),
					progname, "is_valid_namelist",
					strerror(errno));
			}

			goto EXIT;
		}

		if (!(usrarray = strsplit(argv[argc - 1], ",", 0))) {
			fprintf(stderr,
				_("%s: `%s' failed; %s\n"),
				progname, "strsplit", strerror(errno));
			status = errno;
			goto EXIT;
		}

		/* If verbose mode has been requested,
		 * warn about all non-existing users
		 */
		while (args.verbose && (tmp = usrarray[i++])) {
			if (!getpwnam(tmp)) {
				if (errno) {
					fprintf(stderr,
						_("%s: `%s' failed; %s\n"),
						progname, "getpwnam()",
						strerror(errno));
					status = errno;
					goto EXIT;
				} else {
					fprintf(stderr,
						_("%s: warning: %s `%s' "
						  "does not exist\n"),
						progname, _("user"),
						tmp);
				}
			}
		}
	}

	if (!batch) {
		/* If we get multiple usernames, we abort */
		if (usrarray[1]) {
			fprintf(stderr,
				_("%s: multiple users can only be processed "
				  "in batch-mode\n"),
				progname);
			status = EINVAL;
			goto EXIT;
		}

		/* Get the current expiry-information */
		if (!(sp = getspnam(usrarray[0])) && errno) {
			fprintf(stderr,
				_("%s: `%s' failed; %s\n"),
				progname, "getspnam()", strerror(errno));
			status = errno;
			goto EXIT;
		}

		if (!sp) {
			fprintf(stderr,
				_("%s: `%s' does not exist in the "
				  "user shadow database\n"),
				progname, usrarray[0]);
			status = ENOENT;
			goto EXIT;
		}

		args.lstchg = sp->sp_lstchg;
		args.min = sp->sp_min;
		args.max = sp->sp_max;
		args.warn = sp->sp_warn;
		args.inact = sp->sp_inact;
		args.expire = sp->sp_expire;

		/* Note: from this point on we know that username is valid,
		 * since it existed in the user database, hence we can
		 * print it without fear
		 */
		fprintf(stdout,
			_("Changing aging information for `%s'\n"
			  "Enter the new value, or press enter to keep "
			  "the old value\n"),
			usrarray[0]);

		/* Note: From now on we assume that the old information
		 * is untainted; if not some other program is to blame
		 */
		while (1) {
			fprintf(stdout,
				_("\tMinimum Password Age [%ld]: "),
				args.min);
			(void)fflush(stdout);

			if (!(min = input_string(1))) {
				if (errno) {
					fprintf(stderr,
						_("%s: `%s' failed; %s\n"),
						progname, "input_string",
						strerror(errno));
					status = errno;
					goto EXIT;
				} else {
					break;
				}
			}

			if (is_long(min)) {
				fprintf(stderr, _("\nInvalid input\n\n"));
				free(min);
				continue;
			}

			if ((status = string_to_long(min,
						     (void **)&args.min))) {
				fprintf(stderr,
					_("%s: `%s' failed; %s\n"),
					progname, "string_to_long",
					strerror(errno));
				free(min);
				goto EXIT;
			}

			free(min);
			break;
		}

		while (1) {
			fprintf(stdout,
				_("\tMaximum Password Age [%ld]: "),
				args.max);
			(void)fflush(stdout);

			if (!(max = input_string(1))) {
				if (errno) {
					fprintf(stderr,
						_("%s: `%s' failed; %s\n"),
						progname, "input_string",
						strerror(errno));
					status = errno;
					goto EXIT;
				} else {
					break;
				}
			}

			if (is_long(max)) {
				fprintf(stderr, _("\nInvalid input\n\n"));
				free(max);
				continue;
			}

			if ((status = string_to_long(max,
						     (void **)&args.max))) {
				fprintf(stderr,
					_("%s: `%s' failed; %s\n"),
					progname, "string_to_long",
					strerror(errno));
				free(max);
				goto EXIT;
			}

			free(max);
			break;
		}

		while (1) {
			fprintf(stdout,
				_("\tLast Password Change (YYYY-MM-DD) [%s]: "),
				date_to_string(args.lstchg));
			(void)fflush(stdout);

			if (!(lstchg = input_string(1))) {
				if (errno) {
					fprintf(stderr,
						_("%s: `%s' failed; %s\n"),
						progname, "input_string",
						strerror(errno));
					status = errno;
					goto EXIT;
				} else {
					break;
				}
			}

			if (is_valid_date(lstchg)) {
				fprintf(stderr, _("\nInvalid input\n\n"));
				free(lstchg);
				continue;
			}

			if ((status = string_to_date(lstchg,
						     (void **)&args.lstchg))) {
				fprintf(stderr,
					_("%s: `%s' failed; %s\n"),
					progname, "string_to_date",
					strerror(errno));
				free(lstchg);
				goto EXIT;
			}

			free(lstchg);
			break;
		}

		while (1) {
			fprintf(stdout,
				_("\tPassword Expiration Warning [%ld]: "),
				args.warn);
			(void)fflush(stdout);

			if (!(warn = input_string(1))) {
				if (errno) {
					fprintf(stderr,
						_("%s: `%s' failed; %s\n"),
						progname, "input_string",
						strerror(errno));
					status = errno;
					goto EXIT;
				} else {
					break;
				}
			}

			if (is_long(warn)) {
				fprintf(stderr, _("\nInvalid input\n\n"));
				free(warn);
				continue;
			}

			if ((status = string_to_long(warn,
						     (void **)&args.warn))) {
				fprintf(stderr,
					_("%s: `%s' failed; %s\n"),
					progname, "string_to_long",
					strerror(errno));
				free(warn);
				goto EXIT;
			}

			free(warn);
			break;
		}

		while (1) {
			fprintf(stdout,
				_("\tPassword Inactive [%ld]: "),
				args.inact);
			(void)fflush(stdout);

			if (!(inact = input_string(1))) {
				if (errno) {
					fprintf(stderr,
						_("%s: `%s' failed; %s\n"),
						progname, "input_string",
						strerror(errno));
					status = errno;
					goto EXIT;
				} else {
					break;
				}
			}

			if (is_long(inact)) {
				fprintf(stderr, _("\nInvalid input\n\n"));
				free(inact);
				continue;
			}

			if ((status = string_to_long(inact,
						     (void **)&args.inact))) {
				fprintf(stderr,
					_("%s: `%s' failed; %s\n"),
					progname, "string_to_long",
					strerror(errno));
				free(inact);
				goto EXIT;
			}

			free(inact);
			break;
		}

		while (1) {
			fprintf(stdout,
				_("\tAccount Expiration Date (YYYY-MM-DD) [%s]: "),
				date_to_string(args.expire));
			(void)fflush(stdout);

			if (!(expire = input_string(1))) {
				if (errno) {
					fprintf(stderr,
						_("%s: `%s' failed; %s\n"),
						progname, "input_string",
						strerror(errno));
					status = errno;
					goto EXIT;
				} else {
					break;
				}
			}

			if (is_valid_date(expire)) {
				fprintf(stderr, _("\nInvalid input\n\n"));
				free(expire);
				continue;
			}

			if ((status = string_to_date(expire,
						    (void **)&args.expire))) {
				fprintf(stderr,
					_("%s: `%s' failed; %s\n"),
					progname, "string_to_date",
					strerror(errno));
				free(expire);
				goto EXIT;
			}

			free(expire);
			break;
		}
	}

	/* Create filename /etc/shadow.write */
	if (!(spwname = create_filename(SHADOW_FILE, WRITE_EXT))) {
		status = errno;
		goto EXIT;
	}

	/* Create filename /etc/shadow- */
	if (!(spbname = create_filename(SHADOW_FILE, BACKUP_EXT))) {
		status = errno;
		goto EXIT;
	}

	/* Acquire file locks */
	if ((status = lock_files()))
		goto EXIT;

	/* Change umask */
	oldmask = umask(0077);

	/* Open /etc/shadow */
	if (!(sprfp = open_file(SHADOW_FILE, "r"))) {
		status = errno;
		goto EXIT2;
	}

	/* Backup /etc/shadow to /etc/shadow- */
	if ((status = backup_file(SHADOW_FILE, spbname)))
		goto EXIT2;

	/* Copy permissions from /etc/shadow to /etc/shadow- */
	if ((status = copy_file_modes(SHADOW_FILE, spbname)))
		goto EXIT2;

	/* Open /etc/shadow.write */
	if (!(spwfp = open_file(spwname, "w"))) {
		status = errno;
		goto EXIT2;
	}

	/* Perform changes */
	while ((sp = fgetspent(sprfp))) {
		static struct spwd sp2;

		/* Set as an indication that the file has at least 1 entry */
		empty = 0;

		/* Copy the old entry */
		sp2.sp_namp = sp->sp_namp;
		sp2.sp_pwdp = sp->sp_pwdp;
		sp2.sp_flag = sp->sp_flag;

		/* If the entry is part of the array of users to edit,
		 * perform changes; if not, copy the old values
		 */
		if (is_in_array(usrarray, sp->sp_namp)) {
			sp2.sp_lstchg = (args.lstchg != -2) ? args.lstchg :
							      sp->sp_lstchg;
			sp2.sp_min = (args.min != -2) ? args.min :
							sp->sp_min;
			sp2.sp_max = (args.max != -2) ? args.max :
							sp->sp_max;
			sp2.sp_warn = (args.warn != -2) ? args.warn :
							  sp->sp_warn;
			sp2.sp_inact = (args.inact != -2) ? args.inact :
							    sp->sp_inact;
			sp2.sp_expire = (args.expire != -2) ? args.expire :
							      sp->sp_expire;
			changed = 1;
		} else {
			sp2.sp_lstchg = sp->sp_lstchg;
			sp2.sp_min = sp->sp_min;
			sp2.sp_max = sp->sp_max;
			sp2.sp_warn = sp->sp_warn;
			sp2.sp_inact = sp->sp_inact;
			sp2.sp_expire = sp->sp_expire;
		}

		/* Write the entry */
		if ((status = fputspent(&sp2, spwfp))) {
			fprintf(stderr,
				_("%s: `%s' failed; %s\n"),
				progname, "fputspent",
				strerror(errno));
			goto EXIT2;
		}
	}

	/* Make sure no errors occured */
	if (errno && (errno != ENOENT || empty)) {
		fprintf(stderr,
			_("%s: `%s' failed; %s\n"),
			progname, "fgetspent()",
			strerror(errno));
		status = errno;
		goto EXIT2;
	}

	errno = 0;

	/* Close /etc/shadow.write */
	if ((status = close_file(&spwfp)))
		goto EXIT2;

	/* Close /etc/shadow */
	if ((status = close_file(&sprfp)))
		goto EXIT2;

	/* If nothing has changed, don't replace old files */
	if (!changed)
		goto EXIT2;

	/* Everything is in order, move the new file in place */
	if ((status = replace_file(spwname, SHADOW_FILE)))
		goto EXIT2;

	/* Set file permissions properly */
	if ((status = copy_file_modes(spbname, SHADOW_FILE)))
		goto EXIT2;

EXIT2:
	/* Restore umask */
	umask(oldmask);

	/* This file might not exist, but that's ok */
	status = unlink_file(spwname, status);

	/* Release file locks */
	status = unlock_files(status);

EXIT:
	/* Free all allocated memory */
	strfreev(usrarray);
	free(spwname);
	free(spbname);

	return status;
}
