/**
 * @file chgroup.c
 * Change attributes for the specified groups
 *
 * Copyright (C) 2002, 2003, 2004 David Weinehall
 * Copyright (C) 2004, 2006 Free Software Foundation, Inc.
 *
 *  This file is part of GNU Sysutils
 *
 *  GNU Sysutils is free software; you can redistribute it and/or modify
 *  it under the terms of the GNU General Public License as published by
 *  the Free Software Foundation; either version 2 of the License, or
 *  (at your option) any later version.
 *
 *  GNU Sysutils is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU General Public License for more details.
 *
 *  You should have received a copy of the GNU General Public License
 *  along with this program; if not, write to the Free Software Foundation,
 *  Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
 */
#include <argp.h>
#include <grp.h>
#include <errno.h>
#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <gshadow.h>
#include <sys/stat.h>
#include <sys/types.h>

#include "misc.h"
#include "sysutils.h"

#define PRG_NAME "chgroup"	/**< Name shown by --help etc */

extern const char *progname;	/**< Used to store the name of the program */

/** Address to send bug-reports to */
const char *argp_program_bug_address = PACKAGE_BUGREPORT;

/** Usage information */
static char args_doc[] =
	N_("ATTRIBUTE=VALUE... GROUPS");

/** Program synopsis */
static char doc[] =
	N_("Change attributes for the specified groups.\n"
	   "\n"
	   "GROUPS should be a comma-separated list of groups, "
	   "or the word ALL to change attributes for all groups.  "
	   "Note that some attributes can only be changed for one "
	   "group at a time.\n"
	   "\n"
	   "Valid attributes:\n"
	   "\n"
	   "adms                           List of group administrators\n"
	   "group_locked | locked          Lock/Unlock group\n"
	   "groupname (*, **)              The groupname\n"
	   "id | gid (*, **)               The group's id\n"
	   "users                          List of group members\n"
	   "\n"
	   "Attributes marked (*) can only be changed when a single "
	   "group is specified.\n"
	   "Attributes with (**) requires use of the `--force' option");

/** Structure with the available command line options */
static struct argp_option options[] = {
	{ "force", 'f', 0, 0,
	  N_("Force a change of groupname or id"), -2 },
	{ "verbose", 'v', 0, 0,
	  N_("Warn if the specified groups does not exist"), -2 },
	{ 0, 0, 0, 0, 0, 0 }
};

/** Structure to hold output from argument parser */
struct arguments {
	char *const *args;	/**< Arguments */
	int nargs;		/**< Number of arguments */
	int force;		/**< Force a change of groupname or ID */
	int verbose;		/**< Warn about non-existing groups */
};

/**
 * Parse a single option
 *
 * @param key The option
 * @param arg The argument for the option
 * @param state The state of argp
 * @return 0 on success,
 *         ARGP_ERR_UNKNOWN on failure
 */
static error_t parse_opt(int key, char *arg, struct argp_state *state)
{
	struct arguments *args = state->input;
	error_t status = 0;

	switch (key) {
	case 'f':
		args->force = 1;
		break;

	case 'v':
		args->verbose = 1;
		break;

	case ARGP_KEY_INIT:
		args->args = NULL;
		args->nargs = 0;
		args->force = 0;
		args->verbose = 0;
		break;

	case ARGP_KEY_ARG:
		status = ARGP_ERR_UNKNOWN;
		break;

	case ARGP_KEY_ARGS:
		args->args = state->argv + state->next;
		args->nargs = state->argc - state->next;
		break;

	case ARGP_KEY_NO_ARGS:
		argp_usage(state);
		break;

	default:
		(void)arg;
		status = ARGP_ERR_UNKNOWN;
		break;
	}

	return status;
}

/**
 * The program's main-function
 *
 * @param argc The number of arguments
 * @param argv The arguments
 * @return 0 on success, errno on failure
 */
int main(int argc, char *argv[])
{
	FILE *grrfp = NULL;
	FILE *grwfp = NULL;
	struct group *gr;

	FILE *sgrfp = NULL;
	FILE *sgwfp = NULL;
	struct sgrp *sg;

	int empty = 1;
	int changed = 0;

	int j;

	error_t status = 0;

	int singlegroup = 1;

	char **grparray = NULL;

	char *grwname = NULL;
	char *grbname = NULL;
	char *sgwname = NULL;
	char *sgbname = NULL;

	int newlocked = -1;
	gid_t newid = 65535;
	char *newname = NULL;
	char *newadms = NULL;
	char *newusers = NULL;
	char **newadmsv = NULL;
	char **newusersv = NULL;

	mode_t oldmask;

	struct attr attributes[] = {
		{
			.attribute	= AS_ADMS,
			.value		= (void **)&newadms,
			.validator	= is_valid_namelist,
			.converter	= string_to_string
		}, {
			.attribute	= AS_G_LOCKED,
			.value		= (void **)&newlocked,
			.validator	= is_bool,
			.converter	= string_to_bool
		}, {
			.attribute	= AS_LOCKED,
			.value		= (void **)&newlocked,
			.validator	= is_bool,
			.converter	= string_to_bool
		}, {
			.attribute	= AS_GID,
			.value		= (void **)&newid,
			.validator	= is_gid_t,
			.converter	= string_to_gid_t
		}, {
			.attribute	= AS_GROUPNAME,
			.value		= (void **)&newname,
			.validator	= is_valid_name,
			.converter	= string_to_string
		}, {
			.attribute	= AS_ID,
			.value		= (void **)&newid,
			.validator	= is_gid_t,
			.converter	= string_to_gid_t
		}, {
			.attribute	= AS_USERS,
			.value		= (void **)&newusers,
			.validator	= is_valid_namelist,
			.converter	= string_to_string
		}, {
			.attribute	= NULL,
			.value		= NULL,
			.validator	= NULL,
			.converter	= NULL
		}
	};

	/* argp parser */
	struct argp argp = {
		.options	= options,
		.parser		= parse_opt,
		.args_doc	= args_doc,
		.doc		= doc,
	};

	struct arguments args;

	argp_program_version_hook = version;
	argp_err_exit_status = EINVAL;

	errno = 0;

	/* Initialise support for locales, and set the program-name */
	if ((status = init_locales(PRG_NAME)))
		goto EXIT;

	set_author_information(_("Written by David Weinehall.\n"));

	/* Parse command line */
	if ((status = argp_parse(&argp, argc, argv, 0, 0, &args))) {
		if (status != EINVAL)
			fprintf(stderr,
				_("%s: `%s' failed; %s\n"),
				progname, "argp_parse()", strerror(status));

		goto EXIT;
	}

	/* Make sure the caller is a user-admin */
	if ((status = is_useradmin())) {
		if (status == EPERM) {
			fprintf(stderr,
				_("%s: insufficient privileges\n"
				  "You must be a user-administrator to %s.\n"),
				progname, _("change group attributes"));
		} else {
			fprintf(stderr,
				_("%s: `%s' failed; %s\n"),
				progname, "is_useradmin", strerror(errno));
		}

		goto EXIT;
	}

	/* Make sure we get the correct number of arguments */
	if (args.nargs < 2) {
		fprintf(stderr,
			_("%s: too few arguments\n"
			  "Try `%s --help' for more information.\n"),
			progname, progname);
		status = EINVAL;
		goto EXIT;
	}

	/* Parse the attribute=value pairs */
	for (j = 0; j < (args.nargs - 1); j++) {
		if ((status = parse_key_pairs(args.args[j], attributes))) {
			if (status == ENOENT) {
				fprintf(stderr,
					_("%s: invalid attributes\n"
					  "Valid attributes are:\n"
					  "- `%s'\n"
					  "- `%s' | `%s'\n"
					  "- `%s' | `%s'\n"
					  "- `%s'\n"
					  "Try `%s --help' for more "
					  "information.\n"),
					progname,
					AS_ADMS,
					AS_G_LOCKED, AS_LOCKED,
					AS_ID, AS_GID,
					AS_USERS,
					progname);
				status = EINVAL;
			}

			goto EXIT;
		}
	}

	/* If any commands that require `--force' are specified
	 * without the use of `--force', abort.
	 */
	if (!args.force && (newid != 65535 || newname)) {
		fprintf(stderr,
			_("%s: these changes require `--force' to be "
			  "specified\n"),
			progname);
		status = EINVAL;
		goto EXIT;
	}

	/* There are two alternatives here, neither of which are really
	 * pretty; either to read the entire group file once to get
	 * all groupnames, then use them for the ALL list, or to
	 * have separate code for the ALL case and the case of separate
	 * group-entries.  Since the latter is probably the most common,
	 * the latter has been chosen.
	 */
	if (!strcmp(args.args[args.nargs - 1], "ALL")) {
		char *tmp = NULL;

		if (!args.force) {
			fprintf(stderr,
				_("%s: specifying ALL requires `--force' "
				  "to be specified\n"),
				progname);
			status = EINVAL;
			goto EXIT;
		}

		if (!(tmp = get_all_groups())) {
			status = errno;
		} else if (!strlen(tmp)) {
			fprintf(stderr,
				_("%s: could not find any %s; the %s "
				  "might be corrupt\n"),
				progname, _("groups"), _("group database"));
			status = ENOENT;
		} else if (!(grparray = strsplit(tmp, ",", 0))) {
			fprintf(stderr,
				_("%s: `%s' failed; %s\n"),
				progname, "strsplit", strerror(errno));
			status = errno;
		}

		free(tmp);

		if (status)
			goto EXIT;
	} else {
		char *tmp = NULL;
		gid_t i = 0; /* We're scanning <= LASTGID, hence gid_t */

		if (!(grparray = strsplit(args.args[args.nargs - 1], ",", 0))) {
			fprintf(stderr,
				_("%s: `%s' failed; %s\n"),
				progname, "strsplit", strerror(errno));
			status = errno;
			goto EXIT;
		}

		/* If verbose mode has been requested,
		 * warn about all non-existing groups
		 */
		while (args.verbose && (tmp = grparray[i++])) {
			if (!getgrnam(tmp)) {
				if (errno) {
					fprintf(stderr,
						_("%s: `%s' failed; %s\n"),
						progname, "getgrnam()",
						strerror(errno));
					status = errno;
					goto EXIT;
				} else {
					fprintf(stderr,
						_("%s: warning: %s `%s' "
						  "does not exist\n"),
						progname, _("group"),
						tmp);
				}
			}
		}
	}

	/* If more than 1 group is specified, switch to multigroup mode */
	if (grparray[1])
		singlegroup = 0;

	/* If singlegroup commands are specified together
	 * with multiple groups, abort.
	 */
	if (!singlegroup && (newid != 65535 || newname)) {
		fprintf(stderr,
			_("%s: multiple %s specified together with attributes\n"
			  "that can only be used in %s\n"),
			progname, _("groups"), _("single-group mode"));
		status = EINVAL;
		goto EXIT;
	}

	/* Verify that the requested gid is free */
	if (newid != 65535 && (status = is_free_gid(newid))) {
		if (status == EEXIST) {
			fprintf(stderr,
				_("%s: the specified %s already exists\n"),
				progname, _("group id"));
		} else {
			fprintf(stderr,
				_("%s: `%s' failed; %s\n"),
				progname, "is_free_gid", strerror(errno));
		}

		goto EXIT;
	}

	/* Make sure that newid isn't unreasonably large */
	if (newid != 65535 && newid > LASTGID) {
		fprintf(stderr,
			_("%s: the %s cannot be larger than %d\n"),
			progname, _("group id"), LASTGID);
		status = EINVAL;
		goto EXIT;
	}

	/* Verify that the requested groupname is unused */
	if (newname && (status = is_free_groupname(newname))) {
		if (status == EEXIST) {
			fprintf(stderr,
				_("%s: the specified %s already exists\n"),
				progname, _("groupname"));
		} else {
			fprintf(stderr,
				_("%s: `%s' failed; %s\n"),
				progname, "is_free_groupname", strerror(errno));
		}

		goto EXIT;
	}

	/* If newusers or newadms are empty, free and NULL them */
	if (newusers && !strlen(newusers)) {
		free(newusers);
		newusers = NULL;
	}
	if (newadms && !strlen(newadms)) {
		free(newadms);
		newadms = NULL;
	}

	/* Split the list of group members */
	if (newusers && !(newusersv = strsplit(newusers, ",", 0))) {
		fprintf(stderr,
			_("%s: `%s' failed; %s\n"),
			progname, "strsplit", strerror(errno));
		status = errno;
		goto EXIT;
	}

	/* Split the list of group administrators */
	if (newadms && !(newadmsv = strsplit(newadms, ",", 0))) {
		fprintf(stderr,
			_("%s: `%s' failed; %s\n"),
			progname, "strsplit", strerror(errno));
		status = errno;
		goto EXIT;
	}

	/* Verify that all group members exist */
	if (!args.force && newusersv) {
		uid_t i; /* We're scanning <= LASTUID, hence uid_t */

		for (i = 0; newusersv[i]; i++) {
			if (!(status = is_free_username(newusersv[i]))) {
				fprintf(stderr,
					_("%s: one or several specified "
					  "%s does not exist\n"),
					progname, _("group members"));
				status = ENOENT;
				goto EXIT;
			} else if (errno) {
				fprintf(stderr,
					_("%s: `%s' failed; %s\n"),
					progname, "is_free_username",
					strerror(errno));
				status = errno;
				goto EXIT;
			}
		}
	}

	/* Verify that all group administrators exist */
	if (!args.force && newadmsv) {
		uid_t i; /* We're scanning <= LASTUID, hence uid_t */

		for (i = 0; newadmsv[i]; i++) {
			if (!(status = is_free_username(newadmsv[i]))) {
				fprintf(stderr,
					_("%s: one or several specified "
					  "%s does not exist\n"),
					progname, _("group administrators"));
				status = ENOENT;
				goto EXIT;
			} else if (errno) {
				fprintf(stderr,
					_("%s: `%s' failed; %s\n"),
					progname, "is_free_username",
					strerror(errno));
				status = errno;
				goto EXIT;
			}
		}
	}

	/* Create filename /etc/group.write */
	if (!(grwname = create_filename(GROUP_FILE, WRITE_EXT))) {
		status = errno;
		goto EXIT;
	}

	/* Create filename /etc/group- */
	if (!(grbname = create_filename(GROUP_FILE, BACKUP_EXT))) {
		status = errno;
		goto EXIT;
	}

	/* Create filename /etc/gshadow.write */
	if (!(sgwname = create_filename(GSHADOW_FILE, WRITE_EXT))) {
		status = errno;
		goto EXIT;
	}

	/* Create filename /etc/gshadow- */
	if (!(sgbname = create_filename(GSHADOW_FILE, BACKUP_EXT))) {
		status = errno;
		goto EXIT;
	}

	/* Acquire file locks */
	if ((status = lock_files()))
		goto EXIT;

	/* Change umask */
	oldmask = umask(0077);

	/* Open /etc/group */
	if (!(grrfp = open_file(GROUP_FILE, "r"))) {
		status = errno;
		goto EXIT2;
	}

	/* Backup /etc/group to /etc/group- */
	if ((status = backup_file(GROUP_FILE, grbname)))
		goto EXIT2;

	/* Copy permissions from /etc/group to /etc/group- */
	if ((status = copy_file_modes(GROUP_FILE, grbname)))
		goto EXIT2;

	/* Open /etc/group.write */
	if (!(grwfp = open_file(grwname, "w"))) {
		status = errno;
		goto EXIT2;
	}

	/* Open /etc/gshadow */
	if (!(sgrfp = open_file(GSHADOW_FILE, "r"))) {
		status = errno;
		goto EXIT2;
	}

	/* Backup /etc/gshadow to /etc/gshadow- */
	if ((status = backup_file(GSHADOW_FILE, sgbname)))
		goto EXIT2;

	/* Copy permissions from /etc/gshadow to /etc/gshadow- */
	if ((status = copy_file_modes(GSHADOW_FILE, sgbname)))
		goto EXIT2;

	/* Open /etc/gshadow.write */
	if (!(sgwfp = open_file(sgwname, "w"))) {
		status = errno;
		goto EXIT2;
	}

	/* Perform changes on /etc/group */
	while ((gr = fgetgrent(grrfp))) {
		static struct group gr2;

		/* Set as an indication that the file has at least 1 entry */
		empty = 0;

		/* Copy the old entry */
		gr2.gr_passwd = gr->gr_passwd;

		/* If the entry is part of the array of groups to modify,
		 * modify its attributes; if not, copy the old values
		 */
		if (is_in_array(grparray, gr->gr_name)) {
			gr2.gr_name = newname ? newname : gr->gr_name;
			gr2.gr_gid = (newid != 65535) ? newid : gr->gr_gid;
			gr2.gr_mem = newusersv ? newusersv : gr->gr_mem;
		} else {
			gr2.gr_name = gr->gr_name;
			gr2.gr_gid = gr->gr_gid;
			gr2.gr_mem = gr->gr_mem;
		}

		/* Write the entry */
		if ((status = fputgrent(&gr2, grwfp))) {
			fprintf(stderr,
				_("%s: `%s' failed; %s\n"),
				progname, "fputgrent", strerror(errno));

			goto EXIT2;
		}
	}

	/* Make sure no errors occured */
	if (errno && (errno != ENOENT || empty)) {
		fprintf(stderr,
			_("%s: `%s' failed; %s\n"),
			progname, "fgetgrent()",
			strerror(errno));
		status = errno;

		goto EXIT2;
	}

	if (errno == ENOENT)
		errno = 0;

	/* Close /etc/group.write */
	if ((status = close_file(&grwfp)))
		goto EXIT2;

	/* Close /etc/group */
	if ((status = close_file(&grrfp)))
		goto EXIT2;

	/* Perform changes on /etc/gshadow */
	while ((sg = fgetsgent(sgrfp))) {
		static struct sgrp sg2;

		/* Set as an indication that the file has at least 1 entry */
		empty = 0;

		/* Copy the old entry */
		sg2.sg_passwd = sg->sg_passwd;

		/* If the entry is part of the array of groups to modify,
		 * modify its member- and admin-list; if not,
		 * copy the old values
		 */
		if (is_in_array(grparray, sg->sg_name)) {
			sg2.sg_name = newname ? newname : sg->sg_name;
			sg2.sg_adm = newadmsv ? newadmsv : sg->sg_adm;
			sg2.sg_mem = newusersv ? newusersv : sg->sg_mem;
			changed = 1;
		} else {
			sg2.sg_name = sg->sg_name;
			sg2.sg_adm = sg->sg_adm;
			sg2.sg_mem = sg->sg_mem;
		}

		/* Write the entry */
		if ((status = fputsgent(&sg2, sgwfp))) {
			fprintf(stderr,
				_("%s: `%s' failed; %s\n"),
				progname, "fputsgent", strerror(errno));
			goto EXIT2;
		}
	}

	/* Make sure no errors occured */
	if (errno && (errno != ENOENT || empty)) {
		fprintf(stderr,
			_("%s: `%s' failed; %s\n"),
			progname, "fgetsgent",
			strerror(errno));
		status = errno;
		goto EXIT2;
	}

	if (errno == ENOENT)
		errno = 0;

	/* Close /etc/gshadow.write */
	if ((status = close_file(&sgwfp)))
		goto EXIT2;

	/* Close /etc/gshadow */
	if ((status = close_file(&sgrfp)))
		goto EXIT2;

	/* If nothing has changed, don't replace old files */
	if (!changed)
		goto EXIT2;

	/* Everything is in order, move the new files in place */
	if ((status = replace_file(grwname, GROUP_FILE)))
		goto EXIT2;

	if ((status = replace_file(sgwname, GSHADOW_FILE)))
		goto EXIT2;

	/* Set file permissions properly */
	if ((status = copy_file_modes(grbname, GROUP_FILE)))
		goto EXIT2;

	if ((status = copy_file_modes(sgbname, GSHADOW_FILE)))
		goto EXIT2;

EXIT2:
	/* Restore umask */
	umask(oldmask);

	/* These files might not exist, but that's ok */
	status = unlink_file(grwname, status);
	status = unlink_file(sgwname, status);

	/* Release file locks */
	status = unlock_files(status);

EXIT:
	/* Free all allocated memory */
	strfreev(newusersv);
	strfreev(newadmsv);
	strfreev(grparray);
	free(newusers);
	free(newadms);
	free(newname);
	free(grwname);
	free(grbname);
	free(sgwname);
	free(sgbname);

	return status;
}
