/**
 * @file cppw.c
 * Copy the user- or group-database with proper locking
 *
 * Copyright (C) 2002, 2003, 2004 David Weinehall
 * Copyright (C) 2004, 2006 Free Software Foundation, Inc.
 *
 *  This file is part of GNU Sysutils
 *
 *  GNU Sysutils is free software; you can redistribute it and/or modify
 *  it under the terms of the GNU General Public License as published by
 *  the Free Software Foundation; either version 2 of the License, or
 *  (at your option) any later version.
 *
 *  GNU Sysutils is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU General Public License for more details.
 *
 *  You should have received a copy of the GNU General Public License
 *  along with this program; if not, write to the Free Software Foundation,
 *  Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
 */
#include <argp.h>
#include <errno.h>
#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <unistd.h>
#include <sys/stat.h>

#include "misc.h"
#include "sysutils.h"

#define PRG_NAME "cppw"		/**< Name shown by --help etc */
#define PRG_NAME2 "cpgr"	/**< If called as this, default to -g */
#define PRG_NAME3 "cpsp"	/**< If called as this, default to -s */
#define PRG_NAME4 "cpsg"	/**< If called as this, default to -g -s */

extern const char *progname;	/**< Used to store the name of the program */

/** Address to send bug-reports to */
const char *argp_program_bug_address = PACKAGE_BUGREPORT;

/** Usage information */
static char args_doc[] =
	N_("FILE");

/** Program synopsis */
static char doc[] =
	N_("Safely copy FILE to " SYSCONFDIR "/{passwd,shadow,group,gshadow}.\n"
	   "\n"
	   "The program can be called as any of cppw, cpgr, "
	   "cpsp, or cpsg, and will behave accordingly.");

/** Structure with the available command line options */
static struct argp_option options[] = {
	{ "group", 'g', 0, 0,
	  N_("Copy to the group database"), 0 },
	{ "no-shadow", 'n', 0, 0,
	  N_("Copy to the normal database"), 0 },
	{ "passwd", 'p', 0, 0,
	  N_("Copy to the user database"), 0 },
	{ "shadow", 's', 0, 0,
	  N_("Copy to the shadow database"), 0 },
	{ 0, 0, 0, 0, 0, 0 }
};

/** Structure to hold output from argument parser */
struct arguments {
	const char *newfile;		/**< File to copy over the existing */
	/**
	 * Password file to copy to;
	 * F_PASSWD - For /etc/passwd
	 * F_GROUP - For /etc/group
	 */
	int passwd;
	/**
	 * Password file to copy to;
	 * F_NORMAL - For the normal file
	 * F_SHADOW - For the shadow file
	 */
	int shadow;
};

/**
 * Parse a single option
 *
 * @param key The option
 * @param arg The argument for the option
 * @param state The state of argp
 * @return 0 on success,
 *         ARGP_ERR_UNKNOWN on failure
 */
static error_t parse_opt(int key, char *arg, struct argp_state *state)
{
	struct arguments *args = state->input;
	error_t status = 0;

	switch (key) {
	case 'g':
		if (args->passwd == F_PASSWD)
			argp_error(state,
				   _("`-p' cannot be combined with `-g'"));

		args->passwd = F_GROUP;
		break;

	case 'n':
		if (args->shadow == F_SHADOW)
			argp_error(state,
				   _("`-n' cannot be combined with `-s'"));

		args->shadow = F_NORMAL;
		break;

	case 'p':
		if (args->passwd == F_GROUP)
			argp_error(state,
				   _("`-p' cannot be combined with `-g'"));

		args->passwd = F_PASSWD;
		break;

	case 's':
		if (args->shadow == F_NORMAL)
			argp_error(state,
				   _("`-n' cannot be combined with `-s'"));

		args->shadow = F_SHADOW;
		break;

	case ARGP_KEY_INIT:
		args->newfile = NULL;
		args->passwd = F_DEFAULT;
		args->shadow = F_DEFAULT;
		break;

	case ARGP_KEY_ARG:
		if (args->newfile)
			argp_usage(state);

		args->newfile = arg;
		break;

	case ARGP_KEY_NO_ARGS:
		argp_usage(state);

	default:
		status = ARGP_ERR_UNKNOWN;
		break;
	}

	return status;
}

/**
 * The program's main-function
 *
 * @param argc The number of arguments
 * @param argv The arguments
 * @return 0 on success, errno on failure
 */
int main(int argc, char *argv[])
{
	error_t status = 0;

	char *bname = NULL;
	char *cname = NULL;
	const char *fname = NULL;

	struct stat st;

	int passwd = F_DEFAULT;
	int shadow = F_DEFAULT;

	errno = 0;

	/* argp parser */
	struct argp argp = {
		.options	= options,
		.parser		= parse_opt,
		.args_doc	= args_doc,
		.doc		= doc,
	};

	struct arguments args;

	argp_program_version_hook = version;
	argp_err_exit_status = EINVAL;

	/* Initialise support for locales, and set the program-name */
	if ((status = init_locales(PRG_NAME)))
		goto EXIT;

	set_author_information(_("Written by David Weinehall.\n"));

	/* Parse command line */
	if ((status = argp_parse(&argp, argc, argv, 0, 0, &args))) {
		if (status != EINVAL)
			fprintf(stderr,
				_("%s: `%s' failed; %s\n"),
				progname, "argp_parse()", strerror(status));

		goto EXIT;
	}

	/* Setup mode of operation based on program-name */
	if (!strcmp(argv[0], PRG_NAME)) {
		passwd = F_PASSWD;
		shadow = F_NORMAL;
	} else if (!strcmp(argv[0], PRG_NAME2)) {
		passwd = F_GROUP;
		shadow = F_NORMAL;
	} else if (!strcmp(argv[0], PRG_NAME3)) {
		passwd = F_PASSWD;
		shadow = F_SHADOW;
	} else if (!strcmp(argv[0], PRG_NAME4)) {
		passwd = F_GROUP;
		shadow = F_SHADOW;
	}

	if (args.passwd != -1)
		passwd = args.passwd;

	if (passwd == F_DEFAULT)
		passwd = F_PASSWD;

	if (args.shadow != -1)
		shadow = args.shadow;

	if (shadow == F_DEFAULT)
		shadow = F_NORMAL;

	/* Create the proper file-name to copy to */
	if (passwd == F_PASSWD) {
		if (shadow)
			fname = SHADOW_FILE;
		else
			fname = PASSWD_FILE;
	} else {
		if (shadow)
			fname = GSHADOW_FILE;
		else
			fname = GROUP_FILE;
	}

	/* Make sure the caller has root privileges */
	if ((status = is_root())) {
		if (status == EPERM) {
			fprintf(stderr,
				_("%s: insufficient privileges\n"
				  "You must be root to overwrite `%s'.\n"),
				progname, fname);
		} else {
			fprintf(stderr,
				_("%s: `%s' failed; %s\n"),
				progname, "is_root", strerror(errno));
		}

		goto EXIT;
	}

	/* Create filename /etc/<file>- */
	if (!(bname = create_filename(fname, BACKUP_EXT))) {
		status = errno;
		goto EXIT;
	}

	/* Create filename /etc/<file>.copy */
	if (!(cname = create_filename(fname, COPY_EXT))) {
		status = errno;
		goto EXIT;
	}

	/* Acquire file locks */
	if ((status = lock_files()))
		goto EXIT;

	/* Check that the source file /etc/<file> exists */
	if ((stat(fname, &st)) != -1) {
		/* Backup file to /etc/<file>- */
		if ((status = backup_file(fname, bname)))
			goto EXIT2;

		/* Copy permissions from /etc/<file> to /etc/<file>- */
		if ((status = copy_file_modes(fname, bname)))
			goto EXIT2;
	} else if (errno != ENOENT) {
		fprintf(stderr,
			_("%s: `%s' failed; %s\n"),
			progname, "stat()", strerror(errno));
		status = errno;
		goto EXIT2;
	} else {
		errno = 0;
		fprintf(stderr,
			_("%s: warning: `%s' does not exist; no backup made\n"),
			progname, fname);
	}

	/* Copy <newfile> to /etc/<file>.copy */
	if ((status = copy_file(args.newfile, cname)))
		goto EXIT2;

	/* Set the permissions for /etc/<file>.copy */
	if ((status = set_file_perms(cname, "root",
				     args.shadow ? "shadow" : "root",
				     args.shadow ? 0640 : 0644)))
		goto EXIT2;

	/* Finally, move the new file in place */
	status = replace_file(cname, fname);

EXIT2:
	/* This file might not exist, but that's ok */
	status = unlink_file(cname, status);

	/* Release file locks */
	status = unlock_files(status);

EXIT:
	/* Free all allocated memory */
	free(bname);
	free(cname);

	return status;
}
