/**
 * @file gshadow.c
 * Local implementation of gshadow until glibc includes proper support
 *
 * Copyright (C) 2002, 2003, 2004 David Weinehall
 * Copyright (C) 2004 Free Software Foundation, Inc.
 *
 *  This library is free software; you can redistribute it and/or
 *  modify it under the terms of the GNU Lesser General Public
 *  License as published by the Free Software Foundation; either
 *  version 2.1 of the License, or (at your option) any later version.
 *
 *  This library is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 *  Lesser General Public License for more details.
 *
 *  You should have received a copy of the GNU Lesser General Public License
 *  along with this library; if not, write to the Free Software Foundation,
 *  Inc., 59 Temple Place, Suite 330, Boston, MA 02111-1307  USA
 */
#include <gshadow.h>

#define BUFSIZE		BUFSIZ * 4	/**< Size of the buffer to use */

#define MAXADM		1024		/**< Max number of group admins */
#define MAXMEM		1024		/**< Max number of group members */

#define NIS_MARKER	'+'		/**< Marker used by NIS-entries */

static FILE *sgstream = NULL;		/**< Stream used for group shadow */
static struct sgrp sg;			/**< Local group shadow struct copy */

static char sgbuf[BUFSIZE];		/**< Buffer used internally */
static char *admarray[MAXADM + 1];	/**< Array of administrators */
static char *memarray[MAXMEM + 1];	/**< Array of members */

/**
 * Split a list into an array
 *
 * @param list The list to split into an array
 * @param array The array to split the list into
 */
static void split_list(char *list, char **array)
{
	int i = 0;

	while (list && *list) {
		array[i++] = list;

		if ((list = strchr(list, ',')))
			*list++ = '\0';
	}

	array[i] = NULL;
}

/**
 * Rewind gshadow-file stream
 */
void setsgent(void)
{
	if (!sgstream)
		sgstream = fopen(GSHADOW_FILE, "r");
	else
		rewind(sgstream);
}

/**
 * Close gshadow-file stream
 */
void endsgent(void)
{
	if (sgstream)
		(void)fclose(sgstream);

	sgstream = NULL;
}

/**
 * Read the next sgrp entry from a string
 *
 * @param string The string to read from
 * @return A pointer to a struct with the sgrp entry
 */
struct sgrp *sgetsgent(const char *string)
{
	char *admlist = NULL;
	char *memlist = NULL;
	char *p = NULL;

	/* If the string is empty, return */
	if (!string)
		return NULL;

	/* Copy the input string to the buffer */
	strncpy(sgbuf, string, sizeof sgbuf - 1);
	sgbuf[sizeof sgbuf - 1] = '\0';

	/* If we find a newline, end the string there */
	if ((p = strchr(sgbuf, '\n')))
		p = '\0';

	sg.sg_name = sgbuf;

	if (!(p = strchr(sgbuf, ':')))
		return NULL;

	/* Replace the ':' with a '\0' */
	*p++ = '\0';
	sg.sg_passwd = p;

	if (!(p = strchr(p, ':')))
		return NULL;

	/* Replace the ':' with a '\0' */
	*p++ = '\0';
	admlist = p;
	sg.sg_adm = admarray;

	if (!(p = strchr(p, ':')))
		return NULL;

	/* Replace the ':' with a '\0' */
	*p++ = '\0';
	memlist = p;
	sg.sg_mem = memarray;

	/* If there are too many separators, abort */
	if (strchr(p, ':'))
		return NULL;

	/* Split the lists into NULL-terminated arrays */
	split_list(admlist, admarray);
	split_list(memlist, memarray);

	return &sg;
}

/**
 * Read the next sgrp entry from stream
 *
 * @param stream The stream to read from
 * @return A pointer to a struct with the sgrp entry
 */
struct sgrp *fgetsgent(FILE *stream)
{
	char buf[sizeof sgbuf];
	char *p;

	/* Abort if no filepointer is provided */
	if (!stream)
		return 0;

	/* If all is well, replace the first occurence of '\n'
	 * with '\0', then return the parsed entry
	 */
	while (fgets(buf, sizeof buf, stream)) {
		/* Just skip NIS entries */
		if (buf[0] == NIS_MARKER)
			continue;

		if ((p = strchr(buf, '\n')))
			*p = '\0';

		return sgetsgent(buf);
	}

	/* Something went wrong.  Abort */
	return 0;
}

/**
 * Read an entry from the gshadow-file stream, opening it if necessary
 *
 * @return A pointer to a struct with the sgrp entry
 */
struct sgrp *getsgent(void)
{
	if (!sgstream)
		setsgent();

	return fgetsgent(sgstream);
}

/**
 * Search for an entry with a matching name
 *
 * @param name The group name to search for
 * @return A pointer to a struct with the sgrp entry
 */
struct sgrp *getsgnam(const char *name)
{
	endsgent();

	while (getsgent()) {
		if (!strcmp(name, sg.sg_name))
			return &sg;
	}

	return 0;
}
