/**
 * @file lsuser.c
 * List attributes for the specified users
 *
 * Copyright (C) 2002, 2003, 2004 David Weinehall
 * Copyright (C) 2004, 2006 Free Software Foundation, Inc.
 *
 *  This file is part of GNU Sysutils
 *
 *  GNU Sysutils is free software; you can redistribute it and/or modify
 *  it under the terms of the GNU General Public License as published by
 *  the Free Software Foundation; either version 2 of the License, or
 *  (at your option) any later version.
 *
 *  GNU Sysutils is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU General Public License for more details.
 *
 *  You should have received a copy of the GNU General Public License
 *  along with this program; if not, write to the Free Software Foundation,
 *  Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
 */
#include <argp.h>
#include <pwd.h>
#include <errno.h>
#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <unistd.h>
#include <sys/types.h>

#include "misc.h"
#include "sysutils.h"

#define PRG_NAME "lsuser"	/**< Name shown by --help etc */

extern const char *progname;	/**< Used to store the name of the program */

/** Address to send bug-reports to */
const char *argp_program_bug_address = PACKAGE_BUGREPORT;

/** Usage information */
static char args_doc[] =
	N_("[ATTRIBUTE=VALUE...] USERS\n"
	   "-l");

/** Program synopsis */
static char doc[] =
	N_("List attributes for the specified users that matches the\n"
	   "specified attributes.\n"
	   "\n"
	   "USERS should be a comma-separated list of users, "
	   "or the word ALL to list attributes for all users.\n"
	   "\n"
	   "Valid attributes:\n"
	   "\n"
	   "account_locked | locked        Locked/unlocked account\n"
	   "admgroups                      List of the groups that the "
	   "users administers\n"
	   "fname | fullname               The user's full name\n"
	   "gecos                          The user's gecos-field\n"
	   "groups                         The users' group "
	   "membership list\n"
	   "home                           The user's home directory\n"
	   "hphone                         The user's home phone\n"
	   "id | uid                       The user's id\n"
	   "other                          Other information\n"
	   "pgrp                           The users' primary group\n"
	   "room                           The user's room number\n"
	   "shell                          The users' login shell\n"
	   "username                       The username\n"
	   "wphone                         The user's work phone\n");

/** Structure with the available command line options */
static struct argp_option options[] = {
	{ "attr", 'a', N_("ATTRS"), OPTION_ARG_OPTIONAL,
	  N_("Display only the attributes specified "
	     "by a comma-separated list; valid attributes are: "
	     "account_locked | locked, admgroups, gecos, groups, "
	     "home, id | uid, pgrp, and shell"), 0 },
	{ "colon", 'c', 0, 0,
	  N_("Display the attributes as colon-separated records"), 0 },
	{ "stanza", 'f', 0, 0,
	  N_("Display the attributes as stanzas"), 0 },
	{ "gecos", 'g', N_("FIELDS"), OPTION_ARG_OPTIONAL,
	  N_("Separate the gecos fields "
	     "and display only the fields "
	     "specified by a comma-separated "
	     "list; valid fields are: "
	     "fname | fullname, hphone, wphone, room, and other"), 0 },
	{ "list", 'l', 0, 0,
	  N_("List the names of all users, seperated by commas"), 0 },
	{ "verbose", 'v', 0, 0,
	  N_("Warn if the specified users do not exist"), -2 },
	{ 0, 0, 0, 0, 0, 0 }
};

/** Structure to hold output from argument parser */
struct arguments {
	char *const *args;	/**< Arguments */
	const char *attrs;	/**< attribute=value pairs */
	const char *gecos;	/**< GECOS-fields to show information for */
	int nargs;		/**< Number of arguments */
	int colon;		/**< Display as colon separated records */
	int list;		/**< Comma separated list of all groups */
	int stanza;		/**< Display information as stanzas */
	int normal;		/**< Use normal output format */
	int verbose;		/**< Warn about non-existing users */
};

/**
 * Parse a single option
 *
 * @param key The option
 * @param arg The argument for the option
 * @param state The state of argp
 * @return 0 on success,
 *         ARGP_ERR_UNKNOWN on failure
 */
static error_t parse_opt(int key, char *arg, struct argp_state *state)
{
	struct arguments *args = state->input;
	error_t status = 0;

	switch (key) {
	case 'a':
		args->attrs = nstr(arg);
		break;

	case 'g':
		args->gecos = nstr(arg);
		break;

	case 'c':
		args->colon = 1;
		args->normal = 0;
		break;

	case 'f':
		args->stanza = 1;
		args->normal = 0;
		break;

	case 'l':
		args->list = 1;
		break;

	case 'v':
		args->verbose = 1;
		break;

	case ARGP_KEY_INIT:
		args->args = NULL;
		args->nargs = 0;
		args->attrs = NULL;
		args->gecos = NULL;
		args->colon = 0;
		args->list = 0;
		args->normal = 1;
		args->stanza = 0;
		args->verbose = 0;
		break;

	case ARGP_KEY_ARGS:
		args->args = state->argv + state->next;
		args->nargs = state->argc - state->next;
		break;

	case ARGP_KEY_NO_ARGS:
		if (!args->list)
			argp_usage(state);

		break;

	case ARGP_KEY_END:
		if ((args->args) &&
		    (strchr(args->args[args->nargs - 1], '=')))
			argp_usage(state);

		if ((args->list) && (args->args))
			argp_usage(state);

		break;

	default:
		status = ARGP_ERR_UNKNOWN;
		break;
	}

	return status;
}

/**
 * The program's main-function
 *
 * @param argc The number of arguments
 * @param argv The arguments
 * @return 0 on success, errno on failure
 */
int main(int argc, char *argv[])
{
	error_t status = 0;

	unsigned int attr = A_ALL;
	unsigned int gattr = G_NONE;

	int shadowread = 0;
	int gshadowread = 0;
	int isuseradmin = 0;

	char **usrarray = NULL;

	gid_t user_gid = getgid();
	gid_t priv_gid = getegid();

	uid_t i; /* We're scanning <= LASTUID, hence uid_t */

	int j;

	int lslocked = -1;
	uid_t lsid = 65535;
	char *lspgrp = NULL;
	char *lsname = NULL;
	char *lsshell = NULL;
	char *lshomedir = NULL;
	char *lsgroups = NULL;
	char *lsadmgroups = NULL;

	/* GECOS information */
	char *lsgecos = NULL;
	char *lsfname = NULL;
	char *lsroom = NULL;
	char *lswphone = NULL;
	char *lshphone = NULL;
	char *lsother = NULL;

	struct attr attributes[] = {
		{
			.attribute	= AS_A_LOCKED,
			.value		= (void **)&lslocked,
			.validator	= is_bool,
			.converter	= string_to_bool
		}, {
			.attribute	= AS_ADMGROUPS,
			.value		= (void **)&lsadmgroups,
			.validator	= is_valid_namelist,
			.converter	= string_to_string
		}, {
			.attribute	= GS_FNAME,
			.value		= (void **)&lsfname,
			.validator	= is_valid_gecos_field,
			.converter	= string_to_string
		}, {
			.attribute	= GS_FULLNAME,
			.value		= (void **)&lsfname,
			.validator	= is_valid_gecos_field,
			.converter	= string_to_string
		}, {
			.attribute	= AS_GECOS,
			.value		= (void **)&lsgecos,
			.validator	= is_valid_gecos,
			.converter	= string_to_string
		}, {
			.attribute	= AS_GROUPS,
			.value		= (void **)&lsgroups,
			.validator	= is_valid_namelist,
			.converter	= string_to_string
		}, {
			.attribute	= AS_HOME,
			.value		= (void **)&lshomedir,
			.validator	= is_valid_path,
			.converter	= string_to_string
		}, {
			.attribute	= GS_HPHONE,
			.value		= (void **)&lshphone,
			.validator	= is_valid_gecos_field,
			.converter	= string_to_string
		}, {
			.attribute	= AS_ID,
			.value		= (void **)&lsid,
			.validator	= is_uid_t,
			.converter	= string_to_uid_t
		}, {
			.attribute	= AS_LOCKED,
			.value		= (void **)&lslocked,
			.validator	= is_bool,
			.converter	= string_to_bool
		}, {
			.attribute	= GS_OTHER,
			.value		= (void **)&lsother,
			.validator	= is_valid_gecos_other,
			.converter	= string_to_string
		}, {
			.attribute	= AS_PGRP,
			.value		= (void **)&lspgrp,
			.validator	= is_valid_name,
			.converter	= string_to_string
		}, {
			.attribute	= GS_ROOM,
			.value		= (void **)&lsroom,
			.validator	= is_valid_gecos_field,
			.converter	= string_to_string
		}, {
			.attribute	= AS_SHELL,
			.value		= (void **)&lsshell,
			.validator	= is_valid_shell,
			.converter	= string_to_string
		}, {
			.attribute	= AS_UID,
			.value		= (void **)&lsid,
			.validator	= is_uid_t,
			.converter	= string_to_uid_t
		}, {
			.attribute	= AS_USERNAME,
			.value		= (void **)&lsname,
			.validator	= is_valid_name,
			.converter	= string_to_string
		}, {
			.attribute	= GS_WPHONE,
			.value		= (void **)&lswphone,
			.validator	= is_valid_gecos_field,
			.converter	= string_to_string
		}, {
			.attribute	= NULL,
			.value		= NULL,
			.validator	= NULL,
			.converter	= NULL
		}
	};

	/* argp parser */
	struct argp argp = {
		.options	= options,
		.parser		= parse_opt,
		.args_doc	= args_doc,
		.doc		= doc,
	};

	struct arguments args;

	/* Drop sgid */
	setegid(user_gid);
	errno = 0;

	argp_program_version_hook = version;

	/* Initialise support for locales, and set the program-name */
	if ((status = init_locales(PRG_NAME)))
		goto EXIT;

	set_author_information(_("Written by David Weinehall.\n"));

	/* Parse command line */
	if ((status = argp_parse(&argp, argc, argv, 0, 0, &args))) {
		if (status != EINVAL)
			fprintf(stderr,
				_("%s: `%s' failed; %s\n"),
				progname, "argp_parse()", strerror(status));

		goto EXIT;
	}

	/* Find out if the shadow files can be read or not */

	/* Regain sgid */
	setegid(priv_gid);
	errno = 0;

	/* Attempt to get an entry from the shadow password database */
	(void)getspent();

	/* Did something fail? */
	if (errno) {
		if (errno == EACCES || errno == ENOENT) {
			errno = 0;
			attr &= ~A_LOCKED;
		} else {
			status = errno;

			/* Drop sgid */
			setegid(user_gid);
			errno = status;

			fprintf(stderr,
				_("%s: `%s' failed; %s\n"),
				progname, "getspent()", strerror(status));

			goto EXIT;
		}
	} else {
		shadowread = 1;
	}

	/* Attempt to get an entry from the group shadow password database */
	(void)getsgent();

	/* Did something fail? */
	if (errno) {
		if (errno == EACCES || errno == ENOENT) {
			errno = 0;
			attr &= ~A_ADMGROUPS;
		} else {
			status = errno;

			/* Drop sgid */
			setegid(user_gid);
			errno = status;

			fprintf(stderr,
				_("%s: `%s' failed; %s\n"),
				progname, "getsgent()", strerror(status));

			goto EXIT;
		}
	} else {
		gshadowread = 1;
	}

	/* Drop sgid */
	setegid(user_gid);
	errno = 0;

	/* Parse the attributes if --attr is used */
	if (args.attrs) {
		char **attrarray = NULL;

		attr = A_NAME;

		if (!(attrarray = strsplit(args.attrs, ",", 0))) {
			fprintf(stderr,
				_("%s: `%s' failed; %s\n"),
				progname, "strsplit", strerror(errno));
			status = errno;
			goto EXIT;
		}

		for (i = 0; attrarray[i]; i++) {
			if (!strlen(attrarray[i])) {
				continue;
			} else if (!strcmp(attrarray[i], AS_ID)) {
				attr |= A_ID;
			} else if (!strcmp(attrarray[i], AS_UID)) {
				attr |= A_ID;
			} else if (!strcmp(attrarray[i], AS_PGRP)) {
				attr |= A_PGRP;
			} else if (!strcmp(attrarray[i], AS_GROUPS)) {
				attr |= A_GROUPS;
			} else if (!strcmp(attrarray[i], AS_HOME)) {
				attr |= A_HOME;
			} else if (!strcmp(attrarray[i], AS_SHELL)) {
				attr |= A_SHELL;
			} else if (!strcmp(attrarray[i], AS_GECOS)) {
				attr |= A_GECOS;
			} else if (gshadowread &&
				   !strcmp(attrarray[i], AS_ADMGROUPS)) {
				attr |= A_ADMGROUPS;
			} else if (shadowread &&
				   !strcmp(attrarray[i], AS_LOCKED)) {
				attr |= A_LOCKED;
			} else if (shadowread &&
				   !strcmp(attrarray[i], AS_A_LOCKED)) {
				attr |= A_LOCKED;
			} else {
				fprintf(stderr,
					_("%s: invalid attributes "
					  "for `--attr'\n"
					  "Valid attributes are:\n"
				          "%s%sgecos, groups, "
					  "home, id | uid, pgrp, and shell\n"
					  "Try `%s --help' for more "
					  "information.\n"),
					progname,
					shadowread ? "account_locked | "
						     "locked, " : "",
					gshadowread ? "admgroups, " : "",
					progname);
				status = EINVAL;
				strfreev(attrarray);
				goto EXIT;
			}
		}

		strfreev(attrarray);
	}

	if (args.gecos) {
		char **attrarray = NULL;

		if (!(attrarray = strsplit(args.gecos, ",", 0))) {
			fprintf(stderr,
				_("%s: `%s' failed; %s\n"),
				progname, "strsplit", strerror(errno));
			status = errno;
			goto EXIT;
		}

		for (i = 0; attrarray[i]; i++) {
			if (!strcmp(attrarray[i], GS_FNAME)) {
				gattr |= G_FNAME;
			} else if (!strcmp(attrarray[i], GS_FULLNAME)) {
				gattr |= G_FNAME;
			} else if (!strcmp(attrarray[i], GS_ROOM)) {
				gattr |= G_ROOM;
			} else if (!strcmp(attrarray[i], GS_WPHONE)) {
				gattr |= G_WPHONE;
			} else if (!strcmp(attrarray[i], GS_HPHONE)) {
				gattr |= G_HPHONE;
			} else if (!strcmp(attrarray[i], GS_OTHER)) {
				gattr |= G_OTHER;
			} else {
				fprintf(stderr,
					_("%s: invalid attributes "
					  "for `--gecos'\n"
					  "Valid attributes are:\n"
					  "fname | fullname, hphone, "
					  "other, room, and wphone\n"
					  "Try `%s --help' for more "
					  "information.\n"),
					progname, progname);
				status = EINVAL;
				strfreev(attrarray);
				goto EXIT;
			}
		}

		strfreev(attrarray);
	}

	/* There are two alternatives here, neither of which are really
	 * pretty; either to read the entire passwd file once to get
	 * all usernames, then use them for the ALL list, or to
	 * have separate code for the ALL case and the case of separate
	 * user-entries.  Since the latter is probably the most common,
	 * the latter has been chosen.
	 */
	if (args.list || !strcmp(args.args[args.nargs - 1], "ALL")) {
		char *tmp = NULL;

		if (!(tmp = get_all_users())) {
			status = errno;
		} else if (!strlen(tmp)) {
			fprintf(stderr,
				_("%s: could not find any %s; the %s "
				  "might be corrupt\n"),
				progname, _("users"), _("user database"));
			status = ENOENT;
		} else if (args.list) {
			fprintf(stdout, "%s\n", tmp);
		} else if (!(usrarray = strsplit(tmp, ",", 0))) {
			fprintf(stderr,
				_("%s: `%s' failed; %s\n"),
				progname, "strsplit", strerror(errno));
			status = errno;
		}

		free(tmp);

		if (status || args.list)
			goto EXIT;
	} else {
		if (!(usrarray = strsplit(args.args[args.nargs - 1], ",", 0))) {
			fprintf(stderr,
				_("%s: `%s' failed; %s\n"),
				progname, "strsplit", strerror(errno));
			status = errno;
			goto EXIT;
		}
	}

	/* Parse the attribute=value pairs */
	for (j = 0; j < (args.nargs - 1); j++) {
		if ((status = parse_key_pairs(args.args[j], attributes))) {
			if (status == ENOENT) {
				fprintf(stderr,
					_("%s: invalid attributes\n"
					"Valid attributes are:\n"
					"- `%s' | `%s'\n"
					"- `%s'\n"
					"- `%s' | `%s'\n"
					"- `%s'\n"
					"- `%s'\n"
					"- `%s'\n"
					"- `%s'\n"
					"- `%s' | `%s'\n"
					"- `%s'\n"
					"- `%s'\n"
					"- `%s'\n"
					"- `%s'\n"
					"- `%s'\n"
					"- `%s'\n"
					"Try `%s --help' for more "
					"information.\n"),
					progname,
					AS_A_LOCKED, AS_LOCKED,
					AS_ADMGROUPS,
					GS_FNAME, GS_FULLNAME,
					AS_GECOS,
					AS_GROUPS,
					AS_HOME,
					GS_HPHONE,
					AS_ID, AS_GID,
					GS_OTHER,
					AS_PGRP,
					GS_ROOM,
					AS_SHELL,
					AS_USERNAME,
					GS_WPHONE,
					progname);
				status = EINVAL;
			}

			goto EXIT;
		}
	}

	/* Ok, now the only thing left is to list all specified users */
	for (i = 0; usrarray[i]; i++) {
		char *admgrplist = NULL;
		char *grplist = NULL;
		char *pgroup = NULL;
		char **gecosv = NULL;
		struct passwd *pw;
		int locked = 0;

		/* Get the next user entry */
		if (!(pw = getpwnam(usrarray[i])) && errno) {
			fprintf(stderr,
				_("%s: `%s' failed; %s\n"),
				progname, "getpwnam()", strerror(errno));
			status = errno;
			goto EXIT;
		}

		/* Skip non-existing entries */
		if (!pw) {
			if (args.verbose)
				fprintf(stderr,
					_("%s: warning: %s `%s' "
					  "does not exist ... skipping\n"),
					progname, _("user"), usrarray[i]);
			continue;
		}

		isuseradmin = !is_useradmin();
		errno = 0;

		/* Find out if the account is locked
		 * (only works if we have read access to /etc/shadow)
		 */
		if ((isuseradmin || (getuid() == pw->pw_uid)) &&
		    shadowread && attr & A_LOCKED) {
			/* Regain sgid */
			setegid(priv_gid);
			errno = 0;

			if ((locked = is_user_locked(pw->pw_name)) == -1) {
				if (errno != ENOENT) {
					status = errno;

					/* Drop sgid */
					setegid(user_gid);
					errno = 0;

					goto EXIT;
				}

				/* Drop sgid */
				setegid(user_gid);
				errno = 0;

				/* Either we don't have /etc/shadow,
				 * or there was no such user
				 */
				if (args.verbose)
					fprintf(stderr,
						_("%s: warning: `%s' "
						  "does not exist in the "
						  "user shadow database\n"),
						progname, pw->pw_name);
				locked = (strlen(pw->pw_passwd) == 1);
			} else {
				/* Drop sgid */
				setegid(user_gid);
				errno = 0;
			}
		}

		/* Get the list of admgroups
		 * (only works if we have read access to /etc/gshadow)
		 */
		if ((isuseradmin || (getuid() == pw->pw_uid)) &&
		    gshadowread && attr & A_ADMGROUPS) {
			/* Regain sgid */
			setegid(priv_gid);
			errno = 0;

			if (!(admgrplist = get_admgroups(pw->pw_name))) {
				if (errno != ENOENT) {
					status = errno;

					/* Drop sgid */
					setegid(user_gid);
					errno = 0;

					goto EXIT;
				}

				/* Either we don't have /etc/gshadow,
				 * or there was no such user
				 */
				admgrplist = NULL;
			}

			/* Drop sgid */
			setegid(user_gid);
			errno = 0;
		}

		/* Get the list of groups */
		if (!(grplist = get_groups(pw->pw_name))) {
			free(admgrplist);
			status = errno;
			goto EXIT;
		}

		/* Get the GECOS information */
		if (!(gecosv = split_gecos(pw->pw_gecos))) {
			free(admgrplist);
			free(grplist);
			status = errno;
			goto EXIT;
		}

		/* Find out the primary group */
		if (!(pgroup = get_groupname(pw->pw_gid))) {
			if (errno)
				status = errno;

			if (!errno && !(pgroup = strdup(_("(Not Existing)")))) {
				fprintf(stderr,
					_("%s: `%s' failed; %s\n"),
					progname, "strdup()", strerror(errno));
				status = errno;
			}

			if (status) {
				free(admgrplist);
				free(grplist);
				strfreev(gecosv);
				goto EXIT;
			}
		}


		/* Filter based on attributes */
		if ((lslocked > -1) && (lslocked != locked))
			continue;

		if ((lsid != 65535) && (lsid != pw->pw_uid))
			continue;

		if ((lspgrp) && (strcmp(lspgrp, pgroup)))
			continue;

		if ((lsgroups) && (strcmp(lsgroups, grplist)))
			continue;

		if ((lsadmgroups) && (strcmp(lsadmgroups, nstr(admgrplist))))
			continue;

		if ((lshomedir) && (strcmp(lshomedir, pw->pw_dir)))
			continue;

		if ((lsshell) && (strcmp(lsshell, pw->pw_shell)))
			continue;

		if ((lsgecos) && (strcmp(nstr(lsgecos), nstr(pw->pw_gecos))))
			continue;

		if ((lsfname) && (strcmp(lsfname, gecosv[0])))
			continue;

		if ((lsroom) && (strcmp(lsroom, gecosv[1])))
			continue;

		if ((lswphone) && (strcmp(lswphone, gecosv[2])))
			continue;

		if ((lshphone) && (strcmp(lshphone, gecosv[3])))
			continue;

		if ((lsother) && (strcmp(lsother, gecosv[4])))
			continue;

		/* Output all information */
		if (args.normal) {
			fprintf(stdout, "%s", pw->pw_name);

			if ((attr & A_LOCKED) &&
			    ((getuid() == pw->pw_uid) || isuseradmin))
				fprintf(stdout,
					" %s=%s",
					_("locked"),
					locked ? _("yes") : _("no"));

			if (attr & A_ID)
				fprintf(stdout,
					" %s=%d",
					_("id"), pw->pw_uid);

			if (attr & A_PGRP)
				fprintf(stdout,
					" %s=%s",
					_("pgrp"), pgroup);

			if (attr & A_GROUPS)
				fprintf(stdout,
					" %s=%s",
					_("groups"), grplist);

			if ((attr & A_ADMGROUPS) &&
			    ((getuid() == pw->pw_uid) || isuseradmin))
				fprintf(stdout,
					" %s=%s",
					_("admgroups"), nstr(admgrplist));

			if (attr & A_HOME)
				fprintf(stdout,
					" %s=%s",
					_("home"), nstr(pw->pw_dir));

			if (attr & A_SHELL)
				fprintf(stdout,
					" %s=%s",
					_("shell"), nstr(pw->pw_shell));

			if (gattr == G_NONE) {
				if (attr & A_GECOS)
					fprintf(stdout,
						" %s=%s",
						_("gecos"), nstr(pw->pw_gecos));
			} else {
				if (gattr & G_FNAME)
					fprintf(stdout,
						" %s=%s",
						_("fname"), gecosv[0]);

				if (gattr & G_ROOM)
					fprintf(stdout,
						" %s=%s",
						_("room"), gecosv[1]);

				if (gattr & G_WPHONE)
					fprintf(stdout,
						" %s=%s",
						_("wphone"), gecosv[2]);

				if (gattr & G_HPHONE)
					fprintf(stdout,
						" %s=%s",
						_("hphone"), gecosv[3]);

				if (gattr & G_OTHER)
					fprintf(stdout,
						" %s=%s",
						_("other"), gecosv[4]);
			}
		} else if (args.colon) {
			fprintf(stdout,
				"#%s%s%s%s%s%s%s%s%s%s%s%s%s%s%s%s%s",
				_("name"),
				((attr & A_LOCKED) &&
				 ((getuid() == pw->pw_uid) ||
				  isuseradmin)) ? ":" : "",
				((attr & A_LOCKED) &&
				 ((getuid() == pw->pw_uid) ||
				  isuseradmin)) ? _("locked") : "",
				(attr & A_ID) ? ":" : "",
				(attr & A_ID) ? _("id") : "",
				(attr & A_PGRP) ? ":" : "",
				(attr & A_PGRP) ? _("pgrp") : "",
				(attr & A_GROUPS) ? ":" : "",
				(attr & A_GROUPS) ? _("groups") : "",
				((attr & A_ADMGROUPS) &&
				 ((getuid() == pw->pw_uid) ||
				  isuseradmin)) ? ":" : "",
				((attr & A_ADMGROUPS) &&
				 ((getuid() == pw->pw_uid) ||
				  isuseradmin)) ? _("admgroups") : "",
				(attr & A_HOME) ? ":" : "",
				(attr & A_HOME) ? _("home") : "",
				(attr & A_SHELL) ? ":" : "",
				(attr & A_SHELL) ? _("shell") : "",
				(gattr == G_NONE &&
				 attr & A_GECOS) ? ":" : "",
				(gattr == G_NONE &&
				 attr & A_GECOS) ? _("gecos") : "");
			fprintf(stdout, "%s%s%s%s%s%s%s%s%s%s",
				(gattr & G_FNAME) ? ":" : "",
				(gattr & G_FNAME) ? _("fname") : "",
				(gattr & G_ROOM) ? ":" : "",
				(gattr & G_ROOM) ? _("room") : "",
				(gattr & G_WPHONE) ? ":" : "",
				(gattr & G_WPHONE) ? _("wphone") : "",
				(gattr & G_HPHONE) ? ":" : "",
				(gattr & G_HPHONE) ? _("hphone") : "",
				(gattr & G_OTHER) ? ":" : "",
				(gattr & G_OTHER) ? _("other") : "");

			fprintf(stdout, "\n%s", pw->pw_name);

			if ((attr & A_LOCKED) &&
			    ((getuid() == pw->pw_uid) || isuseradmin))
				fprintf(stdout,
					":%s",
					locked ? _("yes") : _("no"));

			if (attr & A_ID)
				fprintf(stdout, ":%d", (int)pw->pw_uid);

			if (attr & A_PGRP)
				fprintf(stdout, ":%s", pgroup);

			if (attr & A_GROUPS)
				fprintf(stdout, ":%s", grplist);

			if ((attr & A_ADMGROUPS) &&
			    ((getuid() == pw->pw_uid) || isuseradmin))
				fprintf(stdout, ":%s", nstr(admgrplist));

			if (attr & A_HOME)
				fprintf(stdout, ":%s", nstr(pw->pw_dir));

			if (attr & A_SHELL)
				fprintf(stdout, ":%s", nstr(pw->pw_shell));

			if (gattr == G_NONE && attr & A_GECOS) {
				fprintf(stdout, ":%s", nstr(pw->pw_gecos));
			} else {
				if (gattr & G_FNAME)
					fprintf(stdout, ":%s", gecosv[0]);

				if (gattr & G_ROOM)
					fprintf(stdout, ":%s", gecosv[1]);

				if (gattr & G_WPHONE)
					fprintf(stdout, ":%s", gecosv[2]);

				if (gattr & G_HPHONE)
					fprintf(stdout, ":%s", gecosv[3]);

				if (gattr & G_OTHER)
					fprintf(stdout, ":%s", gecosv[4]);
			}
		} else {
			fprintf(stdout, "%s:", pw->pw_name);

			if (attr & A_ID)
				fprintf(stdout,
					"\n\t%s=%d",
					_("id"), (int)pw->pw_uid);

			if ((attr & A_LOCKED) &&
			    ((getuid() == pw->pw_uid) || isuseradmin))
				fprintf(stdout,
					"\n\t%s=%s",
					_("locked"),
					locked ? _("yes") : _("no"));

			if (attr & A_PGRP)
				fprintf(stdout,
					"\n\t%s=%s",
					_("pgrp"), pgroup);

			if (attr & A_GROUPS)
				fprintf(stdout,
					"\n\t%s=%s",
					_("groups"), grplist);

			if ((attr & A_ADMGROUPS) &&
			    ((getuid() == pw->pw_uid) || isuseradmin))
				fprintf(stdout,
					"\n\t%s=%s",
					_("admgroups"), nstr(admgrplist));

			if (attr & A_HOME)
				fprintf(stdout,
					"\n\t%s=%s",
					_("home"), nstr(pw->pw_dir));

			if (attr & A_SHELL)
				fprintf(stdout,
					"\n\t%s=%s",
					_("shell"), nstr(pw->pw_shell));

			if (gattr == G_NONE && attr & A_GECOS) {
				fprintf(stdout,
					"\n\t%s=%s",
					_("gecos"), nstr(pw->pw_gecos));
			} else {
				if (gattr & G_FNAME)
					fprintf(stdout,
						"\n\t%s=%s",
						_("fname"), gecosv[0]);

				if (gattr & G_ROOM)
					fprintf(stdout,
						"\n\t%s=%s",
						_("room"), gecosv[1]);

				if (gattr & G_WPHONE)
					fprintf(stdout,
						"\n\t%s=%s",
						_("wphone"), gecosv[2]);

				if (gattr & G_HPHONE)
					fprintf(stdout,
						"\n\t%s=%s",
						_("hphone"), gecosv[3]);

				if (gattr & G_OTHER)
					fprintf(stdout,
						"\n\t%s=%s",
						_("other"), gecosv[4]);
			}
		}

		fprintf(stdout, "\n");

		if (args.stanza)
			fprintf(stdout, "\n");

		strfreev(gecosv);
		free(admgrplist);
		free(grplist);
		free(pgroup);
	}

EXIT:
	/* Free all allocated memory */
	strfreev(usrarray);

	return status;
}
