/**
 * @file sysutils.c
 * Common functions for sysutils
 *
 * Copyright (C) 2002, 2003, 2004 David Weinehall
 * Copyright (C) 2004, 2005, 2006 Free Software Foundation, Inc.
 *
 *  This file is part of GNU Sysutils
 *
 *  GNU Sysutils is free software; you can redistribute it and/or modify
 *  it under the terms of the GNU General Public License as published by
 *  the Free Software Foundation; either version 2 of the License, or
 *  (at your option) any later version.
 *
 *  GNU Sysutils is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU General Public License for more details.
 *
 *  You should have received a copy of the GNU General Public License
 *  along with this program; if not, write to the Free Software
 *  Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
 */
#include "misc.h"		/* Keep this first, to get config.h */

#include <grp.h>
#include <pwd.h>
#include <time.h>
#include <crypt.h>
#include <ctype.h>
#include <errno.h>
#include <stdio.h>
#include <limits.h>
#include <shadow.h>
#include <stdlib.h>
#include <string.h>
#include <unistd.h>
#include <termios.h>
#include <sys/uio.h>
#include <sys/stat.h>
#include <sys/time.h>
#include <sys/types.h>
#ifdef HAVE_UTMPX_H
#include <utmpx.h>
#elif HAVE_UTMP_H
#include <utmp.h>
#endif /* !HAVE_UTMPX_H && !HAVE_UTMP_H */
#ifdef HOST_NAME_MAX
#define HOSTNAMESIZE HOST_NAME_MAX	/**< Size to use for hostname */
#else /* !HOST_NAME_MAX */
#define HOSTNAMESIZE 255		/**< Size to use for hostname */
#endif /* !HOST_NAME_MAX */

#include <gshadow.h>
#include "sysutils.h"

static struct termios saved_attributes;	/**< Saved terminal attributes */
/** Used to store the author information */
static const char *author_information = NULL;
const char *progname;	/**< Used to store the name of the program */

/** Any password, shell, etc shouldn't be more than 32 characters */
#define INPUT_SIZE 32

/** Hack to convert ASCII-representation of a number to an integer */
#define tonum(__number) \
	(__number - '0')

/**
 * Check if the year is a leap year
 *
 * @param __year The year to check
 * @return non-zero if the year is a leap year
 *         zero if the year is not a leap year
 */
#define isleapyear(__year) \
	((!(__year % 4) && (__year % 100)) || !(__year % 400))

/** Number of days per month for non leap years */
static int monthdays[] = {
	31, 28, 31, 30, 31, 30,
	31, 31, 30, 31, 30, 31
};

/**
 * Output programname, packagename, version-number,
 * author-information, copyright information and disclaimer
 *
 * @param file Unused; needed by argp
 * @param state Unused; needed by argp
 */
void version(FILE *file, struct argp_state *state)
{
	/* quench warnings */
	(void)file;
	(void)state;

	fprintf(stdout, "%s (%s) %s\n%s\n%s",
		progname, PACKAGE, VERSION, author_information,
		_("Copyright (C) 2004-2006 Free Software Foundation, Inc.\n"
		  "This is free software; see the source for copying "
		  "conditions.  There is NO\n"
		  "warranty; not even for MERCHANTABILITY or FITNESS FOR "
		  "A PARTICULAR PURPOSE.\n"));
}

/**
 * Setup locales
 *
 * @param name The name of the program to setup locales for
 * @return 0 on success, errno on failure
 */
error_t init_locales(const char *name)
{
	error_t status = 0;

#ifdef ENABLE_NLS
	setlocale(LC_ALL, "");

	if (!bindtextdomain(PACKAGE_NAME, LOCALEDIR) && errno == ENOMEM) {
		status = errno;
		goto EXIT;
	}

	if (!textdomain(PACKAGE_NAME) && errno == ENOMEM) {
		status = errno;
		goto EXIT;
	}

EXIT:
	/* In this error-message we don't use _(), since we don't
	 * know where the locales failed, and we probably won't
	 * get a reasonable result if we try to use them.
	 */
	if (status) {
		fprintf(stderr,
			"%s: `%s' failed; %s\n",
			name, "init_locales", strerror(errno));
	} else {
		errno = 0;
	}
#endif /* ENABLE_NLS */

	progname = name;

	return status;
}

/**
 * Initialise author information
 */
void set_author_information(const char *string)
{
	author_information = string;
}

/**
 * Turn a date in year-month-day format into days after 1970-01-01
 *
 * @param year The year
 * @param month The month
 * @param day The day
 * @return the date as days since 1970-01-01
 */
static long ymd_to_days(const int year, const int month, const int day)
{
	long days = day;
	int i;

	for (i = 1; i < month; i++)
		days += monthdays[i - 1];

	if (month > 2 && isleapyear(year))
		days++;

	for (i = 1970; i < year; i++)
		days += isleapyear(i) ? 366 : 365;

	return days - 1;
}

/**
 * Return maximum user- and group-name length
 *
 * @return Maximum length of user- and group-names in bytes
 */
size_t get_max_namelen(void)
{
	size_t len;

#ifdef HAVE_UTMPX_H
	struct utmpx ut;

	len = sizeof ut.ut_user;
#elif HAVE_UTMP_H
	struct utmp ut;

	len = sizeof ut.ut_user;
#else /* !HAVE_UTMPX_H && !HAVE_UTMP_H */
	len = INPUT_SIZE;
#endif  /* !HAVE_UTMPX_H && !HAVE_UTMP_H */

	return len;
}

/**
 * Return today's date
 *
 * @return Today's date in days since 1970-01-01, or -1 on failure
 */
long get_current_date(void)
{
	static time_t today;
	static struct tm date;
	long retval = -1;

	if (time(&today) == (time_t)-1)
		goto EXIT;

	if (!gmtime_r(&today, &date))
		goto EXIT;

	retval = ymd_to_days(1900 + date.tm_year,
			     date.tm_mon + 1, date.tm_mday);

EXIT:
	if (retval == -1) {
		fprintf(stderr,
			_("%s: `%s' failed; %s\n"),
			progname, "get_current_date", strerror(errno));
	}

	return retval;
}

/**
 * Restore the old terminal attributes for stdin
 *
 * @return 0 on success, errno on failure
 */
static int reset_input_mode(void)
{
	return tcsetattr(STDIN_FILENO, TCSANOW, &saved_attributes);
}

/**
 * Chage terminal attributes for stdin
 *
 * @param echo 1 if the output should be echoed to the screen
 *             0 if not
 * @return 0 on success, errno on failure
 */
static int set_input_mode(const int echo)
{
	struct termios tattr;

	if (!isatty(STDIN_FILENO))
		return ENOTTY;

	if (tcgetattr(STDIN_FILENO, &saved_attributes) == -1)
		return errno;

	if (atexit((void *)reset_input_mode))
		fprintf(stderr,
			_("%s: warning: `%s' failed\n"),
			progname, "atexit()");

	memcpy(&tattr, &saved_attributes, sizeof tattr);

	if (!echo)
		tattr.c_lflag &= ~ECHO;

	return tcsetattr(STDIN_FILENO, TCSAFLUSH, &tattr);
}

/**
 * Accept a string of input from stdin
 *
 * @param echo Should the input be echoed to the screen
 * @return A pointer to the string,
 *         NULL on failure or no input (on failure errno is set)
 */
char *input_string(const int echo)
{
	char input[INPUT_SIZE + 1];
	char *result = NULL;
	int len;

	/* Setup correct input mode for stdin */
	if (set_input_mode(echo))
		goto EXIT;

	/* Read at most INPUT_SIZE characters from stdin */
	if (!(len = read(STDIN_FILENO, input, INPUT_SIZE)) == -1)
		goto EXIT;

	/* Make sure we got some reasonable input */
	if (!len || (len == 1 && input[0] == '\n'))
		goto EXIT;

	if (input[len - 1] == '\n')
		input[len - 1] = 0;

	result = strdup(input);

	/* We need to clean up since we might be inputting passwords */
	memset(input, 0, INPUT_SIZE);

EXIT:
	if (reset_input_mode() && !result) {
		memset(result, 0, strlen(result));
		result = NULL;
	}

	if (!echo)
		fprintf(stdout, "\n");

	return result;
}

/**
 * Input a password and return its cryptographic hash
 *
 * @param salt The salt to use; if salt is NULL, generate a new salt
 * @param check The password to compare with, NULL to skip compare
 * @param match Pointer to value to return result of compare in
 * @return Encrypted password if everything went OK
 *         NULL on failure
 */
char *input_password(const char *salt, const char *check, int *match)
{
	char *newsalt = NULL;
	char *salttmp1 = NULL;
	char *salttmp2 = NULL;
	char *cpassword = NULL;
	char *password = NULL;

	if (!salt) {
		struct timeval tv;

		if (gettimeofday(&tv, (struct timezone *)0) == -1)
			goto EXIT;

		if (!(salttmp1 = long_to_radix64(tv.tv_usec)))
			goto EXIT;

		if (!(salttmp2 = long_to_radix64(tv.tv_sec +
						 getpid() + clock())))
			goto EXIT;

		newsalt = strconcat("$1$", salttmp1, salttmp2, NULL);
	} else {
		newsalt = strdup(salt);
	}

	if (!newsalt)
		goto EXIT;

	if (!(password = input_string(0)))
		goto EXIT;

	if (check && strlen(check) && !strcmp(check, crypt(password, check)))
		*match = 1;

	cpassword = strdup(crypt(password, newsalt));

EXIT:
	/* We need to clean up the password */
	if (password) {
		memset(password, 0, strlen(password));
		free(password);
	}

	free(salttmp2);
	free(salttmp1);
	free(newsalt);

	return cpassword;
}

/**
 * Turn the string into a boolean value
 *
 * @param string The string to check for validity
 * @param[out] value The result of the conversion
 * @return 0 is always returned
 */
error_t string_to_bool(const char *string, void **value)
{
	if (!strcmp(string, "1"))
		*(int *)value = 1;
	else if (!strcasecmp(string, "y"))
		*(int *)value = 1;
	else if (!strcasecmp(string, "yes"))
		*(int *)value = 1;
	else if (!strcasecmp(string, "true"))
		*(int *)value = 1;
	else if (!strcasecmp(string, "on"))
		*(int *)value = 1;
	else if (!strcasecmp(string, "always"))
		*(int *)value = 1;
	else
		*(int *)value = 0;

	return 0;
}

/**
 * Turn the string into a uid_t
 *
 * @param string The string to convert
 * @param[out] value The result of the conversion
 * @return 0 on success
 *         errno on error
 */
error_t string_to_uid_t(const char *string, void **value)
{
	*(uid_t *)value = (uid_t)strtoul(string, NULL, 10);

	return errno;
}

/**
 * Turn the string into a gid_t
 *
 * @param string The string to convert
 * @param[out] value The result of the conversion
 * @return 0 on success
 *         errno on error
 */
error_t string_to_gid_t(const char *string, void **value)
{
	*(gid_t *)value = (gid_t)strtoul(string, NULL, 10);

	return errno;
}

/**
 * Turn the string into a signed long
 *
 * @param string The string to convert
 * @param[out] value The result of the conversion
 * @return 0 on success
 *         errno on error
 */
error_t string_to_long(const char *string, void **value)
{
	*(long *)value = strtol(string, NULL, 10);

	return errno;
}

/**
 * Turn the string into a date
 * Note: no errorcheck occurs; use is_valid_date() to check
 *       for validity first.
 *
 * @param string The string to convert
 * @param[out] value The result of the conversion;
 *             The date is returned as days since 1970-01-01,
 *             -1 if get_current_date fails,
 *             -2 for the default value
 * @return 0 on success
 *         errno on error
 */
error_t string_to_date(const char *string, void **value)
{
	int year, month, day;
	error_t status = 0;

	if (!string || !strlen(string)) {
		*(long *)value = -2;
		goto EXIT;
	}

	if (!strcasecmp(string, "never") || (!strcmp(string, "0"))) {
		*(long *)value = 0;
		goto EXIT;
	}

	if (!strcasecmp(string, "now") || !strcasecmp(string, "today")) {
		*(long *)value = get_current_date();
		status = errno;
		goto EXIT;
	}

	year = tonum(string[0]) * 1000 + tonum(string[1]) * 100 +
	       tonum(string[2]) * 10 + tonum(string[3]);
	month = tonum(string[5]) * 10 + tonum(string[6]);
	day = tonum(string[8]) * 10 + tonum(string[9]);

	*(long *)value = ymd_to_days(year, month, day);

EXIT:
	return status;
}

/**
 * Dummy string converter
 *
 * @param string The string to convert
 * @param[out] value The result of the conversion
 * @return 0 on success
 *         errno on error
 */
error_t string_to_string(const char *string, void **value)
{
	char *tmp = NULL;

	if (!(tmp = strdup(string)))
		return errno;

	if (*value)
		free(*value);

	*value = tmp;

	return 0;
}

/**
 * Turn a date into a string
 *
 * @param days The number of days since 1970-01-01
 * @return A string with the date as YYYY-MM-DD (ISO 8601),
 *         or NULL on failure
 */
char *date_to_string(const long days)
{
	char *str = NULL;
	int year = 1969;
	int month = 0;
	int day;
	long tmp;
	long tmp2 = days;

	if (days == -1 || days == 0) {
		str = strdup("never");
		goto EXIT;
	}

	do {
		year++;
		tmp = tmp2;
		tmp2 = tmp - (isleapyear(year) ? 366 : 365);
	} while (tmp2 >= 0);

	for (month = 0; month < 12; month++) {
		if (isleapyear(year) && month == 2)
			tmp2 = tmp - (monthdays[month] + 1);
		else
			tmp2 = tmp - monthdays[month];

		if (tmp2 < 0)
			break;

		tmp = tmp2;
	}

	day = tmp;

	if (asprintf(&str, "%4d-%02d-%02d",
		     year, month + 1, day + 1) < 0)
		str = NULL;

EXIT:
	if (!str) {
		fprintf(stderr,
			_("%s: `%s' failed; %s\n"),
			progname, "date_to_string", strerror(errno));
	}

	return str;
}

/**
 * Turn a long into a radix64 string
 *
 * @param value The number to convert into a radix64 string
 * @return A radix64-string or NULL on failure
 */
char *long_to_radix64(const long value)
{
	static char buf[8];
	char *str = NULL;
	int tmpvalue = value;
	int i;

	if (value < 0) {
		str = NULL;
	} else {
		for (i = 0; i < 7; i++) {
			int j = tmpvalue % 64;

			if (j < 1)
				buf[i] = '.';
			else if (j == 1)
				buf[i] = '/';
			else if (j < 12)
				buf[i] = '0' - 2 + j;
			else if (j < 38)
				buf[i] = 'A' - 12 + j;
			else if (j < 63)
				buf[i] = 'a' - 38 + j;
			else
				buf[i] = 'z';
			buf[i + 1] = '\0';

			if (!(tmpvalue /= 64))
				break;
		}

		str = strdup(buf);
	}

	if (!str) {
		fprintf(stderr,
			_("%s: `%s' failed; %s\n"),
			progname, "long_to_radix64", strerror(errno));
	}

	return str;
}

/**
 * Check whether the string is a boolean value
 *
 * @param string The string to check for validity
 * @return 0 if the string is a boolean value
 *         EINVAL if the string is not a boolean value
 */
error_t is_bool(const char *string)
{
	if (!string || !strlen(string))
		goto EXIT;

	if (!strcmp(string, "1"))
		return 0;

	if (!strcmp(string, "0"))
		return 0;

	if (!strcasecmp(string, "y"))
		return 0;

	if (!strcasecmp(string, "n"))
		return 0;

	if (!strcasecmp(string, "yes"))
		return 0;

	if (!strcasecmp(string, "no"))
		return 0;

	if (!strcasecmp(string, "true"))
		return 0;

	if (!strcasecmp(string, "false"))
		return 0;

	if (!strcasecmp(string, "on"))
		return 0;

	if (!strcasecmp(string, "off"))
		return 0;

	if (!strcasecmp(string, "always"))
		return 0;

	if (!strcasecmp(string, "never"))
		return 0;

EXIT:
	return EINVAL;
}

/**
 * Check whether the string is a decimal number
 *
 * @param string The string to check for validity
 * @return 0 if the string is a decimal number
 *         EINVAL if the string is not a decimal number
 */
error_t is_decimal(const char *string)
{
	int i;

	if (!string || !strlen(string))
		return EINVAL;

	for (i = 0; string[i]; i++)
		if (!isdigit(string[i]))
			return EINVAL;

	return 0;
}

/**
 * Checks whether the string can be mapped to a uid_t
 *
 *   @param string The string to check for validity
 *  @return 0 if the string can be mapped to a uid_t
 *	    EINVAL if the string contains non-decimal characters
 *	    ERANGE if the string cannot be mapped to a uid_t
 */
error_t is_uid_t(const char *string)
{
	unsigned long tmp;

	if (is_decimal(string))
		return EINVAL;

	tmp = strtoul(string, NULL, 10);

	if (tmp == ULONG_MAX && errno == ERANGE)
		return ERANGE;

	if ((uid_t)tmp != tmp)
		return ERANGE;

	/* 65535 is a reserved id */
	if (tmp == 65535)
		return EINVAL;

	return 0;
}

/**
 * Checks whether the string can be mapped to a gid_t
 *
 *   @param string The string to check for validity
 *  @return 0 if the string can be mapped to a gid_t
 *	    EINVAL if the string contains non-decimal characters
 *	    ERANGE if the string cannot be mapped to a gid_t
 */
error_t is_gid_t(const char *string)
{
	unsigned long tmp;

	if (is_decimal(string))
		return EINVAL;

	tmp = strtoul(string, NULL, 10);

	if (tmp == ULONG_MAX && errno == ERANGE)
		return ERANGE;

	if ((gid_t)tmp != tmp)
		return ERANGE;

	/* 65535 is a reserved id */
	if (tmp == 65535)
		return EINVAL;

	return 0;
}

/**
 * Check whether the string can be mapped to a signed long
 *
 *   @param string The string to check for validity
 *  @return 0 if the string can be mapped to a long
 *	    EINVAL if the string contains non-decimal characters
 *	    ERANGE if the string cannot be mapped to a long
 */
error_t is_long(const char *string)
{
	long tmp;

	if (is_decimal(string))
		return EINVAL;

	tmp = strtol(string, NULL, 10);

	if (((tmp == LONG_MIN) || (tmp == LONG_MAX)) && errno == ERANGE)
		return ERANGE;

	return 0;
}

/**
 * Check whether the string is a valid date
 *
 *   @param string The string to check for validity
 *  @return 0 if the string is a valid date
 *	    EINVAL if the string is in an invalid format
 *	    ERANGE if the string cannot be mapped to a date
 */
error_t is_valid_date(const char *string)
{
	int year, month, day;

	/* An empty string is a special case that is allowed for dates */
	if (!string || !strlen(string))
		return 0;

	/* As is "never" */
	if (!strcasecmp(string, "never"))
		return 0;

	/* And "now" */
	if (!strcasecmp(string, "now"))
		return 0;

	/* And "today" */
	if (!strcasecmp(string, "today"))
		return 0;

	/* 0 is also a legitimate special case */
	if (!strcmp(string, "0"))
		return 0;

	if (!(isdigit(string[0]) && isdigit(string[1]) &&
	    isdigit(string[2]) && isdigit(string[3]) &&
	    string[4] == '-' &&
	    isdigit(string[5]) && isdigit(string[6]) &&
	    string[7] == '-' &&
	    isdigit(string[8]) && isdigit(string[9])))
		return EINVAL;

	year = tonum(string[0]) * 1000 + tonum(string[1]) * 100 +
	       tonum(string[2]) * 10 + tonum(string[3]);
	month = tonum(string[5]) * 10 + tonum(string[6]);
	day = tonum(string[8]) * 10 + tonum(string[9]);

	if (year < 1970)
		return ERANGE;

	if (month < 1 || month > 12 || day < 1)
		return EINVAL;

	if (day > monthdays[month - 1])
		if (day != 29 || month != 2 || !isleapyear(year))
			return EINVAL;

	return 0;
}

/**
 * Check whether the string is an existing path
 *
 *   @param string The string to check for validity
 *  @return 0 if the path exists
 *          errno if the path does not exist
 *          EINVAL if the string is empty
 *          EISDIR if the path exists but is a directory
 */
error_t is_valid_path(const char *string)
{
	struct stat st;

	if (!string || !strlen(string))
		return EINVAL;

	if (stat(string, &st) == -1)
		return errno;

	if (!S_ISDIR(st.st_mode))
		return ENOTDIR;

	return 0;
}

/**
 * Check whether the string is the path to an existing file
 *
 *   @param string The string to check for validity
 *  @return 0 if the filepath is valid
 *          errno if the filepath is not valid
 */
error_t is_valid_filepath(const char *string)
{
	struct stat st;

	/* Check the entire string for invalid characters
	 */
	if (is_valid_string(string, "*?:,="))
		return EINVAL;

	/* The file must exist */
	if (stat(string, &st) == -1)
		return errno;

	/* The file must be a regular file */
	if (!S_ISREG(st.st_mode))
		return EINVAL;

	return 0;
}

/**
 * Check whether the string is the path to an existing file
 *
 * CAVEAT: The path to the shell might not be accessable for the user;
 *         this code does not verify this.
 *
 * @param string The string to check for validity
 * @return 0 if the shell is valid
 *         errno if the shell is not valid
 */
error_t is_valid_shell(const char *string)
{
	struct stat st;

	/* Check the entire string for invalid characters
	 */
	if (is_valid_string(string, "*?:,="))
		return EINVAL;

	/* The file must exist */
	if (stat(string, &st) == -1)
		return errno;

	/* The file must be a regular file */
	if (!S_ISREG(st.st_mode))
		return EINVAL;

	/* The file must be executable */
	if (!(st.st_mode & S_IXOTH))
		return EINVAL;

	return 0;
}

/**
 * Checks whether the shell exists in /etc/shells
 *
 * @param string The string to check for validity
 * @return 0 if the shell is listed
 *         ENOENT if the shell is not listed
 */
error_t is_listed_shell(const char *string)
{
	error_t status = ENOENT;
	char *shell;

	setusershell();

	while ((shell = getusershell())) {
		if (!strcmp(shell, string)) {
			status = 0;
			break;
		}
	}

	endusershell();

	return status;
}

/**
 * Check whether the string contains any invalid characters
 *
 * @param string The string to check for validness
 * @param invalid The chars that are invalid to have in the string
 * @return 0 if the string is printable and contains no invalid characters
 *         EINVAL if the string contains invalid characters
 */
error_t is_valid_string(const char *string, const char *invalid)
{
	int i;

	if (!string || !strlen(string))
		return EINVAL;

	/* Since the manual page for isprint and iscntrl is far
	 * from detailed on whether these ever overlap, both tests
	 * are included just to be sure.  Furthermore, we frown upon
	 * whitespace with the exception of a normal space.
	 */
	for (i = 0; string[i]; i++)
		if (!isprint(string[i]) || iscntrl(string[i]) ||
		    (isspace(string[i]) && string[i] != ' ') ||
		    strchr(invalid, string[i]))
			return EINVAL;

	return 0;
}

/**
 * Make sure that a user- or groupname contains only legal characters
 *
 * @param string The string to check for validness
 * @return 0 if the string is a valid user- or groupname
 *         EINVAL if the string contains invalid characters
 */
error_t is_valid_name(const char *string)
{
	/* A non-existing string is not good, neither is an empty string  */
	if (!string || !strlen(string))
		return EINVAL;

	/* ALL is not allowed */
	if (!strcmp(string, "ALL"))
		return EINVAL;

	/* neither is default */
	if (!strcmp(string, "default"))
		return EINVAL;

	/* no space (is_valid_string checks for other whitespace) */
	if (strchr(string, ' '))
		return EINVAL;

	/* The first character must be [a-zA-Z] */
	if (!isalpha(string[0]))
		return EINVAL;

	/* Verify that the username is not too long */
	if (strlen(string) > get_max_namelen())
		return EINVAL;

	/* Make sure the rest of the string is valid */
	return is_valid_string(string, "%:\"#,=\\/?'`");
}

/**
 * Make sure that a list of user- or groupnames contains only legal characters
 * This function considers empty or non-existing strings to be ok
 * XXX FIXME XXX should use is_valid_name foreach instead
 *
 * @todo Make this function use is_valid_name for each name instead
 * @param string The string to check for validness
 * @return 0 if the string is a valid list of user- or groupnames
 *         EINVAL if the string contains invalid characters
 *         ENOMEM if list-expansion caused out of memory
 */
error_t is_valid_namelist_empty(const char *string)
{
	/* A non-existing string is ok, and so is an empty string */
	if (!string || !strlen(string))
		return 0;

	/* ALL is not allowed */
	if (!strcmp(string, "ALL"))
		return EINVAL;

	/* neither is default */
	if (!strcmp(string, "default"))
		return EINVAL;

	/* no space (is_valid_string checks for other whitespace) */
	if (strchr(string, ' '))
		return EINVAL;

	/* The first character must be [a-zA-Z] */
	if (!isalpha(string[0]))
		return EINVAL;

	/* The list may not begin or end with ',' or contain empty entries */
	if (*string == ',' || strstr(string, ",,") ||
	    *(string + strlen(string) - 1) == ',')
		return EINVAL;

	/* Make sure the rest of the string is valid */
	return is_valid_string(string, "%:\"#=\\/?'`");
}

/**
 * Make sure that a list of user- or groupnames contains only legal characters
 *
 * @param string The string to check for validness
 * @return 0 if the string is a valid list of user- or groupnames
 *         EINVAL if the string contains invalid characters
 *         ENOMEM if list-expansion caused out of memory
 */
error_t is_valid_namelist(const char *string)
{
	/* A non-existing string is not a good name */
	if (!string || !strlen(string))
		return EINVAL;

	return is_valid_namelist_empty(string);
}

/**
 * Check whether a gecos-field contains any invalid characters
 *
 * @param string The string to check for validity
 * @return 0 if the field is printable and contains no invalid characters
 *         EINVAL if the field contains invalid characters or is unprintable
 */
error_t is_valid_gecos_field(const char *string)
{
	return is_valid_string(string, "\\%:=,");
}

/**
 * Check whether a gecos other-field contains any invalid characters
 *
 * @param string The string to check for validity
 * @return 0 if the field is printable and contains no invalid characters
 *         EINVAL if the field contains invalid characters or is unprintable
 */
error_t is_valid_gecos_other(const char *string)
{
	return is_valid_string(string, "\\%:");
}

/**
 * Check whether a gecos string contains any invalid characters
 *
 * @param string The string to check for validity
 * @return 0 if the string is printable and contains no invalid characters
 *         EINVAL if the string contains invalid characters or is unprintable
 *         errno on failure
 */
error_t is_valid_gecos(const char *string)
{
	char **tmp = NULL;
	error_t status = 0;

	if (!(tmp = split_gecos(string))) {
		status = errno;
		goto EXIT;
	}

	if (is_valid_gecos_field(tmp[0]) || is_valid_gecos_field(tmp[1]) ||
	    is_valid_gecos_field(tmp[2]) || is_valid_gecos_field(tmp[3]) ||
	    is_valid_gecos_other(tmp[4]))
		status = EINVAL;

EXIT:
	strfreev(tmp);

	return status;
}

/**
 * Parse an attribute=value pair
 *
 * @param string The string to parse
 * @param attrs A structure with all valid attributes
 * @return 0 if the attribute exists and the value is valid
 *         EINVAL if the syntax is invalid
 *         ENOENT if the attribute does not exist
 *         errno if one of the validators or converters fails
 */
error_t parse_key_pairs(char *string, struct attr attrs[])
{
	error_t status = ENOENT;
	char *value = NULL;
	size_t attrlen;
	char *p;
	int i;

	/* NULL is not a good string,  neither is a string of zero length */
	if (!string || !strlen(string)) {
		fprintf(stderr,
			_("%s: aiiik, this should never "
			  "happen. Alert the programmer!\n"),
			progname);
		status = EINVAL;
		goto EXIT;
	}

	/* Invalid in attribute=value pairs:
	 * o    Lack of '='
	 * o    Leading '='
	 * o    Multiple '='
	 */
	if (!(p = strchr(string, '=')) || p == string ||
	    p != strrchr(string, '=')) {
		fprintf(stderr,
			_("%s: invalid syntax\n"
			  "Try `%s --help' for more information\n"),
			progname, progname);
		status = EINVAL;
		goto EXIT;
	}

	attrlen = p - string;

	/* Compare the potential attribute to all known attributes */
	for (i = 0; attrs[i].attribute; i++) {
		if (!strncmp(string, attrs[i].attribute, attrlen) &&
		    strlen(attrs[i].attribute) == attrlen) {
			value = p + 1;

			if ((status = attrs[i].validator(value))) {
				fprintf(stderr,
					_("%s: invalid value supplied to "
					  "`%s'\n"),
					progname, attrs[i].attribute);
				goto EXIT;
			}

			if ((status = attrs[i].converter(value,
							 attrs[i].value))) {
				fprintf(stderr,
					_("%s: `%s' failed; %s\n"),
					progname, "parse_key_pairs",
					strerror(status));
				goto EXIT;
			}
		}
	}

EXIT:
	return status;
}

/**
 * Create a new filename by merging a filename with an extension
 *
 * @param filename A string with the filename
 * @param extension A string with the extension
 * @return A new string on success
 *         NULL on failure (errno is set)
 */
char *create_filename(const char *filename, const char *extension)
{
	char *newstring = NULL;

	if (!(newstring = strconcat(filename, extension, NULL)))
		fprintf(stderr,
			_("%s: `%s' failed; %s\n"),
			progname, "create_filename", strerror(errno));

	return newstring;
}

/**
 * Open a file
 *
 * @param filename The file to open
 * @param mode The mode flags for the open
 * @return A file-pointer on success,
 *         NULL on failure (errno is set)
 */
FILE *open_file(const char *filename, const char *mode)
{
	FILE *newfp = NULL;

	if (!(newfp = fopen(filename, mode)))
		fprintf(stderr,
			_("%s: `%s' failed; %s\n"),
			progname, "fopen()", strerror(errno));

	return newfp;
}

/**
 * Close a file
 *
 * @param fp The file to close
 * @return 0 on success
 *         errno on failure
 */
error_t close_file(FILE **fp)
{
	if (fclose(*fp))
		fprintf(stderr,
			_("%s: `%s' failed; %s\n"),
			progname, "fclose()", strerror(errno));

	*fp = NULL;

	return errno;
}

/**
 * Link source to target, replacing target
 *
 * @param source A string with the name of the file to backup
 * @param target A string with the name of the file to backup to
 * @return 0 on success, errno on failure
 */
error_t backup_file(const char *source, const char *target)
{
	error_t status = 0;

	if (unlink(target) == -1 && errno != ENOENT) {
		status = errno;
		goto EXIT;
	}

	errno = 0;

	if (link(source, target) == -1)
		status = errno;

EXIT:
	if (status)
		fprintf(stderr,
			_("%s: failed to backup `%s' to `%s'; %s\n"),
			progname, source, target, strerror(errno));

	return status;
}

/**
 * Copy source to target, replacing target
 *
 * @param source A string with the name of the file to copy
 * @param target A string with the name of the file to copy to
 * @return 0 on success, errno on failure
 */
error_t copy_file(const char *source, const char *target)
{
	FILE *rfp = NULL;
	FILE *wfp = NULL;
	mode_t oldmask;
	error_t status = 0;
	int c;

	if (unlink(target) == -1 && errno != ENOENT) {
		status = errno;
		goto EXIT;
	}

	errno = 0;

	if (!(rfp = fopen(source, "r"))) {
		status = errno;
		goto EXIT;
	}

	oldmask = umask(0077);
	wfp = fopen(target, "w");
	umask(oldmask);

	if (!wfp) {
		status = errno;
		fclose(rfp);
		goto EXIT;
	}

	while ((c = fgetc(rfp)) != EOF)
		if ((fputc(c, wfp)) == EOF)
			break;

	if (fclose(wfp) == EOF) {
		status = errno;
		goto EXIT;
	}

	if (fclose(rfp) == EOF)
		status = errno;

EXIT:
	if (status)
		fprintf(stderr,
			_("%s: failed to copy `%s' to `%s'; %s\n"),
			progname, source, target, strerror(errno));

	return status;
}

/**
 * Copy filemodes from source to target
 *
 * @param source A string with the name of the file to copy modes from
 * @param target A string with the name of the file to copy modes to
 * @return 0 on success, errno on failure
 */
error_t copy_file_modes(const char *source, const char *target)
{
	struct stat st;
	error_t status = 0;

	if (lstat(source, &st) == -1) {
		status = errno;
		goto EXIT;
	}

	if (chmod(target, st.st_mode) == -1) {
		status = errno;
		goto EXIT;
	}

	if (lchown(target, st.st_uid, st.st_gid) == -1)
		status = errno;

EXIT:
	if (status)
		fprintf(stderr,
			_("%s: failed to copy file-permissions for `%s' to "
			  "`%s'; %s\n"),
			progname, source, target, strerror(errno));

	return status;
}

/**
 * Set file owner and permissions
 *
 * @param file A string with the name of the file to set perms
 * @param user A string with the file owner name
 * @param group A string with the group owner name
 * @param mode A string with the new mode for the file
 * @return 0 on success, errno on failure
 */
error_t set_file_perms(const char *file, const char *user,
		       const char *group, const mode_t mode)
{
	error_t status = 0;
	struct stat st;
	struct passwd *pw;
	struct group *gr;

	if (stat(file, &st) == -1) {
		status = errno;
		goto EXIT;
	}

	if (!(pw = getpwnam(user))) {
		status = errno;
		goto EXIT;
	}


	if (!(gr = getgrnam(group))) {
		status = errno;
		goto EXIT;
	}

	if (chown(file, pw->pw_uid, gr->gr_gid) == -1) {
		status = errno;
		goto EXIT;
	}

	if (chmod(file, mode) == -1) {
		status = errno;
		goto EXIT;
	}

EXIT:
	if (status)
		fprintf(stderr,
			_("%s: failed to set permissions for `%s'; %s\n"),
			progname, file, strerror(errno));

	return status;
}

/**
 * Replace target with source, atomically
 *
 * @param source A string with the name of the file to copy from
 * @param target A string with the name of the file to copy to
 * @return 0 on success, errno on failure
 */
error_t replace_file(const char *source, const char *target)
{
	error_t status = 0;

	if (rename(source, target) == -1) {
		fprintf(stderr,
			_("%s: failed to replace `%s' with `%s'; %s\n"),
			progname, target, source, strerror(errno));
		status = errno;
	}

	return status;
}

/**
 * Unlink a file with preserved error status
 *
 * @param filename The name of the file to unlink
 * @param status The error value to preserve
 * @return 0 on success, errno on failure
 */
error_t unlink_file(const char *filename, error_t status)
{
	if (!filename)
		return status;

	errno = 0;

	if (unlink(filename) == -1) {
		if (errno == ENOENT) {
			errno = 0;
		} else {
			fprintf(stderr,
				_("%s: `%s' failed; %s\n"),
				progname, "unlink_file", strerror(errno));
			if (!status)
				status = errno;
		}
	}

	errno = status;

	return status;
}

/**
 * Create a home-directory for the user
 * CAVEATS: Leading components are not created
 * Prerequisites:
 *          The user must have been created and the user-entry
 *          written to /etc/passwd
 *
 * @param pw The user's pw-entry
 * @param dirflags Flags specifying certain directives when creating
 *	           the directory:
 *		   DF_GROUPHOMES
 *		   DF_LETTERHOMES
 *		   DF_EXISTSOK
 * @param homeumask The umask to use when creating the directory
 * @param homepath The path to the home-directories; usually /home
 * @return The path to the directory on success,
 *         NULL if we fail
 *         errno = ENOENT if homepath is not set or does not exist
 */
char *create_home_directory(const struct passwd *pw, const int dirflags,
			    const mode_t homeumask, const char *homepath)
{
	mode_t tumask = homeumask;
	char letterstr[2] = " ";
	struct group *gr = NULL;
	char *newpath = NULL;
	char *path = NULL;
	char *tmp = NULL;
	struct stat st;

	/* Does homepath exist? */
	if (stat(homepath, &st) == -1) {
		errno = EEXIST;
		goto EXIT;
	}

	/* Is homepath a directory? */
	if (!S_ISDIR(st.st_mode)) {
		errno = ENOTDIR;
		goto EXIT;
	}

	/* Are we allowed access to seek, read, and write in the directory? */
	if (access(homepath, R_OK | W_OK | X_OK))
		goto EXIT;

	if (!(path = strdup(homepath)))
		goto EXIT;

	if (!(gr = getgrgid(pw->pw_gid)))
		goto EXIT;

	if (dirflags & DF_GROUPHOMES) {
		if (!(tmp = strconcat(path, "/", gr->gr_name, NULL)))
			goto EXIT;

		free(path);
		path = tmp;

		/* If the group-home already exists, just ignore */
		if (mkdir(path, 0) == -1 && errno != EEXIST)
			goto EXIT;

		errno = 0;

		/* The group-directory should have the same
		 * owner and group as the parent directory
		 */
		if (chown(path, st.st_uid, st.st_gid) == -1)
			goto EXIT;

		/* Ditto for flags */
		if (chmod(path, st.st_mode) == -1)
			goto EXIT;
	}

	if (dirflags & DF_LETTERHOMES) {
		letterstr[0] = pw->pw_name[0];
		letterstr[1] = 0;

		if (!(tmp = strconcat(path, "/", letterstr, NULL)))
			goto EXIT;

		free(path);
		path = tmp;

		/* If the letter-home already exists, just ignore */
		if (mkdir(path, 0) == -1 && errno != EEXIST)
			goto EXIT;

		errno = 0;

		/* The letter-home should have the same
		 * owner and group as the parent directory
		 */
		if (chown(path, st.st_uid, st.st_gid) == -1)
			goto EXIT;

		/* Ditto for flags */
		if (chmod(path, st.st_mode) == -1)
			goto EXIT;
	}

	if (!(tmp = strconcat(path, "/", pw->pw_name, NULL)))
		goto EXIT;

	free(path);
	path = tmp;

	/* Create the home-directory */
	if (mkdir(path, 0) == -1) {
		if (errno != EEXIST) {
			goto EXIT;
		} else if (!(dirflags & DF_EXISTSOK)) {
			goto EXIT;
		} else {
			errno = 0;
			goto EXIT2;
		}
	}

	/* The new directory should be owned by pw->pw_uid:pw->pw_gid */
	if (chown(path, pw->pw_uid, pw->pw_gid) == -1)
		goto EXIT;

	/* If we don't have usergroups, don't add g+s */
	if (!(dirflags & DF_USERGROUPS))
		tumask |= 02000;

	if (chmod(path, 02777 & ~tumask) == -1)
		goto EXIT;

EXIT2:
	newpath = strdup(path);

EXIT:
	free(path);

	return newpath;
}

/**
 * Write a passwd entry to the specified file
 *
 * Replacement of the broken putpwent(3) in GNU libc, to preserve
 * NIS compat entries.
 *
 * @param pw The passwd entry to write to the specified file
 * @param target The file to write the passwd entry to
 * @return 0 on success, errno on failure
 */
int fputpwent(const struct passwd *pw, FILE *target)
{
	int status;

	if (pw->pw_name[0] != '+') {
		status = fprintf(target,
				 "%s:%s:%u:%u:%s:%s:%s\n",
				 nstr(pw->pw_name),
				 nstr(pw->pw_passwd),
				 pw->pw_uid,
				 pw->pw_gid,
				 nstr(pw->pw_gecos),
				 nstr(pw->pw_dir),
				 nstr(pw->pw_shell));
	} else {
		status = fprintf(target,
				 "%s:%s:::%s:%s:%s\n",
				 nstr(pw->pw_name),
				 nstr(pw->pw_passwd),
				 nstr(pw->pw_gecos),
				 nstr(pw->pw_dir),
				 nstr(pw->pw_shell));
	}

	return status < 0 ? errno : 0;
}

/**
 * Write a shadow entry to the specified file
 *
 * @param sp The shadow entry to write to the specified file
 * @param target The file to write the shadow entry to
 * @return 0 on success, errno on failure
 */
int fputspent(const struct spwd *sp, FILE *target)
{
	if (fprintf(target,
		    "%s:%s:",
		    nstr(sp->sp_namp), nstr(sp->sp_pwdp)) < 0)
		return errno;

	sputlval(target, sp->sp_lstchg, ":");
	sputlval(target, sp->sp_min, ":");
	sputlval(target, sp->sp_max, ":");
	sputlval(target, sp->sp_warn, ":");
	sputlval(target, sp->sp_inact, ":");
	sputlval(target, sp->sp_expire, ":");
	sputulval(target, sp->sp_flag, "\n");

	return 0;
}

/**
 * Write a group entry to the specified file
 *
 * @param gr The group entry to write to the specified file
 * @param target The file to write the group entry to
 * @return 0 on success, errno on failure
 */
int fputgrent(const struct group *gr, FILE *target)
{
	uid_t i;
	int status;

	status = fprintf(target,
			 "%s:%s:%u:",
			 nstr(gr->gr_name),
			 nstr(gr->gr_passwd),
			 gr->gr_gid);

	if (status < 0)
		return errno;

	for (i = 0; gr->gr_mem && gr->gr_mem[i]; i++) {
		status = fprintf(target,
				 i ? ",%s" : "%s",
				 gr->gr_mem[i]);

		if (status < 0)
			return errno;
	}

	return fprintf(target, "\n") < 0 ? errno : 0;
}

/**
 * Write a gshadow entry to the specified file
 *
 * @param sg The shadow entry to write to the specified file
 * @param target The file to write the gshadow entry to
 * @return 0 on success, errno on failure
 */
int fputsgent(const struct sgrp *sg, FILE *target)
{
	uid_t i;
	int status;

	status = fprintf(target,
			 "%s:%s:",
			 nstr(sg->sg_name),
			 nstr(sg->sg_passwd));

	if (status < 0)
		return errno;

	for (i = 0; sg->sg_adm && sg->sg_adm[i]; i++) {
		status = fprintf(target,
				 i ? ",%s" : "%s",
				 sg->sg_adm[i]);

		if (status < 0)
			return errno;
	}

	if (fprintf(target, ":") < 0)
		return errno;

	for (i = 0; sg->sg_mem && sg->sg_mem[i]; i++) {
		status = fprintf(target,
				 i ? ",%s" : "%s",
				 sg->sg_mem[i]);

		if (status < 0)
			return errno;
	}

	return fprintf(target, "\n") < 0 ? errno : 0;
}

/**
 * Lock /etc/passwd, /etc/shadow, /etc/group and /etc/gshadow
 *
 * @return 0 on success, errno on failure
 */
error_t lock_files(void)
{
	error_t status = 0;

	if (lckpwdf() && errno) {
		fprintf(stderr,
			_("%s: `%s' failed; %s\n"),
			progname, "lock_files", strerror(errno));

		status = errno;
	}

	return status;
}

/**
 * Unlock /etc/passwd, /etc/shadow, /etc/group and /etc/gshadow
 *
 * @param status error status before call to unlock_files
 * @return 0 on success, errno on failure
 */
error_t unlock_files(error_t status)
{
	errno = 0;

	if (ulckpwdf() && errno) {
		fprintf(stderr,
			_("%s: `%s' failed; %s\n"),
			progname, "unlock_files", strerror(errno));

		if (!status)
			status = errno;
	}

	return status;
}

/**
 * Return a string with the name of the user with the id uid
 *
 * @param uid The uid of the user to get the name for
 * @return A username, or NULL on failure
 */
char *get_username(const uid_t uid)
{
	struct passwd *pw;
	char *username;

	if (!(pw = getpwuid(uid)) || !pw->pw_name)
		username = NULL;
	else
		username = strdup(pw->pw_name);

	if (!username) {
		fprintf(stderr,
			_("%s: `%s' failed; %s\n"),
			progname, "get_username", strerror(errno));
	}

	return username;
}

/**
 * Return a string with the name of the group with the id gid
 *
 * @param gid The gid of the group to get the name for
 * @return A group-name, or NULL on failure
 */
char *get_groupname(const gid_t gid)
{
	struct group *gr;
	char *groupname;

	if (!(gr = getgrgid(gid)) || !gr->gr_name)
		groupname = NULL;
	else
		groupname = strdup(gr->gr_name);

	if (!groupname) {
		if (!errno || errno == ENOENT) {
			errno = 0;
		} else {
			fprintf(stderr,
				_("%s: `%s' failed; %s\n"),
				progname, "get_groupname", strerror(errno));
		}
	}

	return groupname;
}

/**
 * Return a string with the name of every user on the system
 *
 * @return A comma-separated string on success, NULL on failure
 */
char *get_all_users(void)
{
	char *users = NULL;
	struct passwd *pw;
	int empty = 1;

	setpwent();

	while ((pw = getpwent())) {
		empty = 0;

		/* Paranoia! */
		if (!strlen(pw->pw_name))
			continue;

		if (!users) {
			users = strdup(pw->pw_name);
		} else {
			char *tmp = strconcat(users, ",", pw->pw_name, NULL);
			free(users);
			users = tmp;
		}

		/* strdup or strconcat failed with ENOMEM */
		if (!users)
			break;
	}

	/* Workaround for silly bug in older glibc */
	if (errno && (errno != ENOENT || empty)) {
		free(users);
		users = NULL;
	} else {
		errno = 0;
		endpwent();

		if (!users)
			users = strdup("");
	}

	/* Having it here also catches a failed strdup */
	if (!users) {
		fprintf(stderr,
			_("%s: `%s' failed; %s\n"),
			progname, "get_all_users", strerror(errno));
	}

	return users;
}

/**
 * Return an array with the uid of every user on the system
 *
 * @param nusers A pointer to return the array-size through
 * @return An array of uids on success, NULL on failure
 */
uid_t *get_all_uids(uid_t *nusers)
{
	uid_t *uids = NULL;
	struct passwd *pw;
	unsigned int size = 0;

	setpwent();

	*nusers = 0;

	while ((pw = getpwent())) {
		/* Paranoia! */
		if (!strlen(pw->pw_name))
			continue;

		if (*nusers == size) {
			/* Allocate uids in chunks of 1024; most systems
			 * don't have that many users, and even if they
			 * do, the overhead won't be that big for the
			 * few reallocs that will be performed
			 */
			size += 64;

			/* Allocate one extra to allow for the '\0' */
			uids = realloc(uids, size * sizeof (uid_t));
		}

		/* realloc failed with ENOMEM */
		if (!uids)
			break;

		uids[(*nusers)++] = pw->pw_uid;
	}

	/* Workaround for silly bug in older glibc */
	if (errno && (errno != ENOENT || *nusers)) {
		if (uids)
			free(uids);

		uids = NULL;
	} else {
		errno = 0;
		endpwent();

		if (!uids)
			uids = NULL;
	}

	/* Report the error */
	if (!uids) {
		fprintf(stderr,
			_("%s: `%s' failed; %s\n"),
			progname, "get_all_uids", strerror(errno));
	}

	return uids;
}

/**
 * Return a string with the name of every group on the system
 *
 * @return A comma-separated string on success, NULL on failure
 */
char *get_all_groups(void)
{
	char *groups = NULL;
	struct group *gr;
	int empty = 1;

	setgrent();

	while ((gr = getgrent())) {
		empty = 0;

		/* Paranoia! */
		if (!strlen(gr->gr_name))
			continue;

		if (!groups) {
			groups = strdup(gr->gr_name);
		} else {
			char *tmp = strconcat(groups, ",", gr->gr_name, NULL);
			free(groups);
			groups = tmp;
		}

		/* strdup or strconcat failed with ENOMEM */
		if (!groups)
			break;
	}

	/* Workaround for silly bug in older glibc */
	if (errno && (errno != ENOENT || empty)) {
		free(groups);
		groups = NULL;
	} else {
		errno = 0;
		endpwent();

		if (!groups)
			groups = strdup("");
	}

	/* Having it here also catches a failed strdup */
	if (!groups) {
		fprintf(stderr,
			_("%s: `%s' failed; %s\n"),
			progname, "get_all_groups", strerror(errno));
	}

	return groups;
}

/**
 * Return a comma-separated string with the name of each group the user
 * is a member of
 *
 * @param username The name of the user
 * @return A string with groupnames, separated by commas, or NULL on failure
 */
char *get_groups(const char *username)
{
	char *groups = NULL;
	struct group *gr;
	int empty = 1;

	/* XXX All functions from here on are un-audited */

	setgrent();

	while ((gr = getgrent())) {
		empty = 0;

		/* Paranoia! */
		if (!strlen(gr->gr_name))
			continue;

		if (gr->gr_mem) {
			uid_t i;

			for (i = 0; gr->gr_mem[i]; i++)
				if (!strcmp(gr->gr_mem[i], username))
					break;

			if (gr->gr_mem[i]) {
				if (!groups) {
					groups = strdup(gr->gr_name);
				} else {
					char *tmp = strconcat(groups, ",",
							      gr->gr_name,
							      NULL);
					free(groups);
					groups = tmp;
				}

				/* strdup or strconcat failed with ENOMEM */
				if (!groups)
					break;
			}
		}
	}

	/* Workaround for silly bug in older glibc */
	if (errno && (errno != ENOENT || empty)) {
		free(groups);
		groups = NULL;
	} else {
		errno = 0;
		endpwent();

		if (!groups)
			groups = strdup("");
	}

	/* Having it here also catches a failed strdup */
	if (!groups) {
		fprintf(stderr,
			_("%s: `%s' failed; %s\n"),
			progname, "get_groups", strerror(errno));
	}

	return groups;
}

/**
 * Return a comma-separated string with the name of each group the user
 * is an administrator for
 *
 * @param username The name of the user
 * @return A string with groupnames, separated by commas, or NULL on failure
 */
char *get_admgroups(const char *username)
{
	char *admgroups = NULL;
	struct sgrp *sg;
	int empty = 1;

	setsgent();

	while ((sg = getsgent())) {
		empty = 0;

		/* Paranoia! */
		if (!strlen(sg->sg_name))
			continue;

		if (sg->sg_adm) {
			uid_t i;

			for (i = 0; sg->sg_adm[i]; i++)
				if (!strcmp(sg->sg_adm[i], username))
					break;

			if (sg->sg_adm[i]) {
				if (!admgroups) {
					admgroups = strdup(sg->sg_name);
				} else {
					char *tmp = strconcat(admgroups, ",",
							      sg->sg_name,
							      NULL);
					free(admgroups);
					admgroups = tmp;
				}

				/* strdup or strconcat failed with ENOMEM */
				if (!admgroups)
					break;
			}
		}
	}

	/* Workaround for silly bug in older glibc */
	if (errno && (errno != ENOENT || empty)) {
		free(admgroups);
		admgroups = NULL;
	} else {
		errno = 0;
		endpwent();

		if (!admgroups)
			admgroups = strdup("");
	}

	/* Having it here also catches a failed strdup */
	if (!admgroups && errno != ENOENT) {
		fprintf(stderr,
			_("%s: `%s' failed; %s\n"),
			progname, "get_admgroups", strerror(errno));
	}

	return admgroups;
}

/**
 * Return a comma-separated string with the name of each user
 * that is a primary member of the group
 *
 * @param gr A pointer to a group struct
 * @return A string with usernames, separated by commas, or NULL on failure
 */
char *get_group_primary_members(const struct group *gr)
{
	char *members = NULL;
	struct passwd *pw;
	int empty = 1;

	setpwent();

	while ((pw = getpwent())) {
		empty = 0;

		/* Paranoia! */
		if (!strlen(pw->pw_name))
			continue;

		if (pw->pw_gid == gr->gr_gid) {
			if (!members) {
				members = strdup(pw->pw_name);
			} else {
				char *tmp = strconcat(members, ",",
						      pw->pw_name, NULL);
				free(members);
				members = tmp;
			}

			/* strdup or strconcat failed with ENOMEM */
			if (!members)
				break;
		}
	}

	/* Workaround for silly bug in older glibc */
	if (errno && (errno != ENOENT || empty)) {
		free(members);
		members = NULL;
	} else {
		errno = 0;
		endpwent();

		if (!members)
			members = strdup("");
	}

	/* Having it here also catches a failed strdup */
	if (!members) {
		fprintf(stderr,
			_("%s: `%s' failed; %s\n"),
			progname, "get_group_primary_members", strerror(errno));
	}

	return members;
}

/**
 * Return a comma-separated string with the name of each user
 * that is a member of the group
 *
 * @param gr A pointer to a group struct
 * @return A string with usernames, separated by commas, or NULL on failure
 */
char *get_group_members(const struct group *gr)
{
	char *members = NULL;
	char **tmp = NULL;
	uid_t i, j;

	if (!(members = get_group_primary_members(gr)))
		return NULL;

	if (!(tmp = strsplit(members, ",", 0)))
		goto EXIT;

	/* Note: while the intersection between gr->gr_mem and tmp is
	 * only counted once per entry, it is still possible to get
	 * duplicates if gr->gr_mem contains several occurences of
	 * the same user (or indeed if /etc/passwd holds several users with
	 * the same username, something that should be regarded as a
	 * serious problem...) This is something for pwck/grpck to fix though,
	 * not this function.
	 */
	for (i = 0; gr->gr_mem[i]; i++) {
		for (j = 0; tmp[j]; j++) {
			char *tmp2;

			if (!strcmp(gr->gr_mem[i], tmp[j]))
				continue;

			/* Only add a ',' if the string has entries */
			if (!strlen(members))
				tmp2 = strdup(gr->gr_mem[i]);
			else
				tmp2 = strconcat(members, ",",
						 gr->gr_mem[i], NULL);
			free(members);
			members = tmp2;
			break;
		}

		/* strdup or strconcat failed with ENOMEM */
		if (!members)
			break;
	}

	strfreev(tmp);

EXIT:
	/* Having it here also catches a failed strdup */
	if (!members) {
		fprintf(stderr,
			_("%s: `%s' failed; %s\n"),
			progname, "get_group_members", strerror(errno));
	}

	return members;
}

/**
 * Return a comma-separated string with the name of each user
 * that is an administrator for the group
 *
 * @param groupname The name of the group
 * @return A string with usernames, separated by commas, or NULL on failure
 */
char *get_group_admins(const char *groupname)
{
	char *adms = NULL;
	struct sgrp *sg;

	if (!(sg = getsgnam(groupname)) ||
	    !(adms = strjoinve(",", sg->sg_adm))) {
		if (!sg && !errno)
			errno = ENOENT;
		else
			fprintf(stderr,
				_("%s: `%s' failed; %s\n"),
				progname, "get_group_admins", strerror(errno));
	}

	return adms;
}

/**
 * Return a free user id
 *
 * @param firstuid The beginning of the range to search
 * @param lastuid The end of the range to search
 * @return A free uid, or 65535 on failure
 */
uid_t get_free_uid(uid_t firstuid, uid_t lastuid)
{
	uid_t i;

	for (i = firstuid; i <= lastuid; i++) {
		/* 65535 is special */
		if (i == 65535)
			continue;

		if (!getpwuid(i))
			break;
	}

	if (errno && errno != ENOENT)
		return 65535;

	errno = 0;

	return i > lastuid ? 65535 : i;
}

/**
 * Return a free group id
 *
 * @param firstgid The beginning of the range to search
 * @param lastgid The end of the range to search
 * @return A free gid, or 65535 on failure
 */
gid_t get_free_gid(gid_t firstgid, gid_t lastgid)
{
	gid_t i;

	for (i = firstgid; i <= lastgid; i++) {
		/* 65535 is special */
		if (i == 65535)
			continue;

		if (!getgrgid(i))
			break;
	}

	if (errno && errno != ENOENT)
		return 65535;

	errno = 0;

	return i > lastgid ? 65535 : i;
}

/**
 * Return the hostname
 *
 * @return The hostname on success,
 *         NULL on failure
 */
char *get_host_name(void)
{
	static char buf[HOSTNAMESIZE + 1];

	if (gethostname(buf, HOSTNAMESIZE) == -1)
		return NULL;

	buf[255] = '\0';

	return strdup(buf);
}

/**
 * Return the password lock status for a password
 *
 * @param password The password
 * @return 1 if the password is locked
 *         0 if not
 */
int is_password_locked(const char *password)
{
	if (password[0] == '!' || strlen(password) == 1)
		return 1;
	else
		return 0;
}

/**
 * Return the password lock status for a user
 *
 * @param username The name of the user
 * @return -1 on failure
 *          1 if the password is locked
 *          0 if not
 */
int is_user_locked(const char *username)
{
	struct spwd *sp;

	if (!(sp = getspnam(username))) {
		if (!errno)
			errno = ENOENT;
		else
			fprintf(stderr,
				_("%s: `%s' failed; %s\n"),
				progname, "is_user_locked", strerror(errno));

		return -1;
	} else {
		return is_password_locked(sp->sp_pwdp);
	}
}

/**
 * Return the password lock status for a group
 *
 * @param groupname The name of the group
 * @return -1 on failure
 *          1 if the password is locked
 *          0 if not
 */
int is_group_locked(const char *groupname)
{
	struct sgrp *sg;

	if (!(sg = getsgnam(groupname))) {
		if (!errno)
			errno = ENOENT;
		else
			fprintf(stderr,
				_("%s: `%s' failed; %s\n"),
				progname, "is_group_locked", strerror(errno));

		return -1;
	} else {
		return is_password_locked(sg->sg_passwd);
	}
}

/**
 * Is the program run by root
 *
 * @return 0 if program is run by a useradmin
 *         EPERM otherwise
 */
error_t is_root(void)
{
	return getuid() ? EPERM : 0;
}

/**
 * Is the program run by a useradmin
 * For now this does the same thing as is_root, but this might change
 * in the future.
 *
 * @return 0 if program is run by a useradmin
 *         EPERM otherwise
 */
error_t is_useradmin(void)
{
	return getuid() ? EPERM : 0;
}

/**
 * Is the program run by a groupadmin
 *
 * @param groupname The name of the group to check against
 * @return errno on failure
 *         0 if program is run by a groupadmin
 *         EPERM otherwise
 *         errno on failure
 */
error_t is_groupadmin(const char *groupname)
{
	struct sgrp *sg = NULL;
	char *username = NULL;
	error_t status = EPERM;
	uid_t uid;

	if (!(uid = getuid())) {
		status = 0;
		goto EXIT;
	}

	if (!(username = get_username(uid)))
		goto EXIT;

	if (!(sg = getsgnam(groupname))) {
		if (!errno)
			errno = ENOENT;

		goto EXIT;
	}

	if (is_in_array(sg->sg_adm, username))
		status = 0;

EXIT:
	if (errno)
		status = errno;

	free(username);

	return status;
}

/**
 * Is the caller of the program the same as the specified name
 *
 * @param name A username
 * @return -1 on failure
 *          1 if the caller of the program == name
 *          0 if not
 */
int is_caller(const char *name)
{
	uid_t uid = getuid();
	char *username;
	int status = 0;

	/* Get the name of the user from the uid */
	if (!(username = get_username(uid)))
		return -1;

	status = strcmp(username, name) ? 0 : 1;

	free(username);

	return status;
}

/**
 * Is the specified uid unused
 *
 * @param uid An uid
 * @return 0 the uid is unused
 *         EEXIST if the uid is used
 *         errno on failure
 */
error_t is_free_uid(const uid_t uid)
{
	if (!getpwuid(uid)) {
		if (errno && errno != ENOENT) {
			return errno;
		} else {
			errno = 0;
			return 0;
		}
	}

	return EEXIST;
}

/**
 * Is the specified gid unused
 *
 * @param gid A gid
 * @return 0 the gid is unused
 *         EEXIST if the gid is used
 *         errno on failure
 */
error_t is_free_gid(const gid_t gid)
{
	if (!getgrgid(gid)) {
		if (errno && errno != ENOENT) {
			return errno;
		} else {
			errno = 0;
			return 0;
		}
	}

	return EEXIST;
}

/**
 * Is the specified username unused
 *
 * @param name A username
 * @return 0 the username is unused
 *         EEXIST if the username is used
 *         errno on failure
 */
error_t is_free_username(const char *name)
{
	if (!getpwnam(name)) {
		if (errno)
			return errno;
		else
			return 0;
	}

	return EEXIST;
}

/**
 * Is the specified groupname unused
 *
 * @param name A groupname
 * @return 0 the groupname is unused
 *         EEXIST if the groupname is used
 *         errno on failure
 */
error_t is_free_groupname(const char *name)
{
	if (!getgrnam(name)) {
		if (errno)
			return errno;
		else
			return 0;
	}

	return EEXIST;
}

/**
 * Splits the gecos into its respective fields, and pads it to always
 * contain 5 fields
 *
 * @param gecos The gecos-string to split
 * @return An array of strings with all the fields, or NULL on failure
 */
char **split_gecos(const char *gecos)
{
	char **sgecos = NULL;
	char *tgecos = NULL;

	/* Cover the case of non-existing gecos-information */
	if (gecos)
		tgecos = strconcat(gecos, ",,,,,", NULL);
	else
		tgecos = strdup(",,,,,");

	/* We split into exactly 5 parts;
	 * fname, room, wphone, hphone, and other
	 */
	if (!tgecos || !(sgecos = strsplit(tgecos, ",", 5))) {
		fprintf(stderr,
			_("%s: `%s' failed; %s\n"),
			progname, "split_gecos", strerror(errno));
	}

	free(tgecos);

	return sgecos;
}

/**
 * Join the 5 gecos fields to one string
 *
 * @param fname The user's full name
 * @param room The user's room number
 * @param wphone The user's work phone
 * @param hphone The user's home phone
 * @param other Additional information
 * @return A string with the gecos information, or NULL on failure
 */
char *join_gecos(const char *fname, const char *room, const char *wphone,
		 const char *hphone, const char *other)
{
	char *gecos;

	if (!(gecos = strjoin(",", nstr(fname), nstr(room), nstr(wphone),
			      nstr(hphone), nstr(other), NULL))) {
		fprintf(stderr,
			_("%s: `%s' failed; %s\n"),
			progname, "join_gecos", strerror(errno));
	}

	return gecos;
}

/**
 * Find an entry in a comma-separated list
 *
 * @param list The list to find the entry in
 * @param entry The entry to find
 * @return A pointer to the entry, or NULL if no such entry exists
 */
static char *find_in_list(const char *list, const char *entry)
{
	char *p = NULL;

	/* Safe-guards for border-cases */
	if (!entry || !strlen(entry))
		goto EXIT;

	p = (char *)list;

	while (p && (p = strstr(p, entry))) {
		if ((p == list || *(p - 1) == ',') &&
		    (*(p + strlen(entry)) == '\0' ||
		     *(p + strlen(entry)) == ','))
			break;
		p++;
	}

EXIT:
	return p;
}

/**
 * Check whether an entry is a part of the list or not
 *
 * @param list The list to find the entry in
 * @param entry The entry to find
 * @return 0 if the entry is not in the list
 *         1 if the entry is in the list
 */
int is_in_list(const char *list, const char *entry)
{
	return find_in_list(list, entry) ? 1 : 0;
}

/**
 * Add an entry to a comma-separated list
 *
 * @param list The comma-separated list to add the entry to
 * @param entry The entry to add to the list
 * @return A new comma-separated list with the specified entry added,
 *         NULL if entry and list both are NULL, or on failure
 *         (on failure, errno is set)
 */
char *add_to_list(const char *list, const char *entry)
{
	char *newlist = NULL;

	/* If the list and the entry is NULL, return NULL */
	if (!list && !entry)
		goto EXIT;

	/* If the list is empty, the new list is just the entry */
	if (!list || !strlen(list)) {
		newlist = strdup(entry);
		goto EXIT;
	}

	/* If the entry is empty, the new list is the old one */
	if (!entry || !strlen(entry)) {
		newlist = strdup(list);
		goto EXIT;
	}

	/* If both list and entry exist and have a strlen() > 0,
	 * concatenate them
	 */
	newlist = strconcat(list, ",", entry, NULL);

EXIT:
	/* Return a pointer to the new list */
	return newlist;
}

/**
 * Add an entry to a comma-separated list
 * Already existing entries are silently ignored
 *
 * @param list The comma-separated list to add the entry to
 * @param entry The entry to add to the list
 * @return A new comma-separated list with the specified entry added,
 *         NULL if entry and list both are NULL, or on failure
 *         (on failure, errno is set)
 */
char *uniq_add_to_list(const char *list, const char *entry)
{
	if (is_in_list(list, entry))
		return strdup(list);
	else
		return add_to_list(list, entry);
}

/**
 * Remove an entry from a comma-separated list
 * Non-existing entries are silently ignored
 *
 * @param list The comma-separated list to remove an entry from
 * @param entry The entry to remove from the list
 * @return A new comma-separated list without the specified entry
 *         or NULL on failure (errno is set)
 */
char *remove_from_list(const char *list, const char *entry)
{
	char **array = NULL;
	char *newlist = NULL;
	uid_t i;

	if (!(newlist = strdup("")))
		goto EXIT;

	if (!(array = strsplit(list, ",", 0)))
		goto EXIT;

	for (i = 0; array[i]; i++) {
		if (!strcmp(array[i], entry))
			continue;

		if (!newlist) {
			newlist = strdup(array[i]);
		} else {
			char *tmp = strconcat(newlist, ",", array[i], NULL);
			free(newlist);
			newlist = tmp;
		}

		/* strdup or strconcat failed with ENOMEM */
		if (!newlist)
			break;
	}

EXIT:
	strfreev(array);

	if (!newlist) {
		fprintf(stderr,
			_("%s: `%s' failed; %s\n"),
			progname, "remove_from_list", strerror(errno));
	}

	return newlist;
}

/**
 * Check whether the list contains duplicate entries
 * Note: If empty entries (,,) are encountered,
 * the list is regarded as non-unique
 *
 * @param list The comma-separated list to check for duplicates
 * @return 0 if the list contains duplicates
 *         1 if all entries are unique
 */
int is_uniq_list(const char *list)
{
	char *p = (char *)list;
	char *c = NULL;

	/* An empty list is unique */
	if (!list || !strlen(list))
		return 1;

	/* A list with empty entries is regarded as non-unique */
	if (*list == ',' || strstr(list, ",,") ||
	    *(list + strlen(list) - 1) == ',')
		return 0;

	/* Scan the list for duplicate entries */
	while (p && (c = strchr(p, ','))) {
		size_t len = c - p;
		char *q = c;

		while (c && (q = ++c)) {
			if (!strncmp(p, q, len) &&
			    (*(q + len) == '\0' || *(q + len) == ','))
				return 0;

			c = strchr(c, ',');
		}
		p = p + len + 1;
	}

	return 1;
}

/**
 * Modify the list so that all entries are unique
 * Note: Empty entries will be removed
 *
 * @param list The comma-separated list to modify
 * @return A new comma-separated list without duplicate entries,
 *         or NULL on failure (errno is set)
 */
char *make_uniq_list(const char *list)
{
	char *p = (char *)list;
	char *newlist = NULL;
	char *tmplist = NULL;
	char *c = NULL;

	/* If the list already is unique, there's no need to modify it */
	if (is_uniq_list(list))
		return strdup(list);

	/* Scan the list for duplicate entries */
	while (p && (c = strchr(p, ','))) {
		size_t len = c - p;
		char *tmp = NULL;

		if (!(tmp = malloc(len + 1))) {
			free(newlist);
			newlist = NULL;
			goto EXIT;
		}

		(void)strncpy(tmp, p, len);
		*(tmp + len) = '\0';

		tmplist = uniq_add_to_list(newlist, tmp);
		free(tmp);
		free(newlist);
		newlist = tmplist;

		p = p + len + 1;
	}

	tmplist = uniq_add_to_list(newlist, p);
	free(newlist);
	newlist = tmplist;

EXIT:
	return newlist;
}

/**
 * Check whether an entry is a member of the array or not
 *
 * @param array The array to find the entry in
 * @param entry The entry to find
 * @return 0 if the entry is not in the array
 *         1 if the entry is in the array
 */
int is_in_array(char **array, const char *entry)
{
	int status = 0;
	uid_t i;

	if (!array || !entry)
		goto EXIT;

	for (i = 0; array[i]; i++) {
		if (!strcmp(array[i], entry)) {
			status = 1;
			break;
		}
	}

EXIT:
	return status;
}

/**
 * Check whether array1 overlaps array2
 *
 * @param array1 The first array to check
 * @param array2 The second array to check
 * @return 0 if the arrays overlap
 *         1 if the arrays are distinct
 */
int is_distinct_array(char **array1, char **array2)
{
	int status = 1;
	uid_t i;

	if (!array1 || !array2)
		goto EXIT;

	for (i = 0; array1[i]; i++) {
		if (is_in_array(array2, array1[i])) {
			status = 0;
			break;
		}
	}

EXIT:
	return status;
}

/**
 * Create the union of array1 and array2
 *
 * @param array1 The first array
 * @param array2 The second array
 * @return A pointer to the new array
 *         NULL on failure; errno is set (FIXME)
 */
char **array_union(char **array1, char **array2)
{
	char **newarray = NULL;
	char *newstr = NULL;
	uid_t i;

	if (!array1 && !array2)
		goto EXIT;

	if (!array1 || !array1[0]) {
		if (!(newstr = strjoinve(",", array2)))
			goto EXIT;
		else
			goto EXIT2;
	}

	if (!(newstr = strjoinve(",", array1)))
		goto EXIT;

	if (!array2)
		goto EXIT2;

	for (i = 0; array2[i]; i++) {
		if (!is_in_array(array1, array2[i])) {
			char *tmp = strjoine(",", newstr, array2[i], NULL);
			free(newstr);
			newstr = tmp;
		}
	}

EXIT2:
	newarray = strsplit(newstr, ",", 0);
	free(newstr);

EXIT:
	return newarray;
}

/**
 * Remove all elements of array2 from array1
 *
 * @param array1 The first array
 * @param array2 The second array
 * @return A pointer to the new array
 *         NULL on failure; errno is set (FIXME)
 */
char **array_cut(char **array1, char **array2)
{
	char **newarray = NULL;
	char *newstr = NULL;
	uid_t i;

	if (!array1)
		goto EXIT;

	if (!array2) {
		if (!(newstr = strjoinve(",", array1)))
			goto EXIT;

		goto EXIT2;
	}

	for (i = 0; array1[i]; i++) {
		if (!is_in_array(array2, array1[i])) {
			if (!newstr) {
				newstr = strdup(array1[i]);
			} else {
				char *tmp;

				tmp = strjoine(",", newstr, array1[i], NULL);
				free(newstr);
				newstr = tmp;
			}
		}
	}

EXIT2:
	newarray = strsplit(newstr, ",", 0);
	free(newstr);

EXIT:
	return newarray;
}

/**
 * Write a message to a tty
 *
 * @param iov An iovec with the message to write
 * @param dst The destination to write it to
 * @return 0 on success, errno on failure
 */
error_t tty_write(const struct iovec *iov, const char *dst)
{
	error_t status = 0;
	FILE *fp = NULL;

	if (!(fp = fopen(dst, "w"))) {
		if (errno == EACCES)
			errno = 0;
		else
			status = errno;

		goto EXIT;
	}

	if (fwrite(iov->iov_base, iov->iov_len, 1, fp) < iov->iov_len) {
		if (errno == EACCES)
			errno = 0;
		else
			status = errno;

		goto EXIT;
	}

	if (fclose(fp) == EOF)
		status = errno;

EXIT:
	return status;
}
