-- Copyright (C) 2009 Papavasileiou Dimitris                             
--                                                                      
-- This program is free software: you can redistribute it and/or modify 
-- it under the terms of the GNU General Public License as published by 
-- the Free Software Foundation, either version 3 of the License, or    
-- (at your option) any later version.                                  
--                                                                      
-- This program is distributed in the hope that it will be useful,      
-- but WITHOUT ANY WARRANTY; without even the implied warranty of       
-- MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the        
-- GNU General Public License for more details.                         
--                                                                      
-- You should have received a copy of the GNU General Public License    
-- along with this program.  If not, see <http://www.gnu.org/licenses/>.

require "frames"
require "transforms"
require "switches"
require "dynamics"
require "physics"
require "morejoints"
require "moremath"
require "transitions"

director = {}

local function callhooks (hooks, ...)
   if type (hooks) == "table" then
      for _, hook in pairs(hooks) do
         if ... and type(...) == "table" then
            hook (unpack(...))
         else
            hook (...)
         end
      end
   elseif type (hooks) == "function" then
      if ... and type(...) == "table" then
         hooks(unpack(...))
      else
         hooks (...)
      end
   end
end

local function adjustorbit (self, radius, azimuth, elevation)
   if not self.setup then
      if radius then
	 self.rest[1] = math.clamp (radius, 0, 1 / 0)
      end

      if azimuth then
	 self.rest[2] = azimuth
      end

      if elevation then
	 self.rest[3] = elevation and math.clamp (elevation, 0, math.pi / 2)
      end

      -- Wait for the first full rho, theta, phi specifcation
      -- and use it as the initial rig configuration
   
      if #self.rest == 3 then
	 self.system.head.position = {self.rest[1] * math.cos (self.rest[2]) *
			       math.sin (self.rest[3]),
			       self.rest[1] * math.sin (self.rest[2]) *
			       math.sin (self.rest[3]),
			       self.rest[1] * math.cos (self.rest[3])}

	 self.system.head.orientation = transforms.relue (0,
						   math.deg(self.rest[3]),
						   math.deg(self.rest[2]))

	 -- Join the rig.
	 
	 self.system.waist.axes = {transforms.fromnode (self.system.head, {0, 1, 0}),
			    {1, 0, 0}, 
			    {0, 0, 1},}
	 self.system.neck.axis = transforms.fromnode (self.system.head, {0, 0, 1})
	 self.system.waist.bodies = {self.system.torso, nil}
	 self.system.neck.bodies = {self.system.torso, self.system.head}

	 self.state = {self.rest[1], self.rest[2], self.rest[3]}
	 self.setup = true
      end
   else
      if radius then
	 self.state[1] = math.clamp (radius, -self.rest[1], 1 / 0)
      end

      if azimuth then
	 self.state[2] = math.clamp (azimuth,
				     -math.pi + self.rest[2], 
				     math.pi + self.rest[2])
      end

      if elevation then
	 self.state[3] = math.clamp (elevation,
				     -math.pi / 2 + self.rest[3],
				     math.pi / 2 + self.rest[3])
      end

      self.system.neck.preload = self.state[1] - self.rest[1]
      self.system.waist.preload = {self.state[3] - self.rest[3],
			    0,
			    self.state[2] - self.rest[2]}
   end
end

local function adjustperson (self, azimuth, elevation)
   if not self.setup then
      if azimuth then
	 self.rest[1] = azimuth
      end

      if elevation then
	 self.rest[2] = elevation and math.clamp (elevation, 0, math.pi / 2)
      end

      -- Wait for the first full theta, phi specifcation
      -- and use it as the initial rig configuration
   
      if #self.rest == 2 then
	 self.head.orientation = transforms.relue (0,
						   math.deg(self.rest[2]),
						   math.deg(self.rest[1]))

	 -- Join the rig.
	 
	 self.neck.axes = {transforms.fromnode (self.head, {0, 1, 0}),
			   {1, 0, 0}, 
			   {0, 0, 1},}
	 self.neck.bodies = {self.head, nil}

	 self.state = {self.rest[1], self.rest[2]}
	 self.setup = true
      end
   else
      if azimuth then
	 self.state[1] = math.clamp (azimuth,
				     -math.pi + self.rest[1], 
				     math.pi + self.rest[1])
      end

      if elevation then
	 self.state[2] = math.clamp (elevation,
				     -math.pi / 2 + self.rest[2],
				     math.pi / 2 + self.rest[2])
      end

      self.neck.preload = {self.state[2] - self.rest[2],
			   0,
			   self.state[1] - self.rest[1]}
   end
end

function director.orbit (values)
   local self, meta, oldindex, oldnewindex

   self = frames.gimbal {
      setup = false,
      rest = {},
      state = {0, 0, 0},

      system = bodies.system {
	 -- The bodies.

	 head = bodies.point {
	    position = {0, 0, 1},
	    mass = physics.spheremass (1, 0.1),
	    eye = frames.observer {}
	 },

	 torso = bodies.point {
	    position = {0, 0, 0},
	    mass = physics.spheremass (1, 0.1),
	 },

	 -- Joints.

	 waist = springs.euler {
	    axes = {{0, 1, 0}, {1, 0, 0}, {0, 0, 1}},

	    stiffness = 3000,
	    damping = 1000,
	 },

	 neck = springs.linear {
	    stiffness = 3000,
	    damping = 1000,
	 },

	 step = function (self)
		   -- 		if self.state[2] >= math.pi and
		   -- 		   self.waist.state[3] < 0 then
		   -- 		   adjustorbit (self,
		   -- 			      self.state[1],
		   -- 			      self.state[2] - 2 * math.pi,
		   -- 			      self.state[3])
		   -- 		elseif self.state[2] <= -math.pi and
		   -- 		   self.waist.state[3] > 0 then
		   -- 		   adjustorbit (self,
		   -- 			      self.state[1],
		   -- 			      self.state[2] + 2 * math.pi,
		   -- 			      self.state[3])
		   -- 		end
		   -- 		print (math.deg(joints.waist.state[3]),
		   -- 		       math.deg(joints.waist.stops[3][1][1]), 
		   -- 		       math.length (self.velocity))
		end
      }
   }

   meta = getmetatable(self)
   meta.damping = 1000
   meta.stiffness = 3000

   oldindex = meta.__index
   oldnewindex = meta.__newindex

   meta.__index = function(table, key)
		     if key == "radius" then
			return table.state[1]
		     elseif key == "azimuth" then
			return table.state[2]
		     elseif key == "elevation" then
			return table.state[3]
		     elseif key == "stiffness" then
			return meta.stiffness
		     elseif key == "damping" then
			return meta.damping
		     elseif key == "speed" then
			return math.length (self.system.head.velocity)
		     else
			return oldindex(table, key)
		     end
		  end

   meta.__newindex = function(table, key, value)
			if key == "radius" then
			   adjustorbit (table, value, nil, nil)
			elseif key == "azimuth" then
			   adjustorbit (table, nil, value, nil)
			elseif key == "elevation" then
			   adjustorbit (table, nil, nil, value)
			elseif key == "stiffness" then
			   meta.stiffness = value

			   self.system.waist.stiffness = value
			   self.system.neck.stiffness = value
			elseif key == "damping" then
			   meta.damping = value

			   self.system.waist.damping = value
			   self.system.neck.damping = value
			else
			   oldnewindex(table, key, value)
			end
		     end

   for key, value in pairs(values) do
      self[key] = value
   end

   return self
end

function director.firstperson (values)
   local self, meta, oldindex, oldnewindex

   self = bodies.system {
      setup = false,
      rest = {},
      state = {0, 0, 0},

      -- The bodies.

      head = bodies.point {
	 mass = physics.spheremass (1, 0.1),
	 eye = frames.observer {}
      },

      -- Joints.

      neck = springs.euler {
	 axes = {{0, 1, 0}, {1, 0, 0}, {0, 0, 1}},

	 stiffness = 3000,
	 damping = 1000,
      },
   }

   meta = getmetatable(self)
   meta.damping = 1000
   meta.stiffness = 3000

   oldindex = meta.__index
   oldnewindex = meta.__newindex

   meta.__index = function(table, key)
		     if key == "azimuth" then
			return table.state[1]
		     elseif key == "elevation" then
			return table.state[2]
		     elseif key == "stiffness" then
			return meta.stiffness
		     elseif key == "damping" then
			return meta.damping
		     else
			return oldindex(table, key)
		     end
		  end

   meta.__newindex = function(table, key, value)
			if key == "azimuth" then
			   adjustperson (table, value, nil)
			elseif key == "elevation" then
			   adjustperson (table, nil, value)
			elseif key == "stiffness" then
			   meta.stiffness = value

			   self.neck.stiffness = value
			elseif key == "damping" then
			   meta.damping = value

			   self.neck.damping = value
			else
			   oldnewindex(table, key, value)
			end
		     end

   for key, value in pairs(values) do
      self[key] = value
   end

   return self
end

function director.script (values)
   local self, meta, oldindex, oldnewindex

   self = frames.timer {
      period = 0,

      tick = function (self, tick, delta, elapsed)
		if not meta.event then
		   local next = 1 / 0

		   for i, event in pairs (meta) do
		      if tonumber (i) and i > elapsed and i < next then
			 next = i
			 meta.event = event
		      end
		   end

		   self.period = next - elapsed
		else
		   callhooks (meta.event, self)

		   meta.event = nil
		   self.period = 0
		end
	     end
   }

   meta = getmetatable(self)

   oldindex = meta.__index
   oldnewindex = meta.__newindex

   meta.__index = function(table, key)
		     if tonumber(key) then
			return meta[key]
		     else
			return oldindex(table, key)
		     end
		  end

   meta.__newindex = function(table, key, value)
			if tonumber(key) then
			   meta[tonumber(key)] = value

			   meta.event = nil
			   self.period = 0
			else
			   oldnewindex(table, key, value)
			end
		     end

   for key, value in pairs(values) do
      self[key] = value
   end

   return self
end
