/* Copyright (C) 2009 Papavasileiou Dimitris                             
 *                                                                      
 * This program is free software: you can redistribute it and/or modify 
 * it under the terms of the GNU General Public License as published by 
 * the Free Software Foundation, either version 3 of the License, or    
 * (at your option) any later version.                                  
 *                                                                      
 * This program is distributed in the hope that it will be useful,      
 * but WITHOUT ANY WARRANTY; without even the implied warranty of       
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the        
 * GNU General Public License for more details.                         
 *                                                                      
 * You should have received a copy of the GNU General Public License    
 * along with this program.  If not, see <http://www.gnu.org/licenses/>.
 */

#include <GL/glu.h>
#include <lua.h>
#include <lauxlib.h>
#include "annotation.h"

static void callhooks (lua_State *L, void *key, int reference)
{
    if (reference != LUA_REFNIL) {
	lua_rawgeti(L, LUA_REGISTRYINDEX, reference);
	
	if (lua_isfunction (L, -1)) {
	    lua_getfield (L, LUA_REGISTRYINDEX, "userdata");
	    lua_pushlightuserdata (L, key);
	    lua_gettable (L, -2);
	    lua_replace (L, -2);
	    
	    lua_call(L, 1, 0);
	} else if (lua_istable (L, -1)) {
	    lua_getfield (L, LUA_REGISTRYINDEX, "userdata");
	    lua_pushnil(L);
	
	    while(lua_next(L, -3)) {
		lua_pushlightuserdata (L, key);
		lua_gettable (L, -4);
		lua_call (L, 1, 0);
	    }
	    
	    lua_pop (L, 2);
	} else {
	    lua_pop (L, 1);
	}
    }
}

@implementation Annotation

-(Annotation *) init
{
    [super init];

    self->thickness = 1;
    self->radius = 1;
    self->angle = 0;

    return self;
}

-(void) transform
{
    GLdouble M[16], P[16], w[3];
    GLfloat *r, *R, *p, *W;
    GLint v[4];
    
    id child;
    int i, j;

    /* Call the node's transform hook since
       we won't propagate the method upwards. */
    
    callhooks (_L, self, self->transform);
    
    /* Calculate our own allocation based on our children. */
    
    self->minimum[0] = 2 * self->padding[0];
    self->minimum[1] = 2 * self->padding[1];
    
    for(child = [self children] ; child ; child = [child sister]) {
	float width, height;

	width = [child width] + 2 * self->padding[0];
	height = [child height] + 2 * self->padding[1];
	
	self->minimum[0] = width > self->minimum[0] ?
	                   width : self->minimum[0];

	self->minimum[1] = height > self->minimum[1] ?
	                   height : self->minimum[1];
    }

    self->allocated[0] = self->minimum[0] > self->requested[0] ?
	                 self->minimum[0] : self->requested[0];
    self->allocated[1] = self->minimum[1] > self->requested[1] ?
	                 self->minimum[1] : self->requested[1];
	
    /* Now reset the children. */
    
    if([self children]) {
	GLfloat delta[3] = {0, 0, 0};

	/* Align the child properly. */
    
	if (self->align[0] < 0) {
	    delta[0] = 0.5 * (self->minimum[0] - self->allocated[0]);
	} else if (self->align[0] > 0) {
	    delta[0] = 0.5 * (self->allocated[0] - self->minimum[0]);
	} else {
	    delta[0] = 0;
	}

	if (self->align[1] < 0) {
	    delta[1] = 0.5 * (self->minimum[1] - self->allocated[1]);
	} else if (self->align[1] > 0) {
	    delta[1] = 0.5 * (self->allocated[1] - self->minimum[1]);
	} else {
	    delta[1] = 0;
	}

	[[self children] setPositionTo: delta];
    }

    /* Calculate the node's transform. */
    
    p = [self position];
    W = [self orientation];
    
    r = [[self parent] translation];
    R = [[self parent] rotation];

    M[3] = 0;
    M[7] = 0;
    M[11] = 0;
    M[15] = 1;
    
    for (i = 0 ; i < 3 ; i += 1) {
	for (j = 0 ; j < 3 ; j += 1) {
	    M[j * 4 + i] = R[i * 3] * W[j] +
		           R[i * 3 + 1] * W[j + 3] +
		           R[i * 3 + 2] * W[j + 6];
	}

	M[12 + i] = R[i * 3] * p[0] +
	            R[i * 3 + 1] * p[1] +
	            R[i * 3 + 2] * p[2] +
		    r[i]; 
    }

    /* Get the projection matrix and viewport, multiply
       the node's transform by the current modelview matrix
       and project to screen space. */
    
    glGetDoublev (GL_PROJECTION_MATRIX, P);
    glGetIntegerv (GL_VIEWPORT, v);

    glMatrixMode (GL_MODELVIEW);
    glPushMatrix();
    glMultMatrixd (M);
    glGetDoublev (GL_MODELVIEW_MATRIX, M);
    gluProject (0, 0, 0, M, P, v, &w[0], &w[1], &w[2]);
    glPopMatrix ();

    /* Just return the position and orientation. */
	
    self->translation[0] = w[0] / v[2] - 0.5 + 0.5 * self->minimum[0] +
	                   self->radius * cos (self->angle) + 0.01;
    self->translation[1] = w[1] / v[3] - 0.5 +
	                   self->radius * sin(self->angle);
    self->translation[2] = -w[2];

    for (i = 0 ; i < 9 ; i += 1) {
	self->rotation[i] = self->orientation[i];
    }

    /* Calculate the homogenous transform matrix. */
    
    self->homogenous[0] = self->rotation[0];
    self->homogenous[1] = self->rotation[3];
    self->homogenous[2] = self->rotation[6];
    self->homogenous[3] = 0;
    
    self->homogenous[4] = self->rotation[1];
    self->homogenous[5] = self->rotation[4];
    self->homogenous[6] = self->rotation[7];
    self->homogenous[7] = 0;
    
    self->homogenous[8] = self->rotation[2];
    self->homogenous[9] = self->rotation[5];
    self->homogenous[10] = self->rotation[8];
    self->homogenous[11] = 0;

    self->homogenous[12] = self->translation[0];
    self->homogenous[13] = self->translation[1];
    self->homogenous[14] = self->translation[2];
    self->homogenous[15] = 1;
    
    for(child = [self children] ; child ; child = [child sister]) {
	[child transform];
    }
}

-(void) traversePass: (int)pass
{
    GLdouble x, y;

    if (pass == 2) {
	x = self->radius * cos (self->angle);
	y = self->radius * sin (self->angle);

	glMatrixMode (GL_PROJECTION);
	glPushMatrix();
	glLoadIdentity();
	glOrtho(-0.5, 0.5, -0.5, 0.5, 0, 1);

	glMatrixMode (GL_MODELVIEW);
	glPushMatrix();
	glLoadMatrixf ([self homogenous]);
    
	glUseProgramObjectARB(0);
	glBlendFunc (GL_SRC_ALPHA, GL_ONE_MINUS_SRC_ALPHA);

	glEnable(GL_LINE_SMOOTH);
	glEnable(GL_POINT_SMOOTH);
	glEnable(GL_BLEND);

	glColor4fv (self->color);
	glLineWidth (self->thickness);
	glPointSize (3 * self->thickness);

	glBegin (GL_POINTS);
	glVertex2d (-0.5 * self->minimum[0] - x - 0.01, -y);
	glEnd();

	glBegin (GL_LINE_STRIP);
	glVertex2d (-0.5 * self->minimum[0], 0);
	glVertex2d (-0.5 * self->minimum[0] - 0.01, 0);
	glVertex2d (-0.5 * self->minimum[0] - x - 0.01, -y);
	glEnd();

	glDisable(GL_BLEND);
	glDisable(GL_LINE_SMOOTH);
	glDisable(GL_POINT_SMOOTH);
    
	glMatrixMode (GL_MODELVIEW);
	glLoadIdentity ();
	
	[super traversePass: pass];
    
	glPopMatrix();
    
	glMatrixMode (GL_PROJECTION);
	glPopMatrix();
    } else {
	[super traversePass: pass];
    }
}

-(void) get
{
    const char *k;
    
    k = lua_tostring (_L, 2);

    if (!xstrcmp(k, "node")) {
	lua_getmetatable (_L, 1);
	lua_replace (_L, 1);
	lua_gettable (_L, 1);
    } else if (!xstrcmp(k, "thickness")) {
	lua_pushnumber (_L, self->thickness);
    } else if (!xstrcmp(k, "radius")) {
	lua_pushnumber (_L, self->radius);
    } else if (!xstrcmp(k, "angle")) {
	lua_pushnumber (_L, self->angle * 180 / M_PI);
    } else {
	[super get];
    }
}

-(void) set
{
    const char *k;

    k = lua_tostring (_L, 2);

    if (!xstrcmp(k, "node")) {
	if (!lua_isnil (_L, 3)) {
	    self->node = *(id *)lua_touserdata (_L, 3);
	} else {
	    self->node = nil;
	}
	
	lua_getmetatable (_L, 1);
	lua_replace (_L, 1);
	lua_settable (_L, 1);
    } else if (!xstrcmp(k, "thickness")) {
	self->thickness = lua_tonumber (_L, 3);
    } else if (!xstrcmp(k, "radius")) {
	self->radius = lua_tonumber (_L, -1);
    } else if (!xstrcmp(k, "angle")) {
	self->angle = lua_tonumber (_L, -1) * M_PI / 180;
   } else {
	[super set];
    }
}

@end
