/* Copyright (C) 2009 Papavasileiou Dimitris                             
 *                                                                      
 * This program is free software: you can redistribute it and/or modify 
 * it under the terms of the GNU General Public License as published by 
 * the Free Software Foundation, either version 3 of the License, or    
 * (at your option) any later version.                                  
 *                                                                      
 * This program is distributed in the hope that it will be useful,      
 * but WITHOUT ANY WARRANTY; without even the implied warranty of       
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the        
 * GNU General Public License for more details.                         
 *                                                                      
 * You should have received a copy of the GNU General Public License    
 * along with this program.  If not, see <http://www.gnu.org/licenses/>.
 */

#include <lua.h>
#include <lauxlib.h>
#include <ctype.h>
#include <string.h>

#include "airplane.h"
#include "thruster.h"
#include "piston.h"

static double *values[3];
static int lengths[3];

static double lookup (double x, double *values, int length)
{
    double *a, *b;
    int k;
    
    if (length > 0) {
	for(k = 0, a = values, b = a + 2;
	    k < length - 4 && b[0] <= x ;
	    k += 2, a = b, b += 2);
	
	return a[1] + (b[1] - a[1]) / (b[0] - a[0]) * (x - a[0]);
    } else {
	return 0;
    }
}

double get_temperature_at (double h)
{
    return lookup(h, values[0], lengths[0]);
}

double get_pressure_at (double h)
{
    return lookup(h, values[1], lengths[1]);
}

double get_density_at (double h)
{
    return lookup(h, values[2], lengths[2]);
}

static char *decapitalize (char *s)
{
    s[0] = tolower(s[0]);

    return s;
}

static int generic_tostring(lua_State *L)
{
    id N;

    N = *(id *)lua_touserdata(L, 1);
    lua_pushstring(L, [N name]);
   
    return 1;
}

static int generic_index(lua_State *L)
{
    id N;

    N = *(id *)lua_touserdata(L, 1);
    
    [N get];

    return 1;
}

static int generic_newindex(lua_State *L)
{
    id N;

    N = *(id *)lua_touserdata(L, 1);
    
    [N set];

    return 0;
}

static int generic_gc(lua_State *L)
{
    id N;

    N = *(id *)lua_touserdata(L, 1);
    
    [N free];

    return 0;
}

static int constructnode (lua_State *L)
{
    Class class;
    id object;

    lua_pushvalue (L, lua_upvalueindex (1));
    class = (Class)lua_touserdata(L, -1);
    lua_pop(L, 1);

    /* Create the userdata... */

    object = [[class alloc] init];
    *(id *)lua_newuserdata(L, sizeof(id)) = object;
    
    lua_newtable (L);
    lua_pushstring(L, "__index");
    lua_pushcfunction(L, (lua_CFunction)generic_index);
    lua_settable(L, -3);
    lua_pushstring(L, "__newindex");
    lua_pushcfunction(L, (lua_CFunction)generic_newindex);
    lua_settable(L, -3);
    lua_pushstring(L, "__tostring");
    lua_pushcfunction(L, (lua_CFunction)generic_tostring);
    lua_settable(L, -3);
    lua_pushstring(L, "__gc");
    lua_pushcfunction(L, (lua_CFunction)generic_gc);
    lua_settable(L, -3);
    lua_setmetatable(L, -2);
    
    /* ...and initialize it. */

    if(lua_istable(L, 1)) {
	lua_pushnil(L);
	
	while(lua_next(L, 1)) {
	    lua_pushvalue(L, -2);
	    lua_insert(L, -2);
	    lua_settable(L, 2);
	}
    }

    /* Add a weak reference. */
    
    lua_pushstring (L, "userdata");
    lua_gettable (L, LUA_REGISTRYINDEX);
    lua_pushlightuserdata (L, object);
    lua_pushvalue (L, 2);
    lua_settable (L, -3);
    lua_pop(L, 1);

    return 1;
}

static int aviation_index (lua_State *L)
{
    const char *k;
    int i;
    
    k = lua_tostring(L, 2);

    if (!strcmp(k, "temperature")) {
	lua_newtable (L);
	
	for (i = 0 ; i < lengths[0] ; i += 1) {
	    lua_pushnumber (L, values[0][i]);
	    lua_rawseti (L, -2, i + 1);
	}
    } else if (!strcmp(k, "pressure")) {
	lua_newtable (L);
	
	for (i = 0 ; i < lengths[1] ; i += 1) {
	    lua_pushnumber (L, values[1][i]);
	    lua_rawseti (L, -2, i + 1);
	}
    } else if (!strcmp(k, "density")) {
	lua_newtable (L);
	
	for (i = 0 ; i < lengths[2] ; i += 1) {
	    lua_pushnumber (L, values[2][i]);
	    lua_rawseti (L, -2, i + 1);
	}
    } else {
	lua_rawget (L, 1);
    }

    return 1;
}

static int aviation_newindex (lua_State *L)
{
    const char *k;
    int i, n;

    k = lua_tostring(L, 2);

    if (!strcmp(k, "temperature")) {
	if (lua_istable (L, 3)) {
	    n = lua_objlen(L, 3);

	    lengths[0] = n;
	    values[0] = (double *)realloc(values[0], n * sizeof(double));

	    for (i = 0 ; i < n ; i += 1) {
		lua_rawgeti(L, 3, i + 1);
		values[0][i] = lua_tonumber(L, -1);

		lua_pop(L, 1);
	    }
	} else {
	    lengths[0] = 0;
	}
    } else if (!strcmp(k, "pressure")) {
	if (lua_istable (L, 3)) {
	    n = lua_objlen(L, 3);

	    lengths[1] = n;
	    values[1] = (double *)realloc(values[1], n * sizeof(double));

	    for (i = 0 ; i < n ; i += 1) {
		lua_rawgeti(L, 3, i + 1);
		values[1][i] = lua_tonumber(L, -1);

		lua_pop(L, 1);
	    }
	} else {
	    lengths[1] = 0;
	}
    } else if (!strcmp(k, "density")) {
	if (lua_istable (L, 3)) {
	    n = lua_objlen(L, 3);

	    lengths[2] = n;
	    values[2] = (double *)realloc(values[2], n * sizeof(double));

	    for (i = 0 ; i < n ; i += 1) {
		lua_rawgeti(L, 3, i + 1);
		values[2][i] = lua_tonumber(L, -1);

		lua_pop(L, 1);
	    }
	} else {
	    lengths[2] = 0;
	}
    } else {
	lua_rawset (L, 1);
    }

    return 0;
}

int luaopen_aviation (lua_State *L)
{
    int i;
    
    Class aircraft[] = {
	[Airplane class]
    };	

    Class propulsion[] = {
	[Thruster class], [Piston class]
    };
    
    lua_newtable (L);
    
    for (i = 0 ; i < sizeof(aircraft) / sizeof(aircraft[0]) ; i += 1) {
	lua_pushlightuserdata (L, aircraft[i]);
	lua_pushcclosure (L, constructnode, 1);
	lua_setfield(L, -2, decapitalize(strdupa([aircraft[i] name])));
    }
    
    lua_setglobal (L, "aircraft");
    
    lua_newtable (L);
    
    for (i = 0 ; i < sizeof(propulsion) / sizeof(propulsion[0]) ; i += 1) {
	lua_pushlightuserdata (L, propulsion[i]);
	lua_pushcclosure (L, constructnode, 1);
	lua_setfield(L, -2, decapitalize(strdupa([propulsion[i] name])));
    }
    
    lua_setglobal (L, "propulsion");

    /* Create the aviation table. */
    
    lua_newtable (L);    
    lua_newtable (L);
    
    lua_pushcfunction (L, aviation_index);
    lua_setfield (L, -2, "__index");

    lua_pushcfunction (L, aviation_newindex);
    lua_setfield (L, -2, "__newindex");

    lua_setmetatable (L, -2);
    lua_setglobal (L, lua_tostring (L, 1));

    return 0;
}
