/* Copyright (C) 2009 Papavasileiou Dimitris                             
 *                                                                      
 * This program is free software: you can redistribute it and/or modify 
 * it under the terms of the GNU General Public License as published by 
 * the Free Software Foundation, either version 3 of the License, or    
 * (at your option) any later version.                                  
 *                                                                      
 * This program is distributed in the hope that it will be useful,      
 * but WITHOUT ANY WARRANTY; without even the implied warranty of       
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the        
 * GNU General Public License for more details.                         
 *                                                                      
 * You should have received a copy of the GNU General Public License    
 * along with this program.  If not, see <http://www.gnu.org/licenses/>.
 */

#include <lua.h>
#include <lauxlib.h>
#include <GL/gl.h>

#include "particulate.h"
#include "fluid.h"

#define STRIDE 11
#define SUBSTRIDE 8

#define POSITION(i) (self->particles + ((i) * STRIDE))
#define VELOCITY(i) (self->particles + ((i) * STRIDE) + 3)
#define COLOR(i) (self->particles + ((i) * STRIDE) + 6)
#define TEMPERATURE(i) (self->particles[((i) * STRIDE) + 9])
#define DENSITY(i) (self->particles[((i) * STRIDE) + 10])
#define BUCKET(i) (self->table + (i) * (self->size + 1))

#define SUBPOSITION(i, j) (self->subparticles + \
			   (((i) * self->clustering + (j)) * SUBSTRIDE))
#define SUBVELOCITY(i, j) (self->subparticles + \
			   (((i) * self->clustering + (j)) * SUBSTRIDE) + 3)
#define ALBEDO(i, j) (self->subparticles[(((i) * self->clustering + (j)) * \
					  SUBSTRIDE) + 6])
#define EXTINCTION(i, j) (self->subparticles[(((i) * self->clustering + (j)) * \
					      SUBSTRIDE) + 7])

static double gaussian (double mu, double sigma)
{
    double x, y, rsquared;

    do {
	x = -1 + 2 * (double)random() / RAND_MAX;
	y = -1 + 2 * (double)random() / RAND_MAX;

	rsquared = x * x + y * y;
    } while (rsquared > 1.0 || rsquared == 0);

    return sigma * y * sqrt (-2.0 * log (rsquared) / rsquared) + mu;
}

static unsigned int hash(int i, int j, int k, unsigned int n)
{
    return ((unsigned int)((i * 73856093) ^
			   (j * 19349663) ^
			   (k * 83492791))) % n;
}

@implementation Fluid

-(Fluid *) init
{
    self = [super init];

    self->scale[0] = 1;
    self->scale[1] = 1;

    self->granularity = 1;
    self->size = 0;
    self->clustering = 0;
    self->generated = 0;
    self->support = 1;
    self->density = 1;
    self->diffusion = 0;
    self->mobility = 0;
    self->stiffness[0] = 1;
    self->stiffness[1] = 1;
    self->viscosity[0] = 1;
    self->viscosity[1] = 1;
    self->hashes = 1033;
    
    self->table = (unsigned int *)realloc (NULL,
					   self->hashes *
					   sizeof (unsigned int));
    self->particles = NULL;
    self->subparticles = NULL;
    
    return self;
}
    
-(void) get
{
    const char *k;
    int i, n;

    if (lua_type (_L, 2) == LUA_TNUMBER) {
	n = (int) lua_tonumber (_L, 2) - 1;
	
        if(n >= 0 && n < self->generated) {
	    lua_newtable (_L);
        
	    for(i = 0; i < 9; i += 1) {
		lua_pushnumber (_L, self->particles[n * STRIDE + i]);
		lua_rawseti (_L, -2, i + 1);
	    }
	} else {
	    lua_pushnil (_L);
	}
    } else {
	k = lua_tostring (_L, 2);

	if (!xstrcmp(k, "scale")) {
	    lua_newtable (_L);
        
	    for(i = 0; i < 2; i += 1) {
		lua_pushnumber (_L, self->scale[i]);
		lua_rawseti (_L, -2, i + 1);
	    }
	} else if (!xstrcmp(k, "state")) {
	    lua_newtable (_L);
        
	    for(i = 0; i < 2; i += 1) {
		lua_pushnumber (_L, self->state[i]);
		lua_rawseti (_L, -2, i + 1);
	    }
	} else if (!xstrcmp(k, "granularity")) {
	    lua_pushnumber (_L, self->granularity);
	} else if (!xstrcmp(k, "size")) {
	    lua_pushnumber (_L, self->size);
	} else if (!xstrcmp(k, "clustering")) {
	    lua_pushnumber (_L, self->clustering);
	} else  if (!xstrcmp(k, "diffusion")) {
	    lua_pushnumber (_L, self->diffusion);
	} else  if (!xstrcmp(k, "mobility")) {
	    lua_pushnumber (_L, self->mobility);
	} else if (!xstrcmp(k, "density")) {
	    lua_pushnumber (_L, self->density);
	} else if (!xstrcmp(k, "support")) {
	    lua_pushnumber (_L, self->support);
	} else if (!xstrcmp(k, "stiffness")) {
	    lua_newtable (_L);
        
	    for(i = 0; i < 2; i += 1) {
		lua_pushnumber (_L, self->stiffness[i]);
		lua_rawseti (_L, -2, i + 1);
	    }
	} else if (!xstrcmp(k, "viscosity")) {
	    lua_newtable (_L);
        
	    for(i = 0; i < 2; i += 1) {
		lua_pushnumber (_L, self->viscosity[i]);
		lua_rawseti (_L, -2, i + 1);
	    }
	} else if (!xstrcmp(k, "hashes")) {
	    lua_pushnumber (_L, self->hashes);
	} else {
	    [super get];
	}
    }
}

-(void) set
{    
    const char *k;
    int i, n;

    if (lua_type (_L, 2) == LUA_TNUMBER) {
	int j;
	
	n = (int) lua_tonumber (_L, 2) - 1;

        if(n >= 0 && n < self->size && lua_istable (_L, 3)) {
	    double mu_0, mu_1, mu_2;
	    double sigma_0, sigma_1, sigma_2;
	    double x, y, z, u, v, w;
	    float *R, *r;

	    R = [self rotation];
	    r = [self translation];

	    /* Transform position and velocity. */
	    
	    lua_rawgeti (_L, 3, 1);
	    x = lua_tonumber (_L, -1);
	    
	    lua_rawgeti (_L, 3, 2);
	    y = lua_tonumber (_L, -1);
	    
	    lua_rawgeti (_L, 3, 3);
	    z = lua_tonumber (_L, -1);
	    
	    lua_rawgeti (_L, 3, 4);
	    u = lua_tonumber (_L, -1);
	    
	    lua_rawgeti (_L, 3, 5);
	    v = lua_tonumber (_L, -1);
	    
	    lua_rawgeti (_L, 3, 6);
	    w = lua_tonumber (_L, -1);
	    
	    lua_pop (_L, 6);

	    self->particles[n * STRIDE + 0] = R[0] * x + R[1] * y + R[2] * z + r[0];
	    self->particles[n * STRIDE + 1] = R[3] * x + R[4] * y + R[5] * z + r[1];
	    self->particles[n * STRIDE + 2] = R[6] * x + R[7] * y + R[8] * z + r[2];

	    self->particles[n * STRIDE + 3] = R[0] * u + R[1] * v + R[2] * w;
	    self->particles[n * STRIDE + 4] = R[3] * u + R[4] * v + R[5] * w;
	    self->particles[n * STRIDE + 5] = R[6] * u + R[7] * v + R[8] * w;
	    
	    /* Color. */
	    
            for(i = 6 ; i < 9 ; i += 1) {
                lua_rawgeti (_L, 3, i + 1);
                self->particles[n * STRIDE + i] = lua_tonumber (_L, -1);
                
                lua_pop (_L, 1);
            }

	    /* Also set the temperature to 1. */
	    
	    self->particles[n * STRIDE + 9] = 1;
	    self->particles[n * STRIDE + 10] = self->density;

	    lua_rawgeti (_L, 3, 10);
	    mu_0 = lua_tonumber (_L, -1);
	    lua_rawgeti (_L, 3, 11);
	    sigma_0 = lua_tonumber (_L, -1);

	    lua_rawgeti (_L, 3, 12);
	    mu_1 = lua_tonumber (_L, -1);
	    lua_rawgeti (_L, 3, 13);
	    sigma_1 = lua_tonumber (_L, -1);

	    lua_rawgeti (_L, 3, 14);
	    mu_2 = lua_tonumber (_L, -1);
	    lua_rawgeti (_L, 3, 15);
	    sigma_2 = lua_tonumber (_L, -1);

	    lua_pop (_L, 6);

	    for (j = 0 ; j < clustering ; j += 1) {
		double rho, theta, phi;
		int k;
		
		k = (n * self->clustering + j) * SUBSTRIDE;

		rho = gaussian (mu_0, sigma_0);

		theta = M_PI * (double)random () / RAND_MAX;
		phi = 2 * M_PI * (double)random () / RAND_MAX;
		
		self->subparticles[k] = rho * cos(theta) * sin(phi);
		self->subparticles[k + 1] = rho * sin(theta) * sin (phi);
		self->subparticles[k + 2] = rho * cos(phi);
		
		self->subparticles[k + 3] = 0;
		self->subparticles[k + 4] = 0;
		self->subparticles[k + 5] = 0;

		self->subparticles[k + 6] = gaussian(mu_1, sigma_1);
		self->subparticles[k + 7] = gaussian(mu_2, sigma_2);
	    }
	    
	    if (n >= self->generated) {
		self->generated = n + 1;
	    }
        }
    } else {
	k = lua_tostring (_L, 2);

	if (!xstrcmp(k, "scale")) {
	    if(lua_istable (_L, 3)) {
		for(i = 0 ; i < 2 ; i += 1) {
		    lua_rawgeti (_L, 3, i + 1);
		    self->scale[i] = lua_tonumber (_L, -1);
                
		    lua_pop (_L, 1);
		}
	    }
	} else if (!xstrcmp(k, "density")) {
	    self->density = lua_tonumber (_L, -1);
	} else if (!xstrcmp(k, "diffusion")) {
	    self->diffusion = lua_tonumber (_L, -1);
	} else if (!xstrcmp(k, "mobility")) {
	    self->mobility = lua_tonumber (_L, -1);
	} else if (!xstrcmp(k, "support")) {
	    self->support = lua_tonumber (_L, -1);

	    self->coefficients[0] = 315 / (64 * M_PI * pow (self->support, 9));
	    self->coefficients[1] = 45 / (M_PI * pow (self->support, 6));
	} else if (!xstrcmp(k, "stiffness")) {
	    if(lua_istable (_L, 3)) {
		for(i = 0 ; i < 2 ; i += 1) {
		    lua_rawgeti (_L, 3, i + 1);
		    self->stiffness[i] = lua_tonumber (_L, -1);
                
		    lua_pop (_L, 1);
		}
	    }
	} else if (!xstrcmp(k, "viscosity")) {
	    if(lua_istable (_L, 3)) {
		for(i = 0 ; i < 2 ; i += 1) {
		    lua_rawgeti (_L, 3, i + 1);
		    self->viscosity[i] = lua_tonumber (_L, -1);
                
		    lua_pop (_L, 1);
		}
	    }
	} else if (!xstrcmp(k, "granularity")) {
	    self->granularity = lua_tonumber (_L, -1);
	} else if (!xstrcmp(k, "size")) {
	    self->size = lua_tonumber (_L, -1);

	    if (self->size < 0) {
		self->size = 0;
	    }
	    
	    if (self->generated > self->size) {
		self->generated = self->size;
	    }

	    self->table = (unsigned int *)realloc (self->table,
						   (self->size + 1) *
						   self->hashes *
						   sizeof (unsigned int));

	    self->particles = (double *)realloc (self->particles,
					     self->size * STRIDE *
					     sizeof (double));

	    self->subparticles = (double *)realloc (self->subparticles,
						    self->size *
						    self->clustering *
						    SUBSTRIDE *
						    sizeof (double));
	} else if (!xstrcmp(k, "clustering")) {
	    self->clustering = lua_tonumber (_L, -1);	    

	    self->subparticles = (double *)realloc (self->subparticles,
						    self->size *
						    self->clustering *
						    SUBSTRIDE *
						    sizeof (double));
	} else if (!xstrcmp(k, "hashes")) {
	    self->hashes = lua_tonumber (_L, -1);

	    self->table = (unsigned int *)realloc (self->table,
						   (self->size + 1) *
						   self->hashes *
						   sizeof (unsigned int));
	} else {
	    [super set];
	}
    }
}

-(void)cleanup
{
    int i;

    /* Calculate fluid state (mean density and
       kinetic energy). */
	
    self->state[0] = 0;
    self->state[1] = 0;

    for (i = 0 ; i < self->generated ; i += 1) {
	double rho_i, *v_i;

	if (self->support > 0) {
	    rho_i = DENSITY(i);
	} else {
	    rho_i = self->density / TEMPERATURE(i);
	}
	
	v_i = VELOCITY(i);
	
	self->state[0] += rho_i;
	self->state[1] += 0.5 * (v_i[0] * v_i[0] +
				 v_i[1] * v_i[1] +
				 v_i[2] * v_i[2]);
    }

    self->state[0] /= self->generated;
    self->state[1] /= self->generated;
   
    [super cleanup];
}

-(void)stepBy: (double)dt
{
    double rho_0, k_g, mu, k_s, mu_s, sigma, h, d, D, K_1, K_2;
    int i, j, s, t, q;

    if ((self->iteration % self->granularity) == 0) {
	dt *= self->granularity;

	d = self->scale[0];
	sigma = self->mobility;
	D = self->diffusion;
	k_g = self->stiffness[0];
	k_s = self->stiffness[1];
	mu = self->viscosity[0];
	mu_s = self->viscosity[1];
	rho_0 = self->density;
    
	h = self->support;

	K_1 = self->coefficients[0];
	K_2 = self->coefficients[1];

	if (h > 0) {
	    /* Reset the hash table. */
    
	    for (i = 0 ; i < self->hashes ; i += 1) {
		BUCKET(i)[0] = 0;
	    }

	    /* Gather particles into buckets. */
	
	    for (i = 0 ; i < self->generated ; i += 1) {
		unsigned int n, *b;

		n = hash(POSITION(i)[0] / h,
			 POSITION(i)[1] / h,
			 POSITION(i)[2] / h,
			 self->hashes);
	
		b = BUCKET(n);
	
		b[0] += 1;
		b[b[0]] = i;
	    }

	    /* Calculate the densities. */
    
	    for (i = 0 ; i < self->generated ; i += 1) {
		double *r_i, rho_i;
		int k, l, m;

		r_i = POSITION(i);
		rho_i = 0;

		/* Find the particle's neighborhood. */
	
		k = r_i[0] / h;
		l = r_i[1] / h;
		m = r_i[2] / h;

		/* And iterate it. */
	
		for (s = -1 ; s <= 1 ; s += 1) {
		    for (t = -1 ; t <= 1 ; t += 1) {
			for (q = -1 ; q <= 1 ; q += 1) {
			    unsigned int *b, n;

			    n = hash(k + s, l + t, m + q, self->hashes);
			    b = BUCKET(n);
	    
			    for (j = 1 ; j <= b[0] ; j += 1) {
				double *r_j, dx, dy, dz, delta;

				r_j = POSITION(b[j]);
		
				dx = r_i[0] - r_j[0];
				dy = r_i[1] - r_j[1];
				dz = r_i[2] - r_j[2];
		
				delta = h * h - (dx * dx + dy * dy + dz * dz);

				if (delta > 0) {
				    rho_i += delta * delta * delta;
				}
			    }
			}
		    }
		}

		DENSITY(i) = rho_i * K_1;
	    }
	}
	
	/* Calculate the accelerations and integrate. */
    
	for (i = 0 ; i < self->generated ; i += 1) {
	    double a_i[3], T_i, T_j, rho_i, rho_0i;
	    double *r_i, *v_i;
	    int k, l, m;

	    r_i = POSITION(i);
	    v_i = VELOCITY(i);
	    T_i = TEMPERATURE(i);

	    a_i[0] = 0;
	    a_i[1] = 0;
	    a_i[2] = 0;
	    
	    if (h > 0) {
		rho_i = DENSITY(i);
		rho_0i = rho_0 / T_i;

		k = r_i[0] / h;
		l = r_i[1] / h;
		m = r_i[2] / h;

		/* Find the particle's neighborhood and iterate it. */
	
		for (s = -1 ; s <= 1  ; s += 1) {
		    for (t = -1 ; t <= 1 ; t += 1) {
			for (q = -1 ; q <= 1 ; q += 1) {
			    unsigned int *b, n;
	    
			    n = hash(k + s, l + t, m + q, self->hashes);
			    b = BUCKET(n);
	    
			    for (j = 1 ; j <= b[0] ; j += 1) {
				double *r_j, dx, dy, dz, du, dv, dw, delta, r;

				r_j = POSITION(b[j]);
			    
				dx = r_i[0] - r_j[0];
				dy = r_i[1] - r_j[1];
				dz = r_i[2] - r_j[2];

				r = sqrt (dx * dx + dy * dy + dz * dz);

				if (r < 1e-6) {
				    r = 1e-6;
				}
				
				delta = h - r;

				if (delta > 0) {
				    double *v_j, a_p, a_mu, rho_j, rho_0j;
				
				    v_j = VELOCITY(b[j]);
				    T_j = TEMPERATURE(b[j]);
				    rho_j = DENSITY(b[j]);
				    rho_0j = rho_0 / T_j;

				    du = v_j[0] - v_i[0];
				    dv = v_j[1] - v_i[1];
				    dw = v_j[2] - v_i[2];

				    /* Acceleration due to fluid pressure. */
				
				    a_p = 0.5 / rho_i / rho_j *
					k_g * (rho_i - rho_0i + rho_j - rho_0j) *
					K_2 * delta * delta / r;

				    /* Acceleration due to fluid viscosity. */
				
				    a_mu = mu / rho_i / rho_j * K_2 * delta;
				    
				    a_i[0] += a_p * dx + a_mu * du;
				    a_i[1] += a_p * dy + a_mu * dv;
				    a_i[2] += a_p * dz + a_mu * dw;
				}
			    }
			}
		    }
		}
	    } else {
		rho_i = rho_0 / T_i;
	    }
		
/* /\* 	    printf ("%.10f\n", TEMPERATURE(i)); *\/ */
	    
	    VELOCITY(i)[0] += a_i[0] * dt;
	    VELOCITY(i)[1] += a_i[1] * dt;
	    VELOCITY(i)[2] += a_i[2] * dt;

	    POSITION(i)[0] += VELOCITY(i)[0] * dt;
	    POSITION(i)[1] += VELOCITY(i)[1] * dt;
	    POSITION(i)[2] += VELOCITY(i)[2] * dt;

	    TEMPERATURE(i) *= exp(D * dt);

	    /* Calculate subparticle velocities. */

	    for (j = 0 ; j < self->clustering ; j += 1) {
		double v[3], *r_ij, *v_ij;
		double rsquared, a_p;

		r_ij = SUBPOSITION (i, j);
		v_ij = SUBVELOCITY (i, j);

		rsquared = r_ij[0] * r_ij[0] +
		           r_ij[1] * r_ij[1] +
		           r_ij[2] * r_ij[2];

		if (sigma > 0) {
		    double r[3];
		    
		    r[0] = r_i[0] + r_ij[0];
		    r[1] = r_i[1] + r_ij[1];
		    r[2] = r_i[2] + r_ij[2];

		    get_turbulence_at (r, v);
		}

		/* Accumulate the subparticle pressure, viscosity
		   and turbulnce terms and integrate. */
		
		a_p = -k_s * (rsquared - d / cbrt(rho_i)) / rsquared;

		SUBVELOCITY(i, j)[0] += (a_p * r_ij[0] - mu_s * v_ij[0] +
					 sigma * v[0]) * dt;
		SUBVELOCITY(i, j)[1] += (a_p * r_ij[1] - mu_s * v_ij[1] +
					 sigma * v[1]) * dt;
		SUBVELOCITY(i, j)[2] += (a_p * r_ij[2] - mu_s * v_ij[2] +
					 sigma * v[2]) * dt;

		SUBPOSITION(i, j)[0] += SUBVELOCITY(i, j)[0] * dt;
		SUBPOSITION(i, j)[1] += SUBVELOCITY(i, j)[1] * dt;
		SUBPOSITION(i, j)[2] += SUBVELOCITY(i, j)[2] * dt;
	    }
	}
    }

    [super stepBy: dt];

    self->iteration += 1;
}

-(void) traversePass: (int)pass
{
    double M[16];
    int i, j;

    if (pass == 2) {
	glMatrixMode (GL_MODELVIEW);
	glGetDoublev (GL_MODELVIEW_MATRIX, M);
   
	for (i = 0 ; i < self->generated ; i += 1) {
	    double rho_i, *r_i, *c_i;

	    r_i = POSITION(i);
	    c_i = COLOR(i);

	    if (self->support > 0) {
		rho_i = DENSITY(i);
	    } else {
		rho_i = self->density / TEMPERATURE(i);
	    }

	    glPushMatrix();
	    glTranslated (r_i[0], r_i[1], r_i[2]);
	
	    glBegin (GL_QUADS);
	
	    for (j = 0 ; j < self->clustering ; j += 1) {
		double *r_ij, d_ij, alpha, tau;
	    
		r_ij = SUBPOSITION(i, j);
		d_ij = self->scale[1] / cbrt(rho_i);
		alpha = ALBEDO (i, j);
		tau = EXTINCTION (i, j) * rho_i / self->density;

		if (tau > 1e-2) {
		    glColor4d (c_i[0] * alpha,
			       c_i[1] * alpha,
			       c_i[2] * alpha,
			       tau);
		
		    glTexCoord2d (0, 0);
		    glVertex3d (r_ij[0] - d_ij * (M[0] + M[1]),
				r_ij[1] - d_ij * (M[4] + M[5]),
				r_ij[2] - d_ij * (M[8] + M[9]));

		    glTexCoord2d (1, 0);
		    glVertex3d (r_ij[0] + d_ij * (M[0] - M[1]),
				r_ij[1] + d_ij * (M[4] - M[5]),
				r_ij[2] + d_ij * (M[8] - M[9]));
	
		    glTexCoord2d (1, 1);
		    glVertex3d (r_ij[0] + d_ij * (M[0] + M[1]),
				r_ij[1] + d_ij * (M[4] + M[5]),
				r_ij[2] + d_ij * (M[8] + M[9]));
	
		    glTexCoord2d (0, 1);
		    glVertex3d (r_ij[0] - d_ij * (M[0] - M[1]),
				r_ij[1] - d_ij * (M[4] - M[5]),
				r_ij[2] - d_ij * (M[8] - M[9]));
		}
	    }

	    glEnd();
	    glPopMatrix();
	}
    }

    [super traversePass: pass];
}

@end

