/* Copyright (C) 2009 Papavasileiou Dimitris                             
 *                                                                      
 * This program is free software: you can redistribute it and/or modify 
 * it under the terms of the GNU General Public License as published by 
 * the Free Software Foundation, either version 3 of the License, or    
 * (at your option) any later version.                                  
 *                                                                      
 * This program is distributed in the hope that it will be useful,      
 * but WITHOUT ANY WARRANTY; without even the implied warranty of       
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the        
 * GNU General Public License for more details.                         
 *                                                                      
 * You should have received a copy of the GNU General Public License    
 * along with this program.  If not, see <http://www.gnu.org/licenses/>.
 */

#include <ctype.h>
#include <string.h>
#include <stdlib.h>
#include <lua.h>
#include <lauxlib.h>
#include <GL/gl.h>
#include <GL/glu.h>

#include "elevation.h" 
#include "atmosphere.h" 
#include "earth.h" 
#include "land.h" 

static char *decapitalize (char *s)
{
    s[0] = tolower(s[0]);

    return s;
}

static int generic_tostring(lua_State *L)
{
    id N;

    N = *(id *)lua_touserdata(L, 1);
    lua_pushstring(L, [N name]);
  
    return 1;
}

static int generic_index(lua_State *L)
{
    id N;

    N = *(id *)lua_touserdata(L, 1);
    
    [N get];
    
    return 1;
}

static int generic_newindex(lua_State *L)
{
    id N;

    N = *(id *)lua_touserdata(L, 1);
    [N set];

    return 0;
}

static int generic_gc(lua_State *L)
{
    id N;

    N = *(id *)lua_touserdata(L, 1);

    [N free];

    return 0;
}

static int elevation_len(lua_State *L)
{
    id M;

    M = *(id *)lua_touserdata(L, 1);

    [M query];
    
    return 1;
}

static int elevation_call (lua_State *L)
{
    id object, parent;

    parent = *(id *)lua_touserdata(L, 1);

    /* Create the userdata... */

    object = [[parent copy] init];
    *(id *)lua_newuserdata(L, sizeof(id)) = object;
    
    lua_newtable (L);
    lua_pushstring(L, "__len");
    lua_pushcfunction(L, elevation_len);
    lua_settable(L, -3);
    lua_pushstring(L, "__index");
    lua_pushcfunction(L, generic_index);
    lua_settable(L, -3);
    lua_pushstring(L, "__newindex");
    lua_pushcfunction(L, generic_newindex);
    lua_settable(L, -3);
    lua_pushstring(L, "__tostring");
    lua_pushcfunction(L, generic_tostring);
    lua_settable(L, -3);
    lua_pushstring(L, "__gc");
    lua_pushcfunction(L, generic_gc);
    lua_settable(L, -3);
    lua_pushstring(L, "__reference");
    lua_pushvalue(L, 1);
    lua_settable(L, -3);
    lua_setmetatable(L, -2);
    
    /* ...and initialize it. */

    if(lua_istable(L, 2)) {
	lua_pushnil(L);
	
	while(lua_next(L, 2)) {
	    lua_pushvalue(L, -2);
	    lua_insert(L, -2);
	    lua_settable(L, 3);
	}
    }

    lua_pushstring (L, "userdata");
    lua_gettable (L, LUA_REGISTRYINDEX);
    lua_pushlightuserdata (L, object);
    lua_pushvalue (L, 3);
    lua_settable (L, -3);
    lua_pop(L, 1);

    return 1;
}

static int elevation_tostring(lua_State *L)
{
    lua_pushstring(L, "Elevation geometry foundry");
    
    return 1;
}

static int elevation_gc(lua_State *L)
{
    id object;

    object = *(id *)lua_touserdata(L, 1);

    [object freeBuffers];
    [object free];

    return 0;
}

static int constructelevation(lua_State *L)
{
    id object;
    int i, size[3];
    float resolution[3];
    
    luaL_checktype(L, 1, LUA_TTABLE);

    lua_pushstring(L, "size");
    lua_gettable(L, 1);

    if(lua_istable(L, 1)) {
        for(i = 0 ; i < 3 ; i += 1) {
            lua_rawgeti(L, -1, i + 1);
            size[i] = lua_tonumber(L, -1);
                
            lua_pop(L, 1);
        }
    }
    
    lua_pop(L, 1);

    lua_pushstring(L, "resolution");
    lua_gettable(L, 1);

    if(lua_istable(L, 1)) {
        for(i = 0 ; i < 3 ; i += 1) {
            lua_rawgeti(L, -1, i + 1);
            resolution[i] = lua_tonumber(L, -1);
                
            lua_pop(L, 1);
        }

	resolution[2] /= UCHAR_MAX;
    }
    
    lua_pop(L, 1);

    /* Create and initialize the static userdata. */
    
    object = [[Elevation alloc] initWith: size[0] by: size[1]
			    tilesOfOrder: size[2]
	                   andResolution: resolution];
    
    *(id *)lua_newuserdata(L, sizeof(id)) = object;

    lua_newtable(L);
    lua_pushstring(L, "__tostring");
    lua_pushcfunction(L, elevation_tostring);
    lua_settable(L, -3);
    lua_pushstring(L, "__call");
    lua_pushcfunction(L, elevation_call);
    lua_settable(L, -3);
    lua_pushstring(L, "__gc");
    lua_pushcfunction(L, elevation_gc);
    lua_settable(L, -3);
    lua_setmetatable(L, -2);

    lua_pushstring (L, "userdata");
    lua_gettable (L, LUA_REGISTRYINDEX);
    lua_pushlightuserdata (L, object);
    lua_pushvalue (L, 2);
    lua_settable (L, -3);
    lua_pop(L, 1);

    return 1;
}

static int constructnode (lua_State *L)
{
    id object;
    Class class;

    lua_pushvalue (L, lua_upvalueindex (1));
    class = (Class)lua_touserdata(L, -1);
    lua_pop(L, 1);

    /* Create the userdata... */

    object = [[class alloc] init];
    *(id *)lua_newuserdata(L, sizeof(id)) = object;
    
    lua_newtable (L);
    lua_pushstring(L, "__index");
    lua_pushcfunction(L, (lua_CFunction)generic_index);
    lua_settable(L, -3);
    lua_pushstring(L, "__newindex");
    lua_pushcfunction(L, (lua_CFunction)generic_newindex);
    lua_settable(L, -3);
    lua_pushstring(L, "__tostring");
    lua_pushcfunction(L, (lua_CFunction)generic_tostring);
    lua_settable(L, -3);
    lua_pushstring(L, "__gc");
    lua_pushcfunction(L, (lua_CFunction)generic_gc);
    lua_settable(L, -3);
    lua_setmetatable(L, -2);
    
    /* ...and initialize it. */

    if(lua_istable(L, 1)) {
	lua_pushnil(L);
	
	while(lua_next(L, 1)) {
	    lua_pushvalue(L, -2);
	    lua_insert(L, -2);
	    lua_settable(L, 2);
	}
    }

    lua_pushstring (L, "userdata");
    lua_gettable (L, LUA_REGISTRYINDEX);
    lua_pushlightuserdata (L, object);
    lua_pushvalue (L, 2);
    lua_settable (L, -3);
    lua_pop(L, 1);

    return 1;
}

int luaopen_nature (lua_State *L)
{
    int i;
    
    Class nodes[] = {
	[Atmosphere class], [Earth class], [Land class]
    };
    
    const luaL_Reg nature[] = {
	{"elevation", constructelevation},
	{NULL, NULL}
    };

    luaL_register (L, "nature", nature);
    
    for (i = 0 ; i < sizeof(nodes) / sizeof(nodes[0]) ; i += 1) {
	lua_pushlightuserdata (L, nodes[i]);
	lua_pushcclosure (L, constructnode, 1);
	lua_setfield(L, -2, decapitalize(strdupa([nodes[i] name])));
    }

    return 0;
}
