/* Copyright (C) 2009 Papavasileiou Dimitris                             
 *                                                                      
 * This program is free software: you can redistribute it and/or modify 
 * it under the terms of the GNU General Public License as published by 
 * the Free Software Foundation, either version 3 of the License, or    
 * (at your option) any later version.                                  
 *                                                                      
 * This program is distributed in the hope that it will be useful,      
 * but WITHOUT ANY WARRANTY; without even the implied warranty of       
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the        
 * GNU General Public License for more details.                         
 *                                                                      
 * You should have received a copy of the GNU General Public License    
 * along with this program.  If not, see <http://www.gnu.org/licenses/>.
 */

#include <lua.h>
#include <lauxlib.h>
#include <GL/gl.h>
#include <GL/glu.h>
#include "paper.h"

@implementation Paper

-(Paper *)init
{
    [super init];

    self->horizontal[0] = 0;
    self->horizontal[1] = 1;

    self->vertical[0] = 0;
    self->vertical[1] = 1;

    self->spacing[0] = 1;
    self->spacing[1] = 1;
    
    return self;
}

-(float) translateAbscissa: (float) x
{
    return self->minimum[0] *
	   ((x - self->horizontal[0]) /
	    (self->horizontal[1] - self->horizontal[0]) - 0.5);
}

-(float) translateOrdinate: (float) y
{
    return self->minimum[1] *
	   ((y - self->vertical[0]) /
	    (self->vertical[1] - self->vertical[0]) - 0.5);
}

-(void) get
{
    const char *k;
    int i;
    
    k = lua_tostring (_L, 2);

    if (!xstrcmp(k, "horizontal")) {
        lua_newtable (_L);
        
        for(i = 0; i < 2; i += 1) {
            lua_pushnumber (_L, self->horizontal[i]);
            lua_rawseti (_L, -2, i + 1);
        }
    } else if (!xstrcmp(k, "vertical")) {
        lua_newtable (_L);
        
        for(i = 0; i < 2; i += 1) {
            lua_pushnumber (_L, self->vertical[i]);
            lua_rawseti (_L, -2, i + 1);
        }
    } else if (!xstrcmp(k, "spacing")) {
        lua_newtable (_L);
        
        for(i = 0; i < 2; i += 1) {
            lua_pushnumber (_L, self->spacing[i]);
            lua_rawseti (_L, -2, i + 1);
        }
    } else {
	[super get];
    }
}

-(void) set
{
    const char *k;
    int i;

    k = lua_tostring (_L, 2);

    if (!xstrcmp(k, "horizontal")) {
        if(lua_istable (_L, 3)) {
            for(i = 0 ; i < 2 ; i += 1) {
                lua_rawgeti (_L, 3, i + 1);
                self->horizontal[i] = lua_tonumber (_L, -1);
                
                lua_pop (_L, 1);
            }
        }
    } else if (!xstrcmp(k, "vertical")) {
        if(lua_istable (_L, 3)) {
            for(i = 0 ; i < 2 ; i += 1) {
                lua_rawgeti (_L, 3, i + 1);
                self->vertical[i] = lua_tonumber (_L, -1);
                
                lua_pop (_L, 1);
            }
        }
    } else if (!xstrcmp(k, "spacing")) {
        if(lua_istable (_L, 3)) {
            for(i = 0 ; i < 2 ; i += 1) {
                lua_rawgeti (_L, 3, i + 1);
                self->spacing[i] = lua_tonumber (_L, -1);
                
                lua_pop (_L, 1);
            }
        }
    } else {
	[super set];
    }
}

-(void) transform
{
    self->minimum[0] = self->requested[0];
    self->minimum[1] = self->requested[1];
    
    self->allocated[0] = self->minimum[0] + 2 * self->padding[0];
    self->allocated[1] = self->minimum[1] + 2 * self->padding[1];
    
    if (self->align[0] < 0) {
	self->position[0] += 0.5 * (self->minimum[0] - self->allocated[0]);
    } else if (self->align[0] > 0) {
	self->position[0] += 0.5 * (self->allocated[0] - self->minimum[0]);
    }

    if (self->align[1] < 0) {
	self->position[1] += 0.5 * (self->minimum[1] - self->allocated[1]);
    } else if (self->align[1] > 0) {
	self->position[1] += 0.5 * (self->allocated[1] - self->minimum[1]);
    }

    [super transform];
}

-(void) traversePass: (int)pass
{
    if (pass == 1) {
	double x, y;
	double modelview[16], projection[16], a[3], b[3];
	int viewport[4];

	glMatrixMode (GL_MODELVIEW);
	glPushMatrix();
	glMultMatrixf ([self homogenous]);

	glBlendFunc (GL_SRC_ALPHA, GL_ONE_MINUS_SRC_ALPHA);
	glLineWidth (1);

	glEnable(GL_LINE_SMOOTH);
	glEnable(GL_BLEND);

	glDepthMask (GL_FALSE);
    
	glTranslatef (-0.5 * self->minimum[0],
		      -0.5 * self->minimum[1],
		      0);

	/* Clip the graph to the available space. */
    
	glGetDoublev(GL_MODELVIEW_MATRIX, modelview);
	glGetDoublev(GL_PROJECTION_MATRIX, projection);
	glGetIntegerv(GL_VIEWPORT, viewport);

	gluProject (0, 0, 0,
		    modelview, projection, viewport,
		    &a[0], &a[1], &a[2]);
	gluProject (self->minimum[0], self->minimum[1], 0,
		    modelview, projection, viewport,
		    &b[0], &b[1], &b[2]);

	glScissor (a[0], a[1], b[0] - a[0], b[1] - a[1]);

	/* Draw the paper. */
    
	glColor4fv([self color]);
    
	glBegin(GL_QUADS);

	glVertex2f(0, 0);
	glVertex2f(self->minimum[0], 0);
	glVertex2f(self->minimum[0], self->minimum[1]);
	glVertex2f(0, self->minimum[1]);

	glEnd();

	/* Draw the axes. */
    
	glColor3f(0.1, 0.1, 0.1);
   
	glBegin(GL_LINES);

	glVertex2f(-self->minimum[0] * self->horizontal[0] /
		   (self->horizontal[1] - self->horizontal[0]),
		   0);
	glVertex2f(-self->minimum[0] * self->horizontal[0] /
		   (self->horizontal[1] - self->horizontal[0]),
		   self->minimum[1]);

	glVertex2f(0,
		   -self->minimum[1] * self->vertical[0] /
		   (self->vertical[1] - self->vertical[0]));
	glVertex2f(self->minimum[0],
		   -self->minimum[1] * self->vertical[0] /
		   (self->vertical[1] - self->vertical[0]));

	glEnd();

	/* Draw the grid. */
    
	glColor3f(0.3, 0.3, 0.3);

	glLineStipple (1, 0x0f0f);
	glEnable (GL_LINE_STIPPLE);
    
	glBegin(GL_LINES);

	for (x = self->horizontal[0];
	     x <= self->horizontal[1];
	     x += self->spacing[0]) {

	    if (x != 0) {
		glVertex2f(self->minimum[0] *
			   (x - self->horizontal[0]) /
			   (self->horizontal[1] - self->horizontal[0]),
			   0);
		glVertex2f(self->minimum[0] *
			   (x - self->horizontal[0]) /
			   (self->horizontal[1] - self->horizontal[0]),
			   self->minimum[1]);
	    }
	}
    
	for (y = self->vertical[0];
	     y <= self->vertical[1];
	     y += self->spacing[1]) {
	
	    if (y != 0) {
		glVertex2f(0,
			   self->minimum[1] *
			   (y - self->vertical[0]) /
			   (self->vertical[1] - self->vertical[0]));
		glVertex2f(self->minimum[0],
			   self->minimum[1] *
			   (y - self->vertical[0]) /
			   (self->vertical[1] - self->vertical[0]));
	    }
	}
    
	glEnd();

	glDisable (GL_LINE_STIPPLE);
	glDisable (GL_LINE_SMOOTH);
	glDisable (GL_BLEND);

	glDepthMask (GL_TRUE);
    
	glPopMatrix();

	glEnable (GL_SCISSOR_TEST);
    
	[super traversePass: pass];
    
	glDisable (GL_SCISSOR_TEST);
    } else {
	[super traversePass: pass];
    }
}

@end
