/* Copyright (C) 2009 Papavasileiou Dimitris                             
 *                                                                      
 * This program is free software: you can redistribute it and/or modify 
 * it under the terms of the GNU General Public License as published by 
 * the Free Software Foundation, either version 3 of the License, or    
 * (at your option) any later version.                                  
 *                                                                      
 * This program is distributed in the hope that it will be useful,      
 * but WITHOUT ANY WARRANTY; without even the implied warranty of       
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the        
 * GNU General Public License for more details.                         
 *                                                                      
 * You should have received a copy of the GNU General Public License    
 * along with this program.  If not, see <http://www.gnu.org/licenses/>.
 */

#include <lua.h>
#include <lauxlib.h>
#include <ode/ode.h>

#include "body.h"
#include "system.h"
#include "polyhedron.h"

extern void addContactJoint (Body *a, Body *b, dVector3 pos, dVector3 normal,
			     dReal depth, dReal mu, dReal bounce);

static int addforce (lua_State *L)
{
    Body *object;
    double F[3], p[3];
    int i;

    object = *(Body **)lua_touserdata (L, 1);

    if (lua_istable (L, 2) && [object body]) {
	for(i = 0; i < 3; i += 1) {
	    lua_rawgeti(L, 2, i + 1);
	    F[i] = lua_tonumber (L, -1);

	    lua_pop(L, 1);
	}

	if (lua_istable (L, 3)) {
	    for(i = 0; i < 3; i += 1) {
		lua_rawgeti(L, 3, i + 1);
		p[i] = lua_tonumber (L, -1);

		lua_pop(L, 1);
	    }
	
	    dBodyAddForceAtPos ([object body],
				F[0], F[1], F[2],
				p[0], p[1], p[2]);
	} else {
	    dBodyAddForce ([object body], F[0], F[1], F[2]);
	}

	dBodyEnable([object body]);
    }

    return 0;
}

static int addrelativeforce (lua_State *L)
{
    Body *object;
    double F[3], p[3];
    int i;

    object = *(Body **)lua_touserdata (L, 1);

    if (lua_istable (L, 2) && [object body]) {
	for(i = 0; i < 3; i += 1) {
	    lua_rawgeti(L, 2, i + 1);
	    F[i] = lua_tonumber (L, -1);

	    lua_pop(L, 1);
	}

	if (lua_istable (L, 3)) {
	    for(i = 0; i < 3; i += 1) {
		lua_rawgeti(L, 3, i + 1);
		p[i] = lua_tonumber (L, -1);

		lua_pop(L, 1);
	    }
	
	    dBodyAddRelForceAtRelPos ([object body],
				      F[0], F[1], F[2],
				      p[0], p[1], p[2]);
	} else {
	    dBodyAddRelForce ([object body], F[0], F[1], F[2]);
	}

	dBodyEnable([object body]);
    }

    return 0;
}

static int addtorque (lua_State *L)
{
    Body *object;
    double T[3];
    int i;

    object = *(Body **)lua_touserdata (L, 1);

    if (lua_istable (L, 2) && [object body]) {
	for(i = 0; i < 3; i += 1) {
	    lua_rawgeti(L, 2, i + 1);
	    T[i] = lua_tonumber (L, -1);

	    lua_pop(L, 1);
	}
	
	dBodyAddTorque ([object body], T[0], T[1], T[2]);
	dBodyEnable([object body]);
    }

    return 0;
}

static int addrelativetorque (lua_State *L)
{
    Body *object;
    double T[3];
    int i;

    object = *(Body **)lua_touserdata (L, 1);

    if (lua_istable (L, 2) && [object body]) {
	for(i = 0; i < 3; i += 1) {
	    lua_rawgeti(L, 2, i + 1);
	    T[i] = lua_tonumber (L, -1);

	    lua_pop(L, 1);
	}
	
	dBodyAddRelTorque ([object body], T[0], T[1], T[2]);\
	dBodyEnable([object body]);
    }

    return 0;
}

static void pushmass (lua_State *L, dMass *mass)
{
    int i, j;
    
    lua_newtable(L);

    lua_pushnumber(L, mass->mass);
    lua_rawseti(L, -2, 1);

    lua_newtable(L);

    for(i = 0 ; i < 3 ; i += 1) {
	lua_pushnumber(L, mass->c[i]);
	lua_rawseti(L, -2, i + 1);
    }

    lua_rawseti(L, -2, 2);

    lua_newtable(L);

    for(j = 0 ; j < 3 ; j += 1) {
	for(i = 0 ; i < 3 ; i += 1) {
	    lua_pushnumber(L, mass->I[j * 4 + i]);
	    lua_rawseti(L, -2, j * 3 + i + 1);
	}
    }

    lua_rawseti(L, -2, 3);
}

static int adjustmass (lua_State *L)
{
    dMass mass;
    dReal m;
    int i, j;
    
    dMassSetZero(&mass);
	
    if(lua_istable (_L, 1)) {
	lua_rawgeti (_L, 1, 1);
	mass.mass = lua_tonumber (_L, -1);
	lua_pop (_L, 1);
	
	lua_rawgeti (_L, 1, 2);

	for(i = 0 ; i < 3 ; i += 1) {
	    lua_rawgeti (_L, -1, i + 1);
		
	    mass.c[i] = lua_tonumber (_L, -1);
		
	    lua_pop (_L, 1);
	}
	
	lua_pop (_L, 1);

	lua_rawgeti (_L, 1, 3);
	
	for(i = 0 ; i < 3 ; i += 1) {
	    for(j = 0 ; j < 3 ; j += 1) {
		lua_rawgeti (_L, -1, i * 3 + j + 1);

		mass.I[i * 4 + j] = lua_tonumber (_L, -1);

		lua_pop (_L, 1);
	    }
	}

	lua_pop (_L, 1);
    }

    m = (dReal)luaL_checknumber (L, 2);

    dMassAdjust (&mass, m);
    pushmass (L, &mass);

    return 1;
}

static int spheremass (lua_State *L)
{
    dMass mass;
    dReal rho, r;

    rho = (dReal)luaL_checknumber (L, 1);
    r = (dReal)luaL_checknumber (L, 2);

    dMassSetSphere (&mass, rho, r);
    pushmass (L, &mass);

    return 1;
}

static int boxmass (lua_State *L)
{
    dMass mass;
    dReal rho, a, b, c;

    rho = (dReal)luaL_checknumber (L, 1);
    a = (dReal)luaL_checknumber (L, 2);
    b = (dReal)luaL_checknumber (L, 3);
    c = (dReal)luaL_checknumber (L, 4);

    dMassSetBox (&mass, rho, a, b, c);
    pushmass (L, &mass);

    return 1;
}

static int capsulemass (lua_State *L)
{
    dMass mass;
    dReal rho, l, r;

    rho = (dReal)luaL_checknumber (L, 1);
    r = (dReal)luaL_checknumber (L, 2);
    l = (dReal)luaL_checknumber (L, 3);

    dMassSetCapsule (&mass, rho, 3, r, l);
    
    pushmass (L, &mass);

    return 1;
}

static int polyhedronmass (lua_State *L)
{
    id polyhedron;
    dGeomID geom;
    dMass mass;
    dReal rho;

    polyhedron = *(id *)lua_touserdata (L, 1);
    rho = (dReal)luaL_checknumber (L, 2);
    
    geom = dCreateTriMesh (NULL, [polyhedron data], NULL, NULL, NULL);
    dMassSetTrimesh (&mass, rho, geom);
    dGeomDestroy (geom);

    printf ("center of mass is at (%f, %f, %f)\n", mass.c[0], mass.c[1], mass.c[2]);
    
    mass.c[0] = 0;
    mass.c[1] = 0;
    mass.c[2] = 0;

    pushmass (L, &mass);

    return 1;
}

static int spring (lua_State *L)
{
    dReal k_d, k_s, h;

    k_s = (dReal)luaL_checknumber (L, 1);
    k_d = (dReal)luaL_checknumber (L, 2);

    lua_getglobal (L, "dynamics");
    lua_getfield (L, -1, "stepsize");
    
    h = (dReal)lua_tonumber (L, -1);

    lua_newtable (L);
    lua_pushnumber (L, 1.0 / (h * k_s + k_d));
    lua_rawseti (L, -2, 1);
    lua_pushnumber (L, h * k_s / (h * k_s + k_d));
    lua_rawseti (L, -2, 2);

    return 1;
}

static int collide (lua_State *L)
{
    dContact contact;
    dJointID j;
    Body *a, *b;
    dReal mu, depth, bounce;
    dVector3 pos, normal;
    int i;

    luaL_checktype (L, 1, LUA_TUSERDATA);
    luaL_checktype (L, 2, LUA_TUSERDATA);
    luaL_checktype (L, 3, LUA_TTABLE);
    luaL_checktype (L, 4, LUA_TTABLE);

    depth = (dReal)luaL_checknumber (L, 5);
    mu = (dReal)luaL_checknumber (L, 6);
    bounce = (dReal)luaL_checknumber (L, 7);

    for(i = 0; i < 3; i += 1) {
	lua_rawgeti(L, 3, i + 1);
	pos[i] = lua_tonumber (L, -1);

	lua_rawgeti(L, 4, i + 1);
	normal[i] = lua_tonumber (L, -1);
	
	lua_pop(L, 2);
    }

    a = *(Body **)lua_touserdata (L, 1);
    b = *(Body **)lua_touserdata (L, 2);

/*     { */
/* 	dReal *u; */

/* 	u = dBodyGetLinearVel ([a body]); */
/* 	printf ("v_a = %f, %f, %f\n", u[0], u[1], u[2]); */
	
/* 	u = dBodyGetLinearVel ([b body]); */
/* 	printf ("v_b = %f, %f, %f\n", u[0], u[1], u[2]); */
/*     } */
    
    contact.surface.mode = dContactBounce | dContactApprox1 | dContactFDir1;
    contact.surface.mu = mu;
    contact.surface.bounce = bounce;
    contact.surface.bounce_vel = 0.01;

    for (i = 0 ; i < 3; i += 1) {
	contact.geom.pos[i] = pos[i];
	contact.geom.normal[i] = normal[i];
    }

    dSafeNormalize3 (contact.geom.normal);

/*     printf ("%f, %f, %f\n", */
/* 	    contact.geom.normal[0], */
/* 	    contact.geom.normal[1], */
/* 	    contact.geom.normal[2]); */
    {
	dVector3 u, v, delta;
	dReal *r, *n, ndotdelta;

	r = pos;
	n = normal;
		
	dSetZero (u, 3);
	dSetZero (v, 3);
		
	if ([a body]) {
	    dBodyGetPointVel ([a body], r[0], r[1], r[2], u);
	}
		    
	if ([b body]) {
	    dBodyGetPointVel ([b body], r[0], r[1], r[2], v);
	}

	delta[0] = u[0] - v[0];
	delta[1] = u[1] - v[1];
	delta[2] = u[2] - v[2];
		
	ndotdelta = dDOT(n, delta);
	contact.fdir1[0] = delta[0] - ndotdelta * n[0]; 
	contact.fdir1[1] = delta[1] - ndotdelta * n[1]; 
	contact.fdir1[2] = delta[2] - ndotdelta * n[2];
    }
    
    contact.geom.depth = depth;
    contact.geom.g1 = [a geom];
    contact.geom.g2 = [b geom];

    j = dJointCreateContact (_WORLD, _GROUP, &contact);

    dJointAttach (j, [a body], [b body]);

    return 0;
}

static int joined (lua_State *L)
{
    Body *a, *b;

    luaL_checktype (L, 1, LUA_TUSERDATA);
    luaL_checktype (L, 2, LUA_TUSERDATA);

    a = *(Body **)lua_touserdata (L, 1);
    b = *(Body **)lua_touserdata (L, 2);

    if ([a body] && [b body]) {
	lua_pushboolean (L, dAreConnected([a body], [b body]));
    } else {
	lua_pushboolean (L, 0);
    }
    
    return 1;
}

static int pointfrombody(lua_State *L)
{
    id object;
    dBodyID body;
    dVector3 w;
    dReal b[3];
    int i;

    luaL_checktype (L, 1, LUA_TUSERDATA);
    luaL_checktype (L, 2, LUA_TTABLE);

    object = *(id *)lua_touserdata (L, 1);
    body = [object body];
    
    for (i = 0 ; i < 3 ; i += 1) {
	lua_rawgeti (L, 2, i + 1);
	b[i] = lua_tonumber(L, -1);
	lua_pop(L, 1);
    }

    dBodyGetRelPointPos (body, b[0], b[1], b[2], w);
    
    lua_newtable(L);
    
    for(i = 0; i < 3; i += 1) {
	lua_pushnumber(L, w[i]);
	lua_rawseti(L, -2, i + 1);
    }

    return 1;
}

static int vectorfrombody(lua_State *L)
{
    id object;
    dVector3 w;
    dReal b[3];
    int i;

    luaL_checktype (L, 1, LUA_TUSERDATA);
    luaL_checktype (L, 2, LUA_TTABLE);

    object = *(id *)lua_touserdata (L, 1);
    
    for (i = 0 ; i < 3 ; i += 1) {
	lua_rawgeti (L, 2, i + 1);
	b[i] = lua_tonumber(L, -1);
	lua_pop(L, 1);
    }

    dBodyVectorToWorld ([object body], b[0], b[1], b[2], w);
    
    lua_newtable(L);
    
    for(i = 0; i < 3; i += 1) {
	lua_pushnumber(L, w[i]);
	lua_rawseti(L, -2, i + 1);
    }

    return 1;
}

static int pointtobody(lua_State *L)
{
    id object;
    dVector3 w;
    dReal b[3];
    int i;

    luaL_checktype (L, 1, LUA_TUSERDATA);
    luaL_checktype (L, 2, LUA_TTABLE);

    object = *(id *)lua_touserdata (L, 1);
    
    for (i = 0 ; i < 3 ; i += 1) {
	lua_rawgeti (L, 2, i + 1);
	b[i] = lua_tonumber(L, -1);
	lua_pop(L, 1);
    }

    dBodyGetPosRelPoint ([object body], b[0], b[1], b[2], w);
    
    lua_newtable(L);
    
    for(i = 0; i < 3; i += 1) {
	lua_pushnumber(L, w[i]);
	lua_rawseti(L, -2, i + 1);
    }

    return 1;
}

static int vectortobody(lua_State *L)
{
    id object;
    dVector3 w;
    dReal b[3];
    int i;

    luaL_checktype (L, 1, LUA_TUSERDATA);
    luaL_checktype (L, 2, LUA_TTABLE);

    object = *(id *)lua_touserdata (L, 1);
    
    for (i = 0 ; i < 3 ; i += 1) {
	lua_rawgeti (L, 2, i + 1);
	b[i] = lua_tonumber(L, -1);
	lua_pop(L, 1);
    }

    dBodyVectorFromWorld ([object body], b[0], b[1], b[2], w);
    
    lua_newtable(L);
    
    for(i = 0; i < 3; i += 1) {
	lua_pushnumber(L, w[i]);
	lua_rawseti(L, -2, i + 1);
    }

    return 1;
}

static int pointvelocity(lua_State *L)
{
    id object;
    dVector3 w;
    dReal b[3];
    int i;

    luaL_checktype (L, 1, LUA_TUSERDATA);
    luaL_checktype (L, 2, LUA_TTABLE);

    object = *(id *)lua_touserdata (L, 1);
    
    for (i = 0 ; i < 3 ; i += 1) {
	lua_rawgeti (L, 2, i + 1);
	b[i] = lua_tonumber(L, -1);
	lua_pop(L, 1);
    }

    dBodyGetPointVel ([object body], b[0], b[1], b[2], w);
    
    lua_newtable(L);
    
    for(i = 0; i < 3; i += 1) {
	lua_pushnumber(L, w[i]);
	lua_rawseti(L, -2, i + 1);
    }

    return 1;
}

static int sleep (lua_State *L)
{
    id object;

    luaL_checktype (L, 1, LUA_TUSERDATA);
    
    object = *(id*)lua_touserdata (L, 1);
    if ([object body]) {
	dBodyDisable ([object body]);
    }
    
    return 0;
}

static int wake (lua_State *L)
{
    id object;

    luaL_checktype (L, 1, LUA_TUSERDATA);

    object = *(id*)lua_touserdata (L, 1);
    if ([object body]) {
	dBodyEnable ([object body]);
    }
    
    return 0;
}

int luaopen_physics (lua_State *L)
{
    const luaL_Reg physics[] = {
	{"sleep", sleep},
	{"wake", wake},
	{"addforce", addforce},
	{"addtorque", addtorque},
	{"addrelativeforce", addrelativeforce},
	{"addrelativetorque", addrelativetorque},
	{"adjustmass", adjustmass},
	{"spheremass", spheremass},
	{"boxmass", boxmass},
	{"capsulemass", capsulemass},
	{"polyhedronmass", polyhedronmass},
	{"spring", spring},
	{"joined", joined},
	{"pointfrombody", pointfrombody},
	{"pointtobody", pointtobody},
	{"vectorfrombody", vectorfrombody},
	{"vectortobody", vectortobody},
	{"pointvelocity", pointvelocity},
	{"collide", collide},
	{NULL, NULL}
    };

    luaL_register (L, "physics", physics);

    return 0;
}
