/* Copyright (C) 2009 Papavasileiou Dimitris                             
 *                                                                      
 * This program is free software: you can redistribute it and/or modify 
 * it under the terms of the GNU General Public License as published by 
 * the Free Software Foundation, either version 3 of the License, or    
 * (at your option) any later version.                                  
 *                                                                      
 * This program is distributed in the hope that it will be useful,      
 * but WITHOUT ANY WARRANTY; without even the implied warranty of       
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the        
 * GNU General Public License for more details.                         
 *                                                                      
 * You should have received a copy of the GNU General Public License    
 * along with this program.  If not, see <http://www.gnu.org/licenses/>.
 */

#include <stdlib.h>
#include <lua.h>
#include <lauxlib.h>
#include <GL/gl.h>
#include <GL/glext.h>
#include "geometry.h"
#include "surface.h"
#include "light.h"
#include "ambient.h"
#include "fog.h"

static const GLchar *header =
"//#version 110							    \n"
"//#pragma optionNV(strict on)                                        \n"
"								    \n"
"const int maxLightCount = 8;					    \n"
"const int lightCount = 1;					    \n"
"								    \n"
"uniform struct lightSource {					    \n"
"    vec3 position, intensityConstant, attenuation;		    \n"
"    mat3 frame;						    \n"
"    mat4 shadowMatrix;						    \n"
"    sampler2DShadow shadowMap;				            \n"
"    sampler2D intensityMap;			    	            \n"
"} lightSources[maxLightCount];					    \n"
"						    		    \n"
"uniform struct {				    	    	    \n"
"    mat3 frame;						    \n"
"    vec3 intensityConstant;		    	    	    	    \n"
"    sampler2D intensityMap;			    	            \n"
"} ambience;				    		    	    \n"
"						    		    \n"
"uniform struct {				    	    	    \n"
"    float offset;				    		    \n"
"    vec2 densityConstant;				    	    \n"
"    vec3 colorConstant;		    	    	    	    \n"
"} fog;				    		    	    	    \n"
"								    \n"
"float shadowFactor (lightSource light, vec3 point)	    	    \n"
"{								    \n"
"    vec4 c;						    	    \n"
"			   		     			    \n"
"    c = light.shadowMatrix * vec4(point, 1.0);	    		    \n"
"			   		     			    \n"
"    return float(shadow2DProj (light.shadowMap, c));       	    \n"
"}								    \n"    
"								    \n"
"float attenuationFactor (lightSource light, vec3 point)	    \n"
"{								    \n"
"    vec3 l;					    	    	    \n"
"    float dsquared;					    	    \n"
"			   		     			    \n"
"    l = light.position - point;	    		    	    \n"
"    dsquared = dot(l, l);	    		    		    \n"
"			   		     			    \n"
"    return 1.0 / dot(light.attenuation,		    	    \n"
"                     vec3(1.0, sqrt(dsquared), dsquared));    	    \n"
"}								    \n"    
"								    \n"
"vec3 luminousIntensity (lightSource light, vec3 vector)	    \n"
"{								    \n"
"    vec2 c;						    	    \n"
"			   		     			    \n"
"    c = 0.5 + 0.5 * vec2(dot(vector, light.frame[0]),	    	    \n"
"                         dot(vector, light.frame[1]));  	    \n"
"			   		     			    \n"
"    if(dot(vector, light.frame[2]) < 0.0) {			    \n"
"        return light.intensityConstant + 			    \n"
"               vec3(texture2D(light.intensityMap, c));		    \n"
"    } else {							    \n"
"        return  light.intensityConstant;			    \n"
"    }								    \n"
"}								    \n"    
"								    \n"
"vec3 ambientIntensity(vec3 normal)			   	    \n"
"{						   	 	    \n"    
"    vec2 c;						    	    \n"
"			   		     			    \n"
"    c = 0.5 + 0.5 * vec2(dot(normal, ambience.frame[0]),    	    \n"
"                         dot(normal, ambience.frame[2]));  	    \n"
"			   		     			    \n"
"    return ambience.intensityConstant + 		   	    \n"
"           vec3(texture2D(ambience.intensityMap, c));      	    \n"
"}						  	 	    \n"
"						   	 	    \n"
"vec3 befog(vec3 intensity, vec3 point)				    \n"
"{						   	 	    \n"    
"    float z, f;					    	    \n"
"						   	 	    \n"
"    z = min(point.z + fog.offset, 0.0);		    	    \n"
"    f = exp(fog.densityConstant[0] * z -			    \n"
"            fog.densityConstant[1] * z * z);    	    	    \n"
"						   	 	    \n"
"    return f * intensity + (1.0 - f) * fog.colorConstant.rgb; 	    \n"
"}						  	 	    \n"
"						   	 	    \n"
"#line 1							    \n";

@implementation Surface

-(Surface *)init
{
    int i;

    self->surface.lightCount =
	glGetUniformLocationARB (self->program, "lightCount");

    self->surface.fog.densityConstant =
	glGetUniformLocationARB (self->program, "fog.densityConstant");

    self->surface.fog.offset =
	glGetUniformLocationARB (self->program, "fog.offset");

    self->surface.fog.colorConstant =
	glGetUniformLocationARB (self->program, "fog.colorConstant");

    self->surface.ambience.frame =
	glGetUniformLocationARB (self->program, "ambience.frame");

    self->surface.ambience.intensityMap =
	glGetUniformLocationARB (self->program, "ambience.intensityMap");

    self->surface.ambience.intensityConstant =
	glGetUniformLocationARB (self->program, "ambience.intensityConstant");
    
    for (i = 0 ; i < 8 ; i += 1) {
	char name[64];

	snprintf(name, 64, "lightSources[%d].position", i);
	self->surface.lightSources[i].position =
	    glGetUniformLocationARB (self->program, name);

	snprintf(name, 64, "lightSources[%d].frame", i);
	self->surface.lightSources[i].frame =
	    glGetUniformLocationARB (self->program, name);

	snprintf(name, 64, "lightSources[%d].shadowMatrix", i);
	self->surface.lightSources[i].shadowMatrix =
	    glGetUniformLocationARB (self->program, name);

	snprintf(name, 64, "lightSources[%d].intensityConstant", i);
	self->surface.lightSources[i].intensityConstant =
	    glGetUniformLocationARB (self->program, name);

	snprintf(name, 64, "lightSources[%d].attenuation", i);
	self->surface.lightSources[i].attenuation =
	    glGetUniformLocationARB (self->program, name);

	snprintf(name, 64, "lightSources[%d].shadowMap", i);
	self->surface.lightSources[i].shadowMap =
	    glGetUniformLocationARB (self->program, name);

	snprintf(name, 64, "lightSources[%d].intensityMap", i);
	self->surface.lightSources[i].intensityMap =
	    glGetUniformLocationARB (self->program, name);
    }
    
    [super init];

    return self;
}

-(void) build
{
    self->program = glCreateProgramObjectARB();
}

-(void)attachVertexSource: (const GLchar *)source
{
    GLchar *newsource;

    newsource = alloca(strlen(header) + strlen(source) + 2);

    strcpy (newsource, header);
    strcat (newsource, source);
    
    self->vertex = glCreateShaderObjectARB(GL_VERTEX_SHADER_ARB);
    glShaderSourceARB(self->vertex, 1, (const GLchar **)&newsource, NULL);
    glCompileShaderARB(self->vertex);
}

-(void)attachFragmentSource: (const GLchar *)source
{
    GLchar *newsource;

    newsource = alloca(strlen(header) + strlen(source) + 2);

    strcpy (newsource, header);
    strcat (newsource, source);
    
    self->fragment = glCreateShaderObjectARB(GL_FRAGMENT_SHADER_ARB);
    glShaderSourceARB(self->fragment, 1, (const GLchar **)&newsource, NULL);
    glCompileShaderARB(self->fragment);
}

-(void)link
{
    glAttachObjectARB(self->program, self->vertex);
    glAttachObjectARB(self->program, self->fragment);
    glLinkProgramARB (self->program);
}

-(void)verify
{
    int p, m, n;
    
    glGetObjectParameterivARB (self->program, GL_OBJECT_LINK_STATUS_ARB, &p);

    if (p != GL_TRUE) {
	printf ("\nThe program for %s nodes did not "
		"build properly.\n", [self name]);
    }

    glGetObjectParameterivARB(self->vertex, GL_OBJECT_INFO_LOG_LENGTH_ARB, &n);

    if (n > 1) {
	char buffer[n];

	glGetInfoLogARB(self->vertex, n, &m, buffer);
	printf ("Info log for the `%s' vertex source follows:\n\n%s\n",
		[self name], buffer);
    }

    glGetObjectParameterivARB(self->fragment, GL_OBJECT_INFO_LOG_LENGTH_ARB, &n);

    if (n > 1) {
	char buffer[n];

	glGetInfoLogARB(self->fragment, n, &m, buffer);
	printf ("Info log for the `%s' fragment source follows:\n\n%s\n",
		[self name], buffer);
    }

    glGetObjectParameterivARB(self->program, GL_OBJECT_INFO_LOG_LENGTH_ARB, &n);

    if (n > 1) {
	char buffer[n];

	glGetInfoLogARB(self->program, n, &m, buffer);
	printf ("Info log for the `%s' program follows:\n\n%s\n",
		[self name], buffer);
    }
}

-(void)traversePass: (int) pass
{
    id parent, child;
    GLint unit;
    int n = 0, hasAmbience = 0, hasFog = 0;

    if (pass > 0) {
	for (parent = [self parent] ; parent ; parent = [parent parent]) {
	    for (child = [parent children] ; child ; child = [child sister]) {
		if ([child isMemberOf: [Light class]]) {
		    glUniform3fvARB(self->surface.lightSources[n].position, 1,
				    [child place]);
		    glUniformMatrix3fvARB(self->surface.lightSources[n].frame,
					  1, GL_TRUE, [child frame]);

		    glUniformMatrix4fvARB(self->surface.lightSources[n].shadowMatrix
					  ,
					  1, GL_FALSE, [child shadowMatrix]);

		    glUniform3fvARB(self->surface.lightSources[n].intensityConstant,
				    1, [child intensityConstant]);

		    glUniform3fvARB(self->surface.lightSources[n].attenuation,
				    1, [child attenuation]);

		    glGetIntegerv(GL_ACTIVE_TEXTURE, &unit);
		    glActiveTexture(unit + 1);
		    glBindTexture(GL_TEXTURE_2D, [child intensityMap]);

		    glUniform1iARB (self->surface.lightSources[n].intensityMap,
				    unit + 1 - GL_TEXTURE0);

		    glActiveTexture(unit + 2);
		    glBindTexture(GL_TEXTURE_2D, [child shadowMap]);

		    glUniform1iARB (self->surface.lightSources[n].shadowMap,
				    unit + 2 - GL_TEXTURE0);

		    n += 1;
		} else if ([child isMemberOf: [Ambient class]] && !hasAmbience) {
		    glGetIntegerv(GL_ACTIVE_TEXTURE, &unit);
		    glActiveTexture(unit + 1);
		    glBindTexture(GL_TEXTURE_2D, [child intensityMap]);

		    glUniform1iARB (self->surface.ambience.intensityMap,
				    unit + 1 - GL_TEXTURE0);
            
		    glUniformMatrix3fvARB(self->surface.ambience.frame,
					  1, GL_TRUE, [child frame]);

		    glUniform3fvARB(self->surface.ambience.intensityConstant,
				    1, [child intensityConstant]);

		    hasAmbience += 1;
		} else if ([child isMemberOf: [Fog class]] && !hasFog) {
		    glUniform1fARB(self->surface.fog.offset, [child offset]);
                
		    glUniform2fvARB(self->surface.fog.densityConstant,
				    1, [child densityConstant]);
                
		    glUniform3fvARB(self->surface.fog.colorConstant,
				    1, [child colorConstant]);

		    hasFog += 1;
		}
	    }
	}

	glUniform1iARB (self->surface.lightCount, n);
	glUniform1iARB (self->surface.hasAmbience, hasAmbience);
	glUniform1iARB (self->surface.hasFog, hasFog);
    }
    
    glEnable (GL_DEPTH_TEST);
    glEnable (GL_CULL_FACE);

    [super traversePass: pass];

    glDisable (GL_DEPTH_TEST);
    glDisable (GL_CULL_FACE);
    
    glUseProgramObjectARB(0);
}

@end
