/* Copyright (C) 2009 Papavasileiou Dimitris                             
 *                                                                      
 * This program is free software: you can redistribute it and/or modify 
 * it under the terms of the GNU General Public License as published by 
 * the Free Software Foundation, either version 3 of the License, or    
 * (at your option) any later version.                                  
 *                                                                      
 * This program is distributed in the hope that it will be useful,      
 * but WITHOUT ANY WARRANTY; without even the implied warranty of       
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the        
 * GNU General Public License for more details.                         
 *                                                                      
 * You should have received a copy of the GNU General Public License    
 * along with this program.  If not, see <http://www.gnu.org/licenses/>.
 */

#include <lua.h>
#include <lauxlib.h>
#include "timer.h"

static void calltimedhooks (lua_State *L, void *key, int reference,
			    int tick, double delta, double elapsed)
{
    if (reference != LUA_REFNIL) {
	lua_rawgeti(L, LUA_REGISTRYINDEX, reference);
	    
	if (lua_isfunction (L, -1)) {
	    lua_getfield (L, LUA_REGISTRYINDEX, "userdata");
	    lua_pushlightuserdata (L, key);
	    lua_gettable (L, -2);
	    lua_replace (L, -2);
	    
	    lua_pushnumber (L, tick);
	    lua_pushnumber (L, delta);
	    lua_pushnumber (L, elapsed);
	    lua_call(L, 4, 0);
	} else if (lua_istable (L, -1)) {
	    lua_getfield (L, LUA_REGISTRYINDEX, "userdata");
	    lua_pushnil(L);
	
	    while(lua_next(L, -3)) {
		lua_pushlightuserdata (L, key);
		lua_gettable (L, -4);

		lua_pushnumber (L, tick);
		lua_pushnumber (L, delta);
		lua_pushnumber (L, elapsed);
		lua_call (L, 4, 0);
	    }
	    
	    lua_pop (L, 2);
	} else {
	    lua_pop (L, 1);
	}
    }
}

@implementation Timer

-(Timer *) init
{
    [super init];

    self->tick = LUA_REFNIL;    
    self->period = 1;
    
    return self;
}

-(void) toggle
{
    [super toggle];

    if ([self linked]) {
	clock_gettime (CLOCK_REALTIME, &self->checkpoint);
	self->elapsed = 0;
	self->delta = 0;
	self->count = 0;
    }
}

-(void) transform
{
    struct timespec time;

    clock_gettime (CLOCK_REALTIME, &time);   

    self->delta = time.tv_sec - self->checkpoint.tv_sec +
	         (time.tv_nsec - self->checkpoint.tv_nsec) / 1e9;

    if (self->delta > self->period) {
	self->elapsed += self->delta;
	self->checkpoint = time;
	self->count += 1;

	calltimedhooks (_L, self,
			self->tick, self->count,
			self->delta, self->elapsed);
    }
    
    [super transform];
}

-(void) get
{
    const char *k;

    k = lua_tostring (_L, 2);

    if (!xstrcmp(k, "period")) {
        lua_pushnumber (_L, self->period);
    } else if (!xstrcmp(k, "tick")) {
        lua_rawgeti (_L, LUA_REGISTRYINDEX, self->tick);
    } else  {
	[super get];
    }
}

-(void) set
{    
    const char *k;

    k = lua_tostring (_L, 2);

    if (!xstrcmp(k, "period")) {
        self->period = lua_tonumber (_L, 3);
    } else if (!xstrcmp(k, "tick")) {
        luaL_unref (_L, LUA_REGISTRYINDEX, self->tick);
        self->tick = luaL_ref (_L, LUA_REGISTRYINDEX);
    } else {
	[super set];	
    }
}

@end
