/* Copyright (C) 2009 Papavasileiou Dimitris                             
 *                                                                      
 * This program is free software: you can redistribute it and/or modify 
 * it under the terms of the GNU General Public License as published by 
 * the Free Software Foundation, either version 3 of the License, or    
 * (at your option) any later version.                                  
 *                                                                      
 * This program is distributed in the hope that it will be useful,      
 * but WITHOUT ANY WARRANTY; without even the implied warranty of       
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the        
 * GNU General Public License for more details.                         
 *                                                                      
 * You should have received a copy of the GNU General Public License    
 * along with this program.  If not, see <http://www.gnu.org/licenses/>.
 */

#include <lua.h>
#include <lauxlib.h>
#include <GL/gl.h>
#include "clock.h"

@implementation Clock

-(Clock *)init
{
    [super init];

    self->markings = nil;
    self->hands = nil;
    
    return self;
}

-(void) transform
{
    id child;
    int i, n;
    
    if (self->align[0] < 0) {
    	self->position[0] += 0.5 * (self->minimum[0] - self->allocated[0]);
    } else if (self->align[0] > 0) {
    	self->position[0] += 0.5 * (self->allocated[0] - self->minimum[0]);
    }

    if (self->align[1] < 0) {
    	self->position[1] += 0.5 * (self->minimum[1] - self->allocated[1]);
    } else if (self->align[1] > 0) {
    	self->position[1] += 0.5 * (self->allocated[1] - self->minimum[1]);
    }

    if (self->markings) {
	for(child = [self->markings children];
	    child;
	    child = [child sister]) {
	    double theta;
	
	    i = atoi([child key]);
	    n = (self->range[1] - self->range[0]) / self->spacing[0];

	    theta = (i - 1) * self->spacing[1] *
    	        (self->spread[1] - self->spread[0]) / n +
    	        self->spread[0];

	    [child position][0] = self->radius[1] * sin (theta);
	    [child position][1] = self->radius[1] * cos (theta);
	    [child position][2] = 0;

	    [child allocated][0] = [child measureWidth];
	    [child allocated][1] = [child measureHeight];
	}
    }

    if (self->hands) {
	for(child = [self->hands children];
	    child;
	    child = [child sister]) {
	    double theta;
	
	    i = atoi([child key]);
	    
	    if (i <= self->size) {
		theta = -((self->reading[i - 1] - self->range[0]) /
			 (self->range[1] - self->range[0]) *
			 (self->spread[1] - self->spread[0]) +
			 self->spread[0]);
	    } else {
		theta = -self->spread[0];
	    }

	    [child position][0] = 0;
	    [child position][1] = 0;
	    [child position][2] = 0;

	    [child orientation][0] = cos(theta) * self->radius[0];
	    [child orientation][1] = -sin(theta) * self->radius[0];
	    [child orientation][2] = 0;

	    [child orientation][3] = sin(theta) * self->radius[0];
	    [child orientation][4] = cos(theta) * self->radius[0];
	    [child orientation][5] = 0;

	    [child orientation][6] = 0;
	    [child orientation][7] = 0;
	    [child orientation][8] = 1 * self->radius[0];
	}
    }
    
    [super transform];
}

-(void) traversePass: (int)pass
{
    if (pass == 2) {
	int i, n;

	glMatrixMode (GL_MODELVIEW);
	glPushMatrix();
	glMultMatrixd ([self matrix]);

	glBlendFunc (GL_SRC_ALPHA, GL_ONE_MINUS_SRC_ALPHA);
	glLineWidth (self->thickness);

	glEnable(GL_LINE_SMOOTH);
	glEnable(GL_BLEND);

	glDepthMask (GL_FALSE);

	/* The face. */

	self->background[3] = self->color[3];
	glColor4dv(self->background);

	glBegin (GL_TRIANGLE_FAN);

	glVertex2d (0, 0);
	for (i = 0 ; i <= 32 ; i += 1) {
	    glVertex2d(self->radius[0] * cos (2 * i * M_PI / 32),
		       self->radius[0] * sin (2 * i * M_PI / 32));
	}
   
	glEnd();

	glColor3dv(self->color);
	
	glBegin (GL_LINE_STRIP);

	for (i = 0 ; i <= 32 ; i += 1) {
	    glVertex2d(self->radius[0] * cos (2 * i * M_PI / 32),
		       self->radius[0] * sin (2 * i * M_PI / 32));
	}
   
	glEnd();

	/* Subdivisions. */
	
	glBegin (GL_LINES);

	n = (self->range[1] - self->range[0]) / self->spacing[0];
	for (i = 0 ; i <= n ; i += 1) {
	    double theta, r_0;
	    
	    r_0 = i % self->spacing[1] ? 0.9 : 0.85;
	    theta = i * (self->spread[1] - self->spread[0]) / n +
		    self->spread[0];
	    
	    glVertex2d(0.95 * self->radius[0] * sin (theta),
		       0.95 * self->radius[0] * cos (theta));
	    
	    glVertex2d(r_0 * self->radius[0] * sin (theta),
		       r_0 * self->radius[0] * cos (theta));
	}

	glEnd();

	glDepthMask (GL_TRUE);

	glDisable(GL_BLEND);
	glDisable(GL_LINE_SMOOTH);
    
	glMatrixMode (GL_MODELVIEW);
	glPopMatrix();
    }
    
    [super traversePass: pass];
}

-(void) set
{
    const char *k;

    k = lua_tostring (_L, 2);

    if (!xstrcmp(k, "hands")) {
	if(lua_isuserdata (_L, 3)) {
	    hands = *(id *)lua_touserdata (_L, 3);
	}
	
 	[super set];
   } else if (!xstrcmp(k, "markings")) {
	if(lua_isuserdata (_L, 3)) {
	    markings = *(id *)lua_touserdata (_L, 3);
	}
	
 	[super set];
    } else {
	[super set];
    }
}

@end
