/* Copyright (C) 2009 Papavasileiou Dimitris                             
 *                                                                      
 * This program is free software: you can redistribute it and/or modify 
 * it under the terms of the GNU General Public License as published by 
 * the Free Software Foundation, either version 3 of the License, or    
 * (at your option) any later version.                                  
 *                                                                      
 * This program is distributed in the hope that it will be useful,      
 * but WITHOUT ANY WARRANTY; without even the implied warranty of       
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the        
 * GNU General Public License for more details.                         
 *                                                                      
 * You should have received a copy of the GNU General Public License    
 * along with this program.  If not, see <http://www.gnu.org/licenses/>.
 */

#include <ode/ode.h>
#include <lua.h>
#include <lualib.h>
#include <lauxlib.h>

#include "contact.h"
#include "body.h"

@implementation Contact

-(Joint *) init
{
    char *list[] = {
	"anchor", "axes", "depth", "elasticity", "force", "friction",
	"restitution"
    };

    self->joint = dJointCreateContact (_WORLD, NULL, &self->contact);

    self->contact.geom.pos[0] = 0;
    self->contact.geom.pos[1] = 0;

    self->contact.geom.pos[2] = 0;

    self->contact.fdir1[0] = 1;
    self->contact.fdir1[1] = 0;
    self->contact.fdir1[2] = 0;

    self->contact.geom.normal[0] = 0;
    self->contact.geom.normal[1] = 0;
    self->contact.geom.normal[2] = 1;

    self->contact.geom.depth = 0;

    self->contact.surface.mode = 0;

    self->contact.surface.bounce = 0;
    self->contact.surface.bounce_vel = 0.01;
    
    self->contact.surface.mu = 0;
    self->contact.surface.mu2 = 0;
    
    self->contact.surface.soft_cfm = 0;
    self->contact.surface.soft_erp = 0;

    self->contact.geom.g1 = 0;
    self->contact.geom.g2 = 0;

    [super init];
    [self add: sizeof (list) / sizeof (char *) Properties: list];

    return self;
}

-(void) get
{
    const char *k;
    int i, j;
    
    k = lua_tostring (_L, -1);

    if (!xstrcmp(k, "anchor")) {
        lua_newtable (_L);
        
        for(j = 0; j < 3; j += 1) {
            lua_pushnumber (_L, self->contact.geom.pos[j]);
            lua_rawseti (_L, -2, j + 1);
        }
    } else if (!xstrcmp(k, "axes")) {
	lua_newtable (_L);
        
	for(j = 0 ; j < 2 ; j += 1) {
	    dReal *a;
	    
	    a = j == 0 ? self->contact.fdir1 : self->contact.geom.normal;
		
	    lua_newtable (_L);
        
	    for(i = 0; i < 3; i += 1) {
		lua_pushnumber (_L, a[i]);
		lua_rawseti (_L, -2, i + 1);
	    }

	    lua_rawseti (_L, -2, j + 1);
	}
    } else if (!xstrcmp(k, "friction")) {
	if (self->contact.surface.mode & dContactApprox1) {
	    if (self->contact.surface.mode & dContactMu2) {
		lua_newtable (_L);
		
		lua_pushnumber (_L, self->contact.surface.mu);
		lua_rawseti (_L, -2, 1);
		
		lua_pushnumber (_L, self->contact.surface.mu2);
		lua_rawseti (_L, -2, 2);
	    } else {
		lua_pushnumber (_L, self->contact.surface.mu);
	    }
	} else {
	    lua_pushnil (_L);
	}
    } else if (!xstrcmp(k, "force")) {
	if (self->contact.surface.mode & dContactApprox1) {
	    lua_pushnil (_L);
	} else {
	    if (self->contact.surface.mode & dContactMu2) {
		lua_newtable (_L);
		
		lua_pushnumber (_L, self->contact.surface.mu);
		lua_rawseti (_L, -2, 1);
		
		lua_pushnumber (_L, self->contact.surface.mu2);
		lua_rawseti (_L, -2, 2);
	    } else {
		lua_pushnumber (_L, self->contact.surface.mu);
	    }
	}
    } else if (!xstrcmp(k, "elasticity")) {
	if (self->contact.surface.mode & dContactSoftCFM) {
	    lua_newtable (_L);
		
	    lua_pushnumber (_L, self->contact.surface.soft_cfm);
	    lua_rawseti (_L, -2, 1);
		
	    lua_pushnumber (_L, self->contact.surface.soft_erp);
	    lua_rawseti (_L, -2, 2);
	} else {
	    lua_pushnil (_L);
	}
    } else if (!xstrcmp(k, "depth")) {
	lua_pushnumber (_L, self->contact.geom.depth);
    } else if (!xstrcmp(k, "restitution")) {
	if (self->contact.surface.mode & dContactBounce) {	    
	    lua_pushnumber (_L, self->contact.surface.bounce);
	} else {
	    lua_pushnil (_L);
	}	    
    } else {
	[super get];
    }
}

-(void) set
{
    const char *k;
    int i, j;
    
    k = lua_tostring (_L, -2);

    if (!xstrcmp(k, "anchor")) {
        if(lua_istable (_L, 3)) {
            for(j = 0 ; j < 3 ; j += 1) {
                lua_rawgeti (_L, 3, j + 1);
                self->contact.geom.pos[j] = lua_tonumber (_L, -1);
                
                lua_pop (_L, 1);
            }
        }
    } else if (!xstrcmp(k, "axes")) {
        if(lua_istable (_L, 3)) {
	    for(j = 0 ; j < 2 ; j += 1) {
		dReal *a;

		a = j == 0 ? self->contact.fdir1 : self->contact.geom.normal;
		
		lua_rawgeti (_L, 3, j + 1);
		
		if(lua_istable (_L, -1)) {
		    for(i = 0 ; i < 3 ; i += 1) {
			lua_rawgeti (_L, -1, i + 1);
			a[i] = lua_tonumber (_L, -1);
                
			lua_pop (_L, 1);
		    }

		    dSafeNormalize3 (a);
		}

		lua_pop (_L, 1);
	    }

	}
    } else if (!xstrcmp(k, "friction")) {
        if(lua_istable (_L, 3)) {
	    self->contact.surface.mode |= dContactApprox1 | dContactMu2 |  dContactFDir1;

	    lua_rawgeti (_L, 3, 1);
	    self->contact.surface.mu = lua_tonumber (_L, -1);

	    lua_rawgeti (_L, 3, 2);
	    self->contact.surface.mu2 = lua_tonumber (_L, -1);

	    lua_pop (_L, 2);
	} else if(lua_tonumber (_L, 3)) {
	    self->contact.surface.mode |= dContactApprox1 |  dContactFDir1;
	    self->contact.surface.mode &= ~dContactMu2;

	    self->contact.surface.mu = lua_tonumber (_L, 3);
	} else {
	    self->contact.surface.mode &= ~(dContactApprox1 | dContactMu2 |  dContactFDir1);
	}	    
    } else if (!xstrcmp(k, "force")) {
        if(lua_istable (_L, 3)) {
	    self->contact.surface.mode |= dContactMu2 |  dContactFDir1;
	    self->contact.surface.mode &= ~dContactApprox1;

	    lua_rawgeti (_L, 3, 1);
	    self->contact.surface.mu = lua_tonumber (_L, -1);

	    lua_rawgeti (_L, 3, 2);
	    self->contact.surface.mu2 = lua_tonumber (_L, -1);

	    lua_pop (_L, 2);
	} else if(lua_tonumber (_L, 3)) {
	    self->contact.surface.mode |= dContactFDir1;
	    self->contact.surface.mode &= ~dContactApprox1;
	    self->contact.surface.mode &= ~dContactMu2;

	    self->contact.surface.mu = lua_tonumber (_L, 3);
	} else {
	    self->contact.surface.mode &= ~(dContactApprox1 | dContactMu2 |  dContactFDir1);
	}
    } else if (!xstrcmp(k, "elasticity")) {
        if(lua_istable (_L, 3)) {
	    self->contact.surface.mode |= dContactSoftERP;
	    self->contact.surface.mode |= dContactSoftCFM;

	    lua_rawgeti (_L, 3, 1);
	    self->contact.surface.soft_cfm = lua_tonumber (_L, -1);

	    lua_rawgeti (_L, 3, 2);
	    self->contact.surface.soft_erp = lua_tonumber (_L, -1);

	    lua_pop (_L, 2);
	} else {
	    self->contact.surface.mode &= ~dContactSoftERP;
	    self->contact.surface.mode &= ~dContactSoftCFM;
	}	    
    } else if (!xstrcmp(k, "depth")) {
	self->contact.geom.depth = lua_tonumber (_L, 3);
    } else if (!xstrcmp(k, "restitution")) {
	if (lua_isnumber (_L, 3)) {
	    self->contact.surface.mode |= dContactBounce;
	    self->contact.surface.bounce = lua_tonumber (_L, 3);
	} else {
	    self->contact.surface.mode &= ~dContactBounce;
	}	    
    } else {
	[super set];
    }
}

-(void) stepBy: (double) h at: (double) t
{
    dBodyID a, b;

    a = dJointGetBody ([self joint], 0);
    b = dJointGetBody ([self joint], 1);

    dJointDestroy ([self joint]);
    self->joint = dJointCreateContact (_WORLD, NULL, &self->contact);
    
    dJointSetFeedback ([self joint], &self->feedback);
    dJointAttach ([self joint], a, b);

    [super stepBy: h at: t];
}

@end
