/* Copyright (C) 2009 Papavasileiou Dimitris                             
 *                                                                      
 * This program is free software: you can redistribute it and/or modify 
 * it under the terms of the GNU General Public License as published by 
 * the Free Software Foundation, either version 3 of the License, or    
 * (at your option) any later version.                                  
 *                                                                      
 * This program is distributed in the hope that it will be useful,      
 * but WITHOUT ANY WARRANTY; without even the implied warranty of       
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the        
 * GNU General Public License for more details.                         
 *                                                                      
 * You should have received a copy of the GNU General Public License    
 * along with this program.  If not, see <http://www.gnu.org/licenses/>.
 */

#include <stdlib.h>
#include <math.h>
#include <lua.h>
#include <lauxlib.h>
#include <GL/gl.h>
#include <GL/glu.h>

#include "meteorology.h"
#include "body.h"
#include "fabric.h"

static void accumulate(float *p, float *q, double *u, double *v,
		       double *F_p, double *F_q, double A_p, double A_q,
		       double k_s, double k_d, double n_0)
{
    double n, F, l[3], dldt[3];

    l[0] = p[0] - q[0];
    l[1] = p[1] - q[1];
    l[2] = p[2] - q[2];

    dldt[0] = u[0] - v[0];
    dldt[1] = u[1] - v[1];
    dldt[2] = u[2] - v[2];
    
    /* Normalize l. */
    
    n = sqrt(l[0] * l[0] + l[1] * l[1] + l[2] * l[2]);

    l[0] /= n;
    l[1] /= n;
    l[2] /= n;
    
    /*                                                                  
     * Spring force magnitude:
     *                                            .
     * F = A * k_s * (||l|| - ||l_0||) + k_d * (l l) */
    
    F = 0.5 * (A_p + A_q) * k_s * (n - n_0) +
	k_d * (l[0] * dldt[0] + l[1] * dldt[1] + l[2] * dldt[2]);

    /* Accumulate the force pair. */
    
    F_p[0] += -F * l[0];
    F_p[1] += -F * l[1];
    F_p[2] += -F * l[2];

    F_q[0] += F * l[0];
    F_q[1] += F * l[1];
    F_q[2] += F * l[2];
}

@implementation Fabric

-(Fabric *) init
{
    char *list[] = {
	"bend", "density", "drag", "granularity", "mobility",
	"shear", "stretch"
    };

    /* Initialize the object. */
    
    [super init];
    [self add: sizeof (list) / sizeof (char *) Properties: list];

    self->granularity = 1;

    self->body = NULL;
    self->reset = 1;
    self->firststep = 1;
    
    self->density = 1;
    self->mobility = 0;
    self->drag = 1;
    
    self->stiffness[0] = 25000;
    self->stiffness[1] = 2000;
    self->stiffness[2] = 1000;

    self->damping[0] = 750;
    self->damping[1] = 100;
    self->damping[2] = 50;
    
    self->vertices = (float *)malloc(self->size[0] * 3 * sizeof(float));
    self->velocity = (double *)malloc(self->size[0] * 3 * sizeof(double));

    return self;
}

-(Fabric *) initWithVertices: (float *)floats
               uvcoordinates: (float *) morefloats
		     indices: (unsigned int *)ints
	      stretchSprings: (int *)stretchNodes
	        shearSprings: (int *)shearNodes
		 bendSprings: (int *)bendNodes
		    andNodes: (int *)knots
		      ofSize: (int[6]) n
{
    int i, s, t, q;
    
    [super init];

    /* Allocate and copy nodes springs etc. */

    self->rest = (float *)malloc(n[0] * 3 * sizeof(float));
    self->uv = (float *)malloc(n[0] * 2 * sizeof(float));
    self->normals = (float *)malloc(n[0] * 3 * sizeof(float));
    self->indices = (unsigned int *)malloc(n[1] * sizeof(unsigned int));
    self->areas = (double *)malloc(n[0] * sizeof(double));
    self->stretchLengths = (double *)malloc(n[2] * sizeof(double));
    self->shearLengths = (double *)malloc(n[3] * sizeof(double));
    self->bendLengths = (double *)malloc(n[4] * sizeof(double));
    self->stretchSprings = (int *)malloc(n[2] * 2 * sizeof(int));
    self->shearSprings = (int *)malloc(n[3] * 2 * sizeof(int));
    self->bendSprings = (int *)malloc(n[4] * 2 * sizeof(int));
    self->nodes = (int *)malloc(n[5] * sizeof(int));
    self->internal = (double *)malloc(n[0] * 3 * sizeof(double));
    self->external = (double *)malloc(n[0] * 3 * sizeof(double));
    
    memcpy (self->rest, floats, n[0] * 3 * sizeof(float));
    memcpy (self->uv, morefloats, n[0] * 2 * sizeof(float));
    memcpy (self->indices, ints, n[1] * sizeof(unsigned int));
    memcpy (self->stretchSprings, stretchNodes, n[2] * 2 * sizeof(int));
    memcpy (self->shearSprings, shearNodes, n[3] * 2 * sizeof(int));
    memcpy (self->bendSprings, bendNodes, n[4] * 2 * sizeof(int));
    memcpy (self->nodes, knots, n[5] * sizeof(int));
    memcpy (self->size, n, 6 * sizeof(int));

    /* Calculate the node areas. */

    memset (self->areas, 0, n[0] * sizeof(double));
    
    for (i = 0 ; i < self->size[1] / 3 ; i += 1) {
    	double A, u[3], v[3], n[3];
	
    	s = self->indices[3 * i];
    	t = self->indices[3 * i + 1];
    	q = self->indices[3 * i + 2];
	    
    	/* Calculate the triangle's normal. */
            
    	v[0] = self->rest[3 * t + 0] - self->rest[3 * s + 0];
    	v[1] = self->rest[3 * t + 1] - self->rest[3 * s + 1];
    	v[2] = self->rest[3 * t + 2] - self->rest[3 * s + 2];

    	u[0] = self->rest[3 * q + 0] - self->rest[3 * s + 0];
    	u[1] = self->rest[3 * q + 1] - self->rest[3 * s + 1];
    	u[2] = self->rest[3 * q + 2] - self->rest[3 * s + 2];
           
    	n[0] = u[1] * v[2] - u[2] * v[1];
    	n[1] = u[2] * v[0] - u[0] * v[2];
    	n[2] = u[0] * v[1] - u[1] * v[0];

    	/* And take the magnitude to calculate one-third
    	   of the triangle's area. */
	
    	A = sqrt(n[0] * n[0] + n[1] * n[1] + n[2] * n[2]) / 6;
	
    	/* Accumulate the trinagle's normal on each of its vertices. */

    	self->areas[s] += A;
    	self->areas[t] += A;
    	self->areas[q] += A;
    }
    
    /* Calculate the rest lengths for the springs. */
    
    for (i = 0 ; i < n[2] ; i += 1) {
    	float *v_0, *v_1;

    	v_0 = &floats[3 * stretchNodes[2 * i]];
    	v_1 = &floats[3 * stretchNodes[2 * i + 1]];
	
    	self->stretchLengths[i] = sqrt((v_0[0] - v_1[0]) * (v_0[0] - v_1[0]) +
    				       (v_0[1] - v_1[1]) * (v_0[1] - v_1[1]) +
    				       (v_0[2] - v_1[2]) * (v_0[2] - v_1[2]));
    }
    
    for (i = 1 ; i < n[3] ; i += 1) {
    	float *v_0, *v_1;

    	v_0 = &floats[3 * shearNodes[2 * i]];
    	v_1 = &floats[3 * shearNodes[2 * i + 1]];
	
    	self->shearLengths[i] = sqrt((v_0[0] - v_1[0]) * (v_0[0] - v_1[0]) +
    				     (v_0[1] - v_1[1]) * (v_0[1] - v_1[1]) +
    				     (v_0[2] - v_1[2]) * (v_0[2] - v_1[2]));
    }
    
    for (i = 0 ; i < n[4] ; i += 1) {
    	float *v_0, *v_1;

    	v_0 = &floats[3 * bendNodes[2 * i]];
    	v_1 = &floats[3 * bendNodes[2 * i + 1]];
	
    	self->bendLengths[i] = sqrt((v_0[0] - v_1[0]) * (v_0[0] - v_1[0]) +
    				    (v_0[1] - v_1[1]) * (v_0[1] - v_1[1]) +
    				    (v_0[2] - v_1[2]) * (v_0[2] - v_1[2]));
    }

    return self;
}

-(void) freeBuffers
{
    free (self->rest);
    free (self->normals);
    free (self->uv);
    free (self->areas);
    free (self->internal);
    free (self->external);
    free (self->indices);
    free (self->nodes);
    free (self->stretchLengths);
    free (self->shearLengths);
    free (self->bendLengths);
    free (self->stretchSprings);
    free (self->shearSprings);
    free (self->bendSprings);
}

-(void) free
{
    free (self->vertices);
    free (self->velocity);

    [super free];
}

-(int) vertices
{
    return self->size[0];
}

-(int) indices
{
    return self->size[1];
}

-(void) toggle
{
    id ancestor;
    
    [super toggle];

    self->body = NULL;
    
    if ([self linked]) {
	self->reset = 1;

	for (ancestor = [self parent];
	     ancestor && ![ancestor isKindOf: [Body class]];
	     ancestor = [ancestor parent]);
	
	self->body = [ancestor body];
    }
}

-(void) transform
{
    [self transformAsRoot];
}

-(void) stepBy: (double) h at: (double) t
{
    const dReal *R, *r;
    double *s;
    float *p, *q;
    
    double h_0;
    int i, j;
	    
    R = dBodyGetRotation (self->body);
    r = dBodyGetPosition (self->body);

    /* Either position just the nodes or the whole mesh relative
       to its parent body. */
    
    for(i = 0 ; i < (self->reset ? self->size[0] : self->size[5]) ; i += 1) {
	p = &self->rest[3 * (self->reset ? i : self->nodes[i])];
	q = &self->vertices[3 * (self->reset ? i : self->nodes[i])];
	s = &self->velocity[3 * (self->reset ? i : self->nodes[i])];
	
	q[0] = R[0] * p[0] + R[1] * p[1] + R[2] * p[2] + r[0];
	q[1] = R[4] * p[0] + R[5] * p[1] + R[6] * p[2] + r[1];
	q[2] = R[8] * p[0] + R[9] * p[1] + R[10] * p[2] + r[2];

	s[0] = 0; s[1] = 0; s[2] = 0;
    }

    self->reset = 0;

    /* Some of these lookups are expensive and updating
       these forces per step probably isn't worth it.
       Just do it once per frame. */
    
    if (self->firststep) {
    	dVector3 g;
	
    	dWorldGetGravity (_WORLD, g);

    	for(i = 0 ; i < self->size[0] ; i += 1) {
    	    double m, rho, V, A, *v, tau[3];
	    float *r;

	    A = self->areas[i];
    	    m = self->density * A;
	    r = &self->vertices[3 * i];
	    v = &self->velocity[3 * i];
	
    	    /* Calculate external forces, turbulence and gravity. */
	
    	    if (self->mobility > 0) {
    	    	get_turbulence_at (r[0], r[1], r[2], t, tau);
    	    }

	    rho = get_density_at (v[3]);
	    V = sqrt (v[0] * v[0] + v[1] * v[1] + v[2] * v[2]);

    	    self->external[3 * i + 0] = self->mobility * tau[0] + m * g[0] -
		0.5 * rho * V * A * self->drag * v[0];
    	    self->external[3 * i + 1] = self->mobility * tau[1] + m * g[1] -
		0.5 * rho * V * A * self->drag * v[1];
    	    self->external[3 * i + 2] = self->mobility * tau[2] + m * g[2] -
		0.5 * rho * V * A * self->drag * v[2];
    	}

	self->firststep = 0;
    }

    for (j = 0, h_0 = h / self->granularity; j < self->granularity ; j += 1) {
	memset (self->internal, 0, self->size[0] * 3 * sizeof (double));
    
	/* Accumulate all spring forces. */
    
	for(i = 0 ; i < self->size[2] ; i += 1) {
	    int e_0, e_1;

	    e_0 = self->stretchSprings[2 * i];
	    e_1 = self->stretchSprings[2 * i + 1];
	
	    accumulate(&self->vertices[3 * e_0], &self->vertices[3 * e_1],
		       &self->velocity[3 * e_0], &self->velocity[3 * e_1],
		       &self->internal[3 * e_0], &self->internal[3 * e_1],
		       self->areas[e_0], self->areas[e_1],
		       self->stiffness[0], self->damping[0],
		       self->stretchLengths[i]);
	}

	for(i = 0 ; i < self->size[3] ; i += 1) {
	    int e_0, e_1;

	    e_0 = self->shearSprings[2 * i];
	    e_1 = self->shearSprings[2 * i + 1];
	
	    accumulate(&self->vertices[3 * e_0], &self->vertices[3 * e_1],
		       &self->velocity[3 * e_0], &self->velocity[3 * e_1],
		       &self->internal[3 * e_0], &self->internal[3 * e_1],
		       self->areas[e_0], self->areas[e_1],
		       self->stiffness[1], self->damping[1],
		       self->shearLengths[i]);
	}

	for(i = 0 ; i < self->size[4] ; i += 1) {
	    int e_0, e_1;

	    e_0 = self->bendSprings[2 * i];
	    e_1 = self->bendSprings[2 * i + 1];
	
	    accumulate(&self->vertices[3 * e_0], &self->vertices[3 * e_1],
		       &self->velocity[3 * e_0], &self->velocity[3 * e_1],
		       &self->internal[3 * e_0], &self->internal[3 * e_1],
		       self->areas[e_0], self->areas[e_1],
		       self->stiffness[2], self->damping[2],
		       self->bendLengths[i]);
	}

	/* Take care of attached nodes. */
    
	for(i = 0 ; i < self->size[5] ; i += 1) {
	    /* If the fabric is attached to a body propagate
	       the forces applied to the node to the body. */
	    
	    if (self->body) {
		double c;

		c = 1.0 / self->granularity;

		dBodyAddRelForceAtPos (self->body,
				       c * self->internal[3 * self->nodes[i] + 0],
				       c * self->internal[3 * self->nodes[i] + 1],
				       c * self->internal[3 * self->nodes[i] + 2],
				       self->vertices[3 * self->nodes[i] + 0],
				       self->vertices[3 * self->nodes[i] + 1],
				       self->vertices[3 * self->nodes[i] + 2]);	    }
	    
	    /* Make sure nodes stay put. */
	    
	    self->internal[3 * self->nodes[i] + 0] = 0;
	    self->internal[3 * self->nodes[i] + 1] = 0;
	    self->internal[3 * self->nodes[i] + 2] = 0;

	    self->external[3 * self->nodes[i] + 0] = 0;
	    self->external[3 * self->nodes[i] + 1] = 0;
	    self->external[3 * self->nodes[i] + 2] = 0;
	}

	/* And integrate. */

	for(i = 0 ; i < self->size[0] ; i += 1) {
	    double *a, *F_i, *F_e, m;
	    float *b;

	    m = self->density * self->areas[i];
	
	    a = &self->velocity[3 * i];
	    b = &self->vertices[3 * i];
	    F_i = &self->internal[3 * i];
	    F_e = &self->external[3 * i];

	    /* And step the system. */

	    a[0] += (F_i[0] + F_e[0]) / m * h_0;
	    a[1] += (F_i[1] + F_e[1]) / m * h_0;
	    a[2] += (F_i[2] + F_e[2]) / m * h_0;

	    b[0] += a[0] * h_0;
	    b[1] += a[1] * h_0;
	    b[2] += a[2] * h_0;
	}
    }
    
    [super stepBy: h at: t];
}

-(void) prepare
{
    double u[3], v[3], n[3];
    int i, s, t, q;
	
    memset (self->normals, 0, self->size[0] * 3 * sizeof(float));
	
    /* Update the vertex normals for the mesh. */
	
    for (i = 0 ; i < self->size[1] / 3 ; i += 1) {
	s = self->indices[3 * i];
	t = self->indices[3 * i + 1];
	q = self->indices[3 * i + 2];
	    
	/* Calculate the triangle's normal. */
            
	v[0] = self->vertices[3 * t + 0] - self->vertices[3 * s + 0];
	v[1] = self->vertices[3 * t + 1] - self->vertices[3 * s + 1];
	v[2] = self->vertices[3 * t + 2] - self->vertices[3 * s + 2];

	u[0] = self->vertices[3 * q + 0] - self->vertices[3 * s + 0];
	u[1] = self->vertices[3 * q + 1] - self->vertices[3 * s + 1];
	u[2] = self->vertices[3 * q + 2] - self->vertices[3 * s + 2];
           
	n[0] = u[1] * v[2] - u[2] * v[1];
	n[1] = u[2] * v[0] - u[0] * v[2];
	n[2] = u[0] * v[1] - u[1] * v[0];

	/* Accumulate the trinagle's normal on each of its vertices. */
	    
	self->normals[3 * s + 0] += n[0];
	self->normals[3 * s + 1] += n[1];
	self->normals[3 * s + 2] += n[2];
            
	self->normals[3 * t + 0] += n[0];
	self->normals[3 * t + 1] += n[1];
	self->normals[3 * t + 2] += n[2];
            
	self->normals[3 * q + 0] += n[0];
	self->normals[3 * q + 1] += n[1];
	self->normals[3 * q + 2] += n[2];
    }
    
    [super prepare];
}

-(void) traversePass: (int)pass
{
    if (pass < 2) {
	int i, j, k;

	glMatrixMode (GL_MODELVIEW);
	glPushMatrix();
	glMultMatrixd ([self matrix]);
	
    	glEnable (GL_DEPTH_TEST);
    	glEnable (GL_CULL_FACE);

	glBegin (GL_TRIANGLES);
	    
	if (pass == 0) {
	    for (i = 0 ; i < self->size[1] ; i += 1) {
		j = self->indices[i];
		    
		glVertex3fv (&self->vertices[3 * j]);
	    }
	} else {
	    /* Draw one side. */
		
	    for (i = 0 ; i < self->size[1] / 3 ; i += 1) {
		for (j = 0 ; j < 3 ; j += 1) {
		    k = self->indices[3 * i + j];

		    glNormal3fv (&self->normals[3 * k]);
		    glTexCoord2fv (&self->uv[2 * k]);
		    glVertex3fv (&self->vertices[3 * k]);
		}
	    }

	    /* Then the other with reversed winding and normals. */
		
	    for (i = 0 ; i < self->size[1] / 3 ; i += 1) {
		for (j = 2 ; j >= 0 ; j -= 1) {
		    k = self->indices[3 * i + j];

		    glNormal3f (-self->normals[3 * k],
				-self->normals[3 * k + 1],
				-self->normals[3 * k + 2]);
		    glTexCoord2fv (&self->uv[2 * k]);
		    glVertex3fv (&self->vertices[3 * k]);
		}
	    }
	}

	glEnd();

#if 0
	if (pass == 1) {
	    int i;

	    glUseProgramObjectARB(0);

	    glColor3f (1, 0, 0);
	    glLineWidth (1);
	    glBegin (GL_LINES);

	    for (i = 0 ; i < 2 * self->size[2] ; i += 1) {
		glVertex3fv (&self->vertices[3 * self->stretchSprings[i]]);
	    }

	    glEnd();

	    glColor3f (0, 1, 0);
	    glBegin (GL_LINES);

	    for (i = 0 ; i < 2 * self->size[3] ; i += 1) {
		glVertex3fv (&self->vertices[3 * self->shearSprings[i]]);
	    }

	    glEnd();

	    glColor3f (0, 0, 1);
	    glBegin (GL_LINES);

	    for (i = 0 ; i < 2 * self->size[4] ; i += 1) {
		glVertex3fv (&self->vertices[3 * self->bendSprings[i]]);
	    }

	    glEnd();

	    glColor3f (1, 1, 1);
	    glPointSize (2);
	    glBegin (GL_POINTS);

	    for (i = 0 ; i < self->size[0] ; i += 1) {
		glVertex3fv (&self->vertices[3 * i]);
	    }

	    glEnd();
	}
#endif
	    
	glDisable (GL_DEPTH_TEST);
    	glDisable (GL_CULL_FACE);
	glPopMatrix();
    }

    [super traversePass: pass];
}
 
-(void) finish
{
    self->firststep = 1;

    [super finish];
}
   
-(void) get
{
    const char *k;

    k = lua_tostring (_L, 2);

    if (!xstrcmp(k, "density")) {
	lua_pushnumber (_L, self->density);
    } else if (!xstrcmp(k, "mobility")) {
	lua_pushnumber (_L, self->mobility);
    } else if (!xstrcmp(k, "drag")) {
	lua_pushnumber (_L, self->drag);
    } else if (!xstrcmp(k, "stretch")) {
	lua_newtable (_L);

	lua_pushnumber (_L, self->stiffness[0]);
	lua_rawseti (_L, -2, 1);
	lua_pushnumber (_L, self->damping[0]);
	lua_rawseti (_L, -2, 2);
    } else if (!xstrcmp(k, "shear")) {
	lua_newtable (_L);

	lua_pushnumber (_L, self->stiffness[1]);
	lua_rawseti (_L, -2, 1);
	lua_pushnumber (_L, self->damping[1]);
	lua_rawseti (_L, -2, 2);
    } else if (!xstrcmp(k, "bend")) {
	lua_newtable (_L);

	lua_pushnumber (_L, self->stiffness[2]);
	lua_rawseti (_L, -2, 1);
	lua_pushnumber (_L, self->damping[2]);
	lua_rawseti (_L, -2, 2);
    } else if (!xstrcmp(k, "granularity")) {
	lua_pushnumber (_L, self->granularity);
    } else {
	[super get];
    }
}

-(void) set
{
    const char *k;

    k = lua_tostring (_L, 2);

    if (!xstrcmp(k, "density")) {
	self->density = lua_tonumber (_L, 3);
    } else if (!xstrcmp(k, "mobility")) {
	self->mobility = lua_tonumber (_L, 3);
    } else if (!xstrcmp(k, "drag")) {
	self->drag = lua_tonumber (_L, 3);
    } else if (!xstrcmp(k, "stretch")) {
	if(lua_istable (_L, 3)) {
	    lua_rawgeti (_L, 3, 1);
	    self->stiffness[0] = lua_tonumber (_L, -1);
	    lua_rawgeti (_L, 3, 2);
	    self->damping[0] = lua_tonumber (_L, -1);
	    lua_pop (_L, 2);
	}
    } else if (!xstrcmp(k, "shear")) {
	if(lua_istable (_L, 3)) {
	    lua_rawgeti (_L, 3, 1);
	    self->stiffness[1] = lua_tonumber (_L, -1);
	    lua_rawgeti (_L, 3, 2);
	    self->damping[1] = lua_tonumber (_L, -1);
	    lua_pop (_L, 2);
	}
    } else if (!xstrcmp(k, "bend")) {
	if(lua_istable (_L, 3)) {
	    lua_rawgeti (_L, 3, 1);
	    self->stiffness[2] = lua_tonumber (_L, -1);
	    lua_rawgeti (_L, 3, 2);
	    self->damping[2] = lua_tonumber (_L, -1);
	    lua_pop (_L, 2);
	}
    } else if (!xstrcmp(k, "granularity")) {
	self->granularity = lua_tonumber (_L, -1);
    } else if (!xstrcmp(k, "position") ||
	       !xstrcmp(k, "orientation")) {
	/* Do nothing for now. */
    } else {
	[super set];
    }
}

@end
