/* Copyright (C) 2009 Papavasileiou Dimitris                             
 *                                                                      
 * This program is free software: you can redistribute it and/or modify 
 * it under the terms of the GNU General Public License as published by 
 * the Free Software Foundation, either version 3 of the License, or    
 * (at your option) any later version.                                  
 *                                                                      
 * This program is distributed in the hope that it will be useful,      
 * but WITHOUT ANY WARRANTY; without even the implied warranty of       
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the        
 * GNU General Public License for more details.                         
 *                                                                      
 * You should have received a copy of the GNU General Public License    
 * along with this program.  If not, see <http://www.gnu.org/licenses/>.
 */

#include <AL/al.h>
#include <lua.h>
#include <lualib.h>
#include <lauxlib.h>
#include <time.h>

#include "source.h"

static void callhooks (lua_State *L, void *key, int reference)
{
    if (reference != LUA_REFNIL) {
	lua_rawgeti(L, LUA_REGISTRYINDEX, reference);
	
	if (lua_isfunction (L, -1)) {
	    lua_getfield (L, LUA_REGISTRYINDEX, "userdata");
	    lua_pushlightuserdata (L, key);
	    lua_gettable (L, -2);
	    lua_replace (L, -2);
	    
	    luaX_call(L, 1, 0);
	} else if (lua_istable (L, -1)) {
	    int i, n;
	    
	    n = lua_objlen (L, -1);

	    lua_getfield (L, LUA_REGISTRYINDEX, "userdata");

	    for (i = 0 ; i < n ; i += 1) {
		lua_rawgeti (L, -2, i + 1);

		lua_pushlightuserdata (L, key);
		lua_gettable (L, -3);
		luaX_call (L, 1, 0);
	    }
	    
	    lua_pop (L, 2);
	} else {
	    lua_pop (L, 1);
	}
    }
}

@implementation Source

-(Source *) init
{
    char *list[] = {
	"bounds", "finished", "gain", "pitch", "reference", "rolloff",
	"state"
    };

    [super init];
    [self add: sizeof (list) / sizeof (char *) Properties: list];

    self->finished = LUA_REFNIL;

    alGenSources (1, &self->index);
    alSourcef (self->index, AL_BUFFER, 0);

    return self;
}

-(void) free
{
     alDeleteSources (1, &self->index);

     luaL_unref (_L, LUA_REGISTRYINDEX, self->finished);

     [super free];
}

-(ALint) index
{
    return self->index;
}

-(void) toggle
{
    ALint i;

    if (!linked) {
	alSourcePlay (self->index);
    } else {
	alSourcePause (self->index);

	do {
	    alGetSourcei (self->index, AL_SOURCE_STATE, &i);
	} while (i == AL_PLAYING);
    }

    [super toggle];
}

-(void) transform
{
    double *r;

    [super transform];
    
    r = [self translation];

    self->velocity[0] = (r[0] - self->velocity[0]) * self->delta;
    self->velocity[1] = (r[1] - self->velocity[1]) * self->delta;
    self->velocity[2] = (r[2] - self->velocity[2]) * self->delta;
	
    alSource3f (self->index, AL_POSITION, r[0], r[1], r[2]);
    alSourcefv (self->index, AL_VELOCITY, self->velocity);
}

-(void) begin
{
    self->delta = 0;

    self->velocity[0] = [self translation][0];
    self->velocity[1] = [self translation][1];
    self->velocity[2] = [self translation][2];
}

-(void) stepBy: (double) h at: (double)t
{
    self->delta += h;
    
    [super stepBy: h at: t];
}

-(void) finish
{
    ALenum newstate;
    
    alGetSourcei (self->index, AL_SOURCE_STATE, &newstate);

    if (newstate != self->state && newstate == AL_STOPPED) {
	callhooks (_L, self, self->finished);
    }

    self->state = newstate;
}

-(void) get
{    
    const char *k;
    
    k = lua_tostring (_L, -1);

    if (!xstrcmp(k, "gain")) {
	ALfloat gain;

	alGetSourcef (self->index, AL_GAIN, &gain);

	lua_pushnumber (_L, gain);
    } else if (!xstrcmp(k, "bounds")) {
	ALfloat gain;
	
        lua_newtable (_L);
	
	alGetSourcef (self->index, AL_MIN_GAIN, &gain);
	lua_pushnumber (_L, gain);
	lua_rawseti (_L, -2, 1);
        
	alGetSourcef (self->index, AL_MAX_GAIN, &gain);
	lua_pushnumber (_L, gain);
	lua_rawseti (_L, -2, 2);
    } else if (!xstrcmp(k, "reference")) {
	ALfloat distance;

	alGetSourcef (self->index, AL_REFERENCE_DISTANCE, &distance);
	lua_pushnumber (_L, distance);
    } else if (!xstrcmp(k, "rolloff")) {
	ALfloat factor;

	alGetSourcef (self->index, AL_ROLLOFF_FACTOR, &factor);
	lua_pushnumber (_L, factor);
    } else if (!xstrcmp(k, "pitch")) {
	ALfloat factor;

	alGetSourcef (self->index, AL_PITCH, &factor);
	lua_pushnumber (_L, factor);
    } else if (!xstrcmp(k, "state")) {
	if (self->state == AL_PLAYING) {
	    lua_pushliteral (_L, "playing");
	} else if (self->state == AL_PAUSED) {
	    lua_pushliteral (_L, "paused");
	} else if (self->state == AL_STOPPED || self->state == AL_INITIAL) {
	    lua_pushliteral (_L, "stopped");
	} else {
	    lua_pushliteral (_L, "huh?");
	}
    } else if (!xstrcmp(k, "finished")) {
        lua_rawgeti (_L, LUA_REGISTRYINDEX, self->finished);
    } else {
	[super get];
    }
}

-(void) set
{
    const char *k;

    k = lua_tostring (_L, -2);

    if (!xstrcmp(k, "gain")) {
	alSourcef (self->index, AL_GAIN, lua_tonumber (_L, 3));
    } else if (!xstrcmp(k, "bounds")) {
	if (lua_istable (_L, 3)) {
	    lua_rawgeti (_L, 3, 1);
	    alSourcef (self->index, AL_MIN_GAIN, lua_tonumber (_L, -1));

	    lua_rawgeti (_L, 3, 2);
	    alSourcef (self->index, AL_MAX_GAIN, lua_tonumber (_L, -1));

	    lua_pop (_L, 2);
	}
    } else if (!xstrcmp(k, "reference")) {
	alSourcef (self->index, AL_REFERENCE_DISTANCE, lua_tonumber (_L, 3));
    } else if (!xstrcmp(k, "rolloff")) {
	alSourcef (self->index, AL_ROLLOFF_FACTOR, lua_tonumber (_L, 3));
    } else if (!xstrcmp(k, "pitch")) {
	alSourcef (self->index, AL_PITCH, lua_tonumber (_L, 3));
    } else if (!xstrcmp(k, "finished")) {
        luaL_unref (_L, LUA_REGISTRYINDEX, self->finished);
        self->finished = luaL_ref (_L, LUA_REGISTRYINDEX);
    } else {
	[super set];
    }
}

@end
