/* Copyright (C) 2009 Papavasileiou Dimitris                             
 *                                                                      
 * This program is free software: you can redistribute it and/or modify 
 * it under the terms of the GNU General Public License as published by 
 * the Free Software Foundation, either version 3 of the License, or    
 * (at your option) any later version.                                  
 *                                                                      
 * This program is distributed in the hope that it will be useful,      
 * but WITHOUT ANY WARRANTY; without even the implied warranty of       
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the        
 * GNU General Public License for more details.                         
 *                                                                      
 * You should have received a copy of the GNU General Public License    
 * along with this program.  If not, see <http://www.gnu.org/licenses/>.
 */

#include <stdlib.h>
#include <math.h>
#include <lua.h>
#include <lauxlib.h>
#include <GL/gl.h>
#include <GL/glu.h>
#include "static.h"

#define USE_VBO

static int noarrays, nobufferobjects;

@implementation Static

-(Static *) initWithVertices: (GLfloat *)floats
		  andIndices: (GLushort *)shorts
		      ofSize: (int[2]) n
{
    unsigned int b[2];

    /* Get the configuration. */
    
    lua_getglobal (_L, "options");

    lua_getfield (_L, -1, "noarrays");
    noarrays = lua_toboolean (_L, -1);
    lua_pop (_L, 1);

    lua_getfield (_L, -1, "nobufferobjects");
    nobufferobjects = lua_toboolean (_L, -1);
    lua_pop (_L, 2);

    /* Initialize the object. */
    
    self = [super init];

    /* Create the buffer objects. */

    if (!nobufferobjects) {
	glGenBuffersARB(2, b);
    
	glBindBufferARB(GL_ARRAY_BUFFER, b[0]);
	glBufferDataARB(GL_ARRAY_BUFFER, n[0] * 8 * sizeof(GLfloat),
			floats, GL_STATIC_DRAW);
	

	glBindBufferARB(GL_ELEMENT_ARRAY_BUFFER, b[1]);
	glBufferDataARB(GL_ELEMENT_ARRAY_BUFFER, n[1] * sizeof(GLushort),
			shorts, GL_STATIC_DRAW);

	self->buffers[0] = b[0];
	self->buffers[1] = b[1];
    } else {
	self->vertices = (GLfloat *)malloc(n[0] * 8 * sizeof(GLfloat));
	self->indices = (GLushort *)malloc(n[1] * sizeof(GLushort));

	memcpy (self->vertices, floats, n[0] * 8 * sizeof(GLfloat));
	memcpy (self->indices, shorts, n[1] * sizeof(GLushort));
    }
    
    self->size[0] = n[0];
    self->size[1] = n[1];

    return self;
}

-(void) freeBuffers
{
    if (!nobufferobjects) {
	glDeleteBuffersARB(2, self->buffers);
    } else {
	free (self->vertices);
	free (self->indices);
    }
}

-(int) vertices
{
    return self->size[0];
}

-(int) indices
{
    return self->size[1];
}

-(void) traversePass: (int)pass
{
    if (pass < 2) {
	glMatrixMode (GL_MODELVIEW);
	glPushMatrix();
	glMultMatrixd ([self matrix]);
	
/*     glPolygonMode (GL_FRONT_AND_BACK, GL_LINE); */
	glEnable (GL_DEPTH_TEST);
	glEnable (GL_CULL_FACE);
    
	glClientActiveTexture (GL_TEXTURE0);

	if (!nobufferobjects) {
	    glEnableClientState(GL_VERTEX_ARRAY);
	    glEnableClientState(GL_NORMAL_ARRAY);
	    glEnableClientState(GL_TEXTURE_COORD_ARRAY);

	    glBindBufferARB(GL_ARRAY_BUFFER, self->buffers[0]);

	    glVertexPointer(3, GL_FLOAT, 32, (void *)0);
	    glNormalPointer(GL_FLOAT, 32, (void *)(3 * sizeof(GLfloat)));
	    glTexCoordPointer(2, GL_FLOAT, 32, (void *)(6 * sizeof(GLfloat)));
	    
	    glBindBufferARB(GL_ELEMENT_ARRAY_BUFFER, self->buffers[1]);
	    glDrawElements(GL_TRIANGLES, self->size[1],
			   GL_UNSIGNED_SHORT, (void *)0);

	    glBindBufferARB(GL_ARRAY_BUFFER, 0);
	    glBindBufferARB(GL_ELEMENT_ARRAY_BUFFER, 0);

	    glDisableClientState(GL_VERTEX_ARRAY);
	    glDisableClientState(GL_NORMAL_ARRAY);
	    glDisableClientState(GL_TEXTURE_COORD_ARRAY);
	} else if (!noarrays) {
	    glEnableClientState(GL_VERTEX_ARRAY);
	    glEnableClientState(GL_NORMAL_ARRAY);
	    glEnableClientState(GL_TEXTURE_COORD_ARRAY);

	    glVertexPointer(3, GL_FLOAT, 32, self->vertices);
	    glNormalPointer(GL_FLOAT, 32, self->vertices + 3);
	    glTexCoordPointer(2, GL_FLOAT, 32, self->vertices + 6);

	    glDrawElements(GL_TRIANGLES, self->size[1],
			   GL_UNSIGNED_SHORT, self->indices);

	    glDisableClientState(GL_VERTEX_ARRAY);
	    glDisableClientState(GL_NORMAL_ARRAY);
	    glDisableClientState(GL_TEXTURE_COORD_ARRAY);
	} else {
	    int i, j;
	    
	    glBegin (GL_TRIANGLES);
	
	    for (i = 0 ; i < self->size[1] ; i += 1) {
		j = self->indices[i];
		
		glNormal3fv (&self->vertices[8 * j + 3]);
		glTexCoord2fv (&self->vertices[8 * j + 6]);
		glVertex3fv (&self->vertices[8 * j]);
	    }

	    glEnd();
	}

	glPopMatrix();

	glDisable (GL_DEPTH_TEST);
	glDisable (GL_CULL_FACE);
    }

    [super traversePass: pass];
}

@end
