/* $Id: bufferio.c 658 2006-05-13 14:50:30Z jim $
   teebu - An archiving tool
   Copyright (C) 2006 Jim Farrand

   This program is free software; you can redistribute it and/or modify it
   under the terms of the GNU General Public License as published by the Free
   Software Foundation; either version 2 of the License, or (at your option)
   any later version.

   This program is distributed in the hope that it will be useful, but WITHOUT
   ANY WARRANTY; without even the implied warranty of MERCHANTABILITY or
   FITNESS FOR A PARTICULAR PURPOSE.  See the GNU General Public License for
   more details.

   You should have received a copy of the GNU General Public License along with
   this program; if not, write to the Free Software Foundation, Inc., 51
   Franklin Street, Fifth Floor, Boston, MA  02110-1301  USA
 */


#include <assert.h>

#include "logging.h"
#include "bufferio.h"

/**********
 * Output *
 **********/

struct bufferio_out_data
{
  out_stream_t out_base;
  bool out_strict_blocking;
  bool out_own_base;
  iobuffer_t out_buffer;        // Should never be left full
};

static output_err_t
do_flush (struct bufferio_out_data *data)
{
  assert (data);
  return output_all (data->out_base, &data->out_buffer);
}

static output_err_t
my_flush (void *uncast_data)
{
  struct bufferio_out_data *data = uncast_data;
  if (!data)
    return OUTPUT_ERR_BAD;

  // The buffer is never left full, so if strict_blocking is set, ignore the flush
  // Also ignore if there is no data
  if (data->out_strict_blocking
      || 0 == iobuffer_data_size (&data->out_buffer))
    return OUTPUT_OK;

  return do_flush (data);
}

static output_err_t
my_output_limited (void *uncast_data, iobuffer_t * output, size_t amount)
{
  struct bufferio_out_data *data = uncast_data;
  if (!data)
    return OUTPUT_ERR_BAD;

  iobuffer_copy_limited (output, &data->out_buffer, amount);

  // Never leave the buffer full
  if (0 == iobuffer_free_size (&data->out_buffer))
    return do_flush (data);

  return OUTPUT_OK;
}

static output_err_t
my_output_mark (void *uncast_data)
{
  struct bufferio_out_data *data = uncast_data;
  if (!data)
    return OUTPUT_ERR_BAD;

  if (iobuffer_data_size (&data->out_buffer) > 0)
    {
      output_err_t err = do_flush (data);
      if (OUTPUT_OK != err)
        return err;
    }

  return output_mark (data->out_base);
}

static output_err_t
my_close_out (void *uncast_data)
{
  struct bufferio_out_data *data = uncast_data;
  if (!data)
    return OUTPUT_ERR_BAD;

  if (iobuffer_data_size (&data->out_buffer) > 0)
    {
      output_err_t err = do_flush (data);
      if (OUTPUT_OK != err)
        return err;
    }

  return close_out (data->out_base);
}

static void
my_release_out (void *uncast_data)
{
  struct bufferio_out_data *data = uncast_data;
  assert (uncast_data);

  if (data->out_own_base)
    release_out (data->out_base);

  free (iobuffer_data_block (&data->out_buffer));

  free (data);
}

static out_stream_type_t my_out_type = {
  .output_limited = my_output_limited,
  .output_mark = my_output_mark,
  .flush = my_flush,
  .close_out = my_close_out,
  .release_out = my_release_out
};

out_stream_t
bufferio_open_out (out_stream_t outs, size_t bufsize, bool strict_blocking,
                   bool own_base)
{
  struct bufferio_out_data *data =
    malloc (sizeof (struct bufferio_out_data) + bufsize);
  if (!data)
    MEMFAILED ();

  char *buf = malloc (bufsize);
  if (!buf)
    MEMFAILED ();

  init_iobuffer_with (&data->out_buffer, bufsize, 0, buf);

  data->out_base = outs;
  data->out_own_base = own_base;
  data->out_strict_blocking = strict_blocking;

  return open_out (&my_out_type, data);
}

/*********
 * Input *
 *********/

struct bufferio_in_data
{
  in_stream_t in_base;
  bool in_own_base;
  iobuffer_t in_buffer;
};

/* Call when there is no data in the buffer, to cause it to be refilled. */
static input_err_t
refill_buffer (struct bufferio_in_data *data)
{
  assert (0 == iobuffer_data_size (&data->in_buffer));

  input_err_t err = input_all (data->in_base, &data->in_buffer);
  if (0 == iobuffer_data_size (&data->in_buffer))
    return err;

  return INPUT_OK;
}

static input_err_t
my_input_limited (void *uncast_data, iobuffer_t * buf, size_t max)
{
  struct bufferio_in_data *data = uncast_data;
  if (!data)
    return INPUT_ERR_BAD;

  if (0 == iobuffer_data_size (&data->in_buffer))
    {
      input_err_t err = refill_buffer (data);
      if (INPUT_OK != err)
        return err;
    }

  assert (iobuffer_data_size (&data->in_buffer) > 0);
  iobuffer_copy_limited (&data->in_buffer, buf, max);

  return INPUT_OK;
}

static input_err_t
my_input_mark (void *uncast_data)
{
  struct bufferio_in_data *data = uncast_data;
  if (!data)
    return INPUT_ERR_BAD;

  if (iobuffer_data_size (&data->in_buffer) > 0)
    return INPUT_ERR_NO_MARK;

  return input_mark (data->in_base);
}

static input_err_t
my_close_in (void *uncast_data)
{
  struct bufferio_in_data *data = uncast_data;
  if (!data)
    return INPUT_ERR_BAD;

  return close_in (data->in_base);
}

static void
my_release_in (void *uncast_data)
{
  struct bufferio_in_data *data = uncast_data;
  assert (data);

  if (data->in_own_base)
    release_in (data->in_base);

  free (iobuffer_data_block (&data->in_buffer));
  free (data);
}

static in_stream_type_t my_in_type = {
  .input_limited = my_input_limited,
  .input_mark = my_input_mark,
  .close_in = my_close_in,
  .release_in = my_release_in
};

in_stream_t
bufferio_open_in (in_stream_t ins, size_t bufsize, bool own_base)
{
  struct bufferio_in_data *data = malloc (sizeof (struct bufferio_in_data));
  if (!data)
    MEMFAILED ();

  data->in_base = ins;
  data->in_own_base = own_base;

  char *buf = malloc (bufsize);
  if (!buf)
    MEMFAILED ();

  init_iobuffer_with (&data->in_buffer, bufsize, 0, buf);

  return open_in (&my_in_type, data);
}
