/* $Id: commandprocess.c 658 2006-05-13 14:50:30Z jim $
   teebu - An archiving tool
   Copyright (C) 2006 Jim Farrand

   This program is free software; you can redistribute it and/or modify it
   under the terms of the GNU General Public License as published by the Free
   Software Foundation; either version 2 of the License, or (at your option)
   any later version.

   This program is distributed in the hope that it will be useful, but WITHOUT
   ANY WARRANTY; without even the implied warranty of MERCHANTABILITY or
   FITNESS FOR A PARTICULAR PURPOSE.  See the GNU General Public License for
   more details.

   You should have received a copy of the GNU General Public License along with
   this program; if not, write to the Free Software Foundation, Inc., 51
   Franklin Street, Fifth Floor, Boston, MA  02110-1301  USA
 */


#include <unistd.h>
#include <stdio.h>
#include <malloc.h>
#include <stdlib.h>
#include <errno.h>
#include <string.h>
#include <assert.h>
#include <sys/select.h>
#include <sys/wait.h>

#include "logging.h"
#include "commandprocess.h"
#include "unixmisc.h"

struct comproc
{
  int stdin;                    /* stdin fd to write to */
  int stdout;                   /* stdout fd to read from */
  bool stdin_done;              /* true iff stdin has been closed */
  bool stdout_done;             /* true iff stdout has been closed */
  int pid;                      /* pid of child */
  void (*consumer) (void *, const char *, size_t);      /* function that consumes input */
  void *consumer_data;          /* Custom data for consumer. */
  char *buffer;                 /* buffer for process io */
  size_t buffer_size;
  const char *filename;
  args_t argv;
  args_t envp;
    bool (*arg_cons) (void *, args_t argv, args_t envp);
  void *arg_cons_data;
};

static char **
copy_args (char *const *passed)
{
  if (NULL == passed)
    return NULL;

  int count;
  for (count = 0; passed[count] != NULL; count++)
    /* spin */ ;

  char **copy = calloc (sizeof (char *), count + 1);

  for (int i = 0; i < count; i++)
    {
      copy[i] = strdup (passed[i]);
    }

  return copy;
}

static void
free_args (char *const *passed)
{
  if (!passed)
    return;

  for (int i = 0; passed[i] != NULL; i++)
    free (passed[i]);

  free ((char **) passed);
}

bool
start_comproc (comproc_t cp)
{
  assert (cp);
  assert (-1 == cp->pid);

  cp->stdin_done = false;
  cp->stdout_done = false;
  cp->pid = -1;

  // Create pipes
  int filedes[2];
  if (-1 == pipe (filedes))
    {
      LOG (ERROR, "Error: Failed to create pipe (in create_comproc)");
      return false;             // error creating pipe!
    }

  const int stdin_read = filedes[0];
  cp->stdin = filedes[1];

  if (-1 == pipe (filedes))
    {
      LOG (ERROR, "Error: Failed to create pipe (in create_comproc)");
      close (cp->stdin);
      close (stdin_read);
      return false;             // error creating pipe!
    }

  cp->stdout = filedes[0];
  const int stdout_write = filedes[1];

  if (!set_close_on_exec (cp->stdin))
    return false;

  if (!set_close_on_exec (cp->stdout))
    return false;

  int pid = fork ();
  if (-1 == pid)
    {
      // Forking hell!
      close (cp->stdin);
      close (cp->stdout);
      close (stdin_read);
      close (stdout_write);
      LOG (ERROR, "Error: Failed to fork child process (in create_comproc)");
      return false;             // error forking child process
    }
  else if (0 == pid)
    {
      // We are in the child:
      //      Close undeeded ends of pipes
      //      This is our copy of the filedesc, it doesn't affect it's
      //      counterpart in the main process
      // These now close on exec
      // close(cp->stdin) ;
      // close(cp->stdout) ;
      // Redirect stdin and stdout
      DEBUGF ("Child process reading from %d, writing to %d", stdin_read,
              stdout_write);
      if (-1 == dup2 (stdin_read, 0))
        {
          close (stdin_read);
          close (stdout_write);
          LOG (ERROR, "Error: Failed to dup stdin (in create_comproc)");
          exit (100);
        }

      if (-1 == dup2 (stdout_write, 1))
        {
          close (stdin_read);
          close (stdout_write);
          LOG (ERROR, "Error: Failed to dup stdout (in create_comproc)");
          exit (100);
        }

      close (stdin_read);
      close (stdout_write);

      if (cp->arg_cons)
        {
          if (!(cp->arg_cons) (cp->arg_cons_data, cp->argv, cp->envp))
            return false;
        }

      //      Exec subprocess
      if (-1 == execvp (cp->filename, cp->argv))
        {
          const int err = errno;
          close (stdin_read);
          close (stdout_write);
          LOGF (ERROR,
                "Error: Failed to exec subprocess: %s (in create_comproc): %s\n",
                strerror (err), cp->argv[0]);
          exit (100);
        }

      // We won't really get here, because execve doesn't return on success
      // But this keeps the compiler happy
      return false;
    }
  else
    {
      DEBUGF ("Parent process reading from %d, writing to %d", cp->stdout,
              cp->stdin);
      // We are in the parent
      //      Close uneeded ends of pipes
      close (stdin_read);
      close (stdout_write);

      cp->pid = pid;

      return true;
    }
}

comproc_t
create_comproc (void (*consumer) (void *, const char *, size_t),
                void *consumer_data,
                size_t bufsize,
                const char *filename,
                char *const argv[],
                char *const envp[],
                bool (*arg_cons) (void *, args_t, args_t),
                void *arg_cons_data)
{
  // Alloc struct
  comproc_t r = (comproc_t) malloc (sizeof (struct comproc));
  if (!r)
    return NULL;                // out of mem!

  // Create buffer
  r->buffer = malloc (bufsize);
  if (!r->buffer)
    {
      free (r);
      return NULL;              // error allocating buffer
    }

  r->buffer_size = bufsize;
  r->consumer = consumer;
  r->consumer_data = consumer_data;
  r->filename = filename;
  r->argv = copy_args (argv);
  r->envp = copy_args (envp);
  r->pid = -1;
  r->arg_cons = arg_cons;
  r->arg_cons_data = arg_cons_data;

  /* if(! restart_comproc(r)) {
     free(r->buffer) ;
     free(r) ;
     return NULL ;
     } */

  return r;
}

void
release_comproc (comproc_t cp)
{
  if (-1 != cp->pid)
    finish_comproc_input (cp);

  free_args (cp->argv);
  free_args (cp->envp);
  free (cp->buffer);
  free (cp);
}

#define READY_STDIN  0x01
#define READY_STDOUT 0x02

/* Waits for process to be ready to input or output.  Returns a bitset of above
 * values, 0 if an error occurred. */
static int
wait_for_ready (comproc_t cp)
{
  assert (cp);
  assert (-1 != cp->pid);

  fd_set wfds, rfds;
  int n = 0;

  FD_ZERO (&wfds);
  FD_ZERO (&rfds);

  if (!cp->stdin_done)
    {
      FD_SET (cp->stdin, &wfds);
      if (cp->stdin >= n)
        n = cp->stdin + 1;
    }

  if (!cp->stdout_done)
    {
      FD_SET (cp->stdout, &rfds);
      if (cp->stdout >= n)
        n = cp->stdout + 1;
    }

  // Nothing to wait on
  if (0 == n)
    return 0;

  int retval = select (n, &rfds, &wfds, NULL, NULL);
  if (-1 == retval)
    {
      LOGF (WARNING,
            "Warning: Select returned error: %s (in wait_for_ready)\n",
            strerror (errno));
      return 0;
    }

  int r = 0;
  if (FD_ISSET (cp->stdin, &wfds))
    r |= READY_STDIN;

  if (FD_ISSET (cp->stdout, &rfds))
    r |= READY_STDOUT;

  //  Should always return one or other
  assert (r != 0);

  return r;
}

/* Send output to process.  Returns amount of output done, which may be less
 * than the amount requested, or 0 if an error occurred.  (Outputting 0 bytes
 * will also return 0.) */
static size_t
send_input_to_comproc (comproc_t cp, const char *buf, size_t offset,
                       size_t len)
{
  assert (len > 0);

  // Output up to this amount, to get an atomic write
  if (PIPE_BUF < len)
    len = PIPE_BUF;

  ssize_t amount = write (cp->stdin, buf + offset, len);
  if (-1 == amount)
    {
      LOGF (ERROR, "Error: Failed to output to subprocess: %s\n",
            strerror (errno));
      return 0;
    }

  return amount;
}

static void
receive_output_from_comproc (comproc_t cp)
{
  ssize_t amount = read (cp->stdout, cp->buffer, cp->buffer_size);
  if (0 == amount)
    {
      cp->stdout_done = true;
      close (cp->stdout);
    }
  else if (-1 == amount)
    {
      LOGF (ERROR, "Warning: Failed to input from subprocess: %s\n",
            strerror (errno));
    }
  else
    {
      (*cp->consumer) (cp->consumer_data, cp->buffer, amount);
    }
}

/* Send data to the process.  Return true if successfully sent. */
bool
send_to_comproc (comproc_t cp, const char *buf, size_t len)
{
  assert (cp);
  assert (buf);

  if (-1 == cp->pid)
    start_comproc (cp);

  if (cp->stdin_done)
    return false;

  int done = 0;
  while (done < len)
    {
      // If ready for input, send a block of input
      // Do this first to keep pipe full
      int readyness = wait_for_ready (cp);

      if (!readyness)
        break;

      if (readyness & READY_STDIN)
        {
          size_t amount = send_input_to_comproc (cp, buf, done, len);
          // Escape if we failed to send any data
          if (amount <= 0)
            break;

          done += amount;
        }

      // If ready for output, do a block of output
      if (readyness & READY_STDOUT)
        receive_output_from_comproc (cp);
    }

  return done == len;
}

void
finish_comproc_input (comproc_t cp)
{
  if (-1 == cp->pid)
    return;                     // Already closed

  // Close processes input
  close (cp->stdin);
  cp->stdin_done = true;

  while (!cp->stdout_done)
    {
      int readyness = wait_for_ready (cp);
      if (readyness & READY_STDOUT)
        receive_output_from_comproc (cp);
      else
        assert (false);         // Should never happen
    }

  int status;
  if (cp->pid != waitpid (cp->pid, &status, 0))
    {
      LOG (ERROR, "Error: waitpid failed (in finish_comproc_input)");
      return;
    }
  cp->pid = -1;

  if (0 != status)
    {
      LOGF (ERROR, "Warning: process returned: %d", status);
    }
}
