/* $Id: datapairstore.h 717 2006-05-31 20:52:44Z jim $
   teebu - An archiving tool
   Copyright (C) 2006 Jim Farrand

   This program is free software; you can redistribute it and/or modify it
   under the terms of the GNU General Public License as published by the Free
   Software Foundation; either version 2 of the License, or (at your option)
   any later version.

   This program is distributed in the hope that it will be useful, but WITHOUT
   ANY WARRANTY; without even the implied warranty of MERCHANTABILITY or
   FITNESS FOR A PARTICULAR PURPOSE.  See the GNU General Public License for
   more details.

   You should have received a copy of the GNU General Public License along with
   this program; if not, write to the Free Software Foundation, Inc., 51
   Franklin Street, Fifth Floor, Boston, MA  02110-1301  USA
 */


/**
 * Abstraction over files, which allows storage of arbitrary sized data chunks
 * tagged with a string label.  The representation is efficient for both short
 * and long data chunks.  A short data chunk (< 128 bytes) with a short label
 * requires only two bytes additional storage over the combined size of the
 * label and data.  A long data chunk (< * 2^64) bytes with a short label
 * requires 11bytes of additional storage.
 *
 * The actual file format can cope with labels and data of any size, but the
 * current implementation is limited to data chunk sizes which fit in off_t,
 * and labels which fit in memory.
 */

#ifndef DATAPAIRSTORE_H
#define DATAPAIRSTORE_H

#include <stdlib.h>
#include "nonstdio.h"
#include "statio.h"

typedef struct dps_out *dps_out_t;

/* Open a datapair store for output. */
dps_out_t open_dps_out (out_stream_t out_base, off_t chunk_size);

/* Close datapair output store. */
bool close_dps_out (dps_out_t);

/* Release a datapair output store handle.  The handle becomes invalid and must
 * not be used. */
void release_dps_out (dps_out_t);

/* Output the given data with the given name. */
bool output_small_pair (dps_out_t, const char *name, const char *data);

bool output_small_pair_size (dps_out_t, const char *name, size_t,
                             const char *data, size_t);

/* Create a datapair of the given size and name, and return a handle which the
 * data for that pair will be written to.  You must not  call output_big_pair,
 * output_small_pair or close_dps_out on this datapair store until
 * output_end_big_pair has been called. */
out_stream_t output_big_pair (dps_out_t, const char *name, size_t data_len);

out_stream_t output_big_pair_size (dps_out_t, const char *name,
                                   size_t name_len, size_t data_len);

/* Call to indicate end of a big_pair data.  The stream returned by
 * output_big_pair is closed (if it has not been already).  */
bool output_end_big_pair (dps_out_t);

/* Output a mark on the underlying stream. */
bool dps_output_mark (dps_out_t);

/* Enable output filters. */
bool enable_output_filters (dps_out_t dps, bool gzip, bool bzip2,
                            const char *gpg_passphase_path);

typedef struct dps_in *dps_in_t;

dps_in_t open_dps_in (in_stream_t in_base, size_t max_name_len, iostat_t *stats);
bool close_dps_in (dps_in_t);
void release_dps_in (dps_in_t);

/* Input a new pair. */
bool dps_input_pair (dps_in_t, input_err_t *);

/* Return name of current pair. */
char *dps_input_pair_name (dps_in_t);

/* Return size of current pair. */
size_t dps_input_data_size (dps_in_t);

/* Skip data associated with current pair. */
bool dps_skip_data_in (dps_in_t);

/* skip rest of this chunk */
bool dps_skip_chunk (dps_in_t);

/* Read the data into memory and return a pointer to it.  The pointer is valid
 * until the next call to dps_input_small_data, or until the dps_in is
 * released. */
char *dps_input_small_data (dps_in_t dps);

/* Return a file handle to read data from. */
in_stream_t dps_input_big_data (dps_in_t);

/* Finish with handle returned by input_big_data.  Handle is closed, if it has
 * not been closed already. */
bool dps_finish_big_input (dps_in_t);

/* Input a mark. */
bool dps_input_mark (dps_in_t dps);

/* Try to recover from an error.  */
bool dps_input_recovery (dps_in_t dps);

bool enable_input_filters (dps_in_t dps, bool gzip, bool bzip2,
                           bool gpg, const char *gpg_passphase_path);

#endif
