/* $Id: deque.c 658 2006-05-13 14:50:30Z jim $
   teebu - An archiving tool
   Copyright (C) 2006 Jim Farrand

   This program is free software; you can redistribute it and/or modify it
   under the terms of the GNU General Public License as published by the Free
   Software Foundation; either version 2 of the License, or (at your option)
   any later version.

   This program is distributed in the hope that it will be useful, but WITHOUT
   ANY WARRANTY; without even the implied warranty of MERCHANTABILITY or
   FITNESS FOR A PARTICULAR PURPOSE.  See the GNU General Public License for
   more details.

   You should have received a copy of the GNU General Public License along with
   this program; if not, write to the Free Software Foundation, Inc., 51
   Franklin Street, Fifth Floor, Boston, MA  02110-1301  USA
 */


#include <assert.h>
#include <string.h>

#include "deque.h"

struct node
{
  struct node *prev, *next;
  void *data;
};

/* Nodes form a circular linked list.  Data is between start and end, free
 * space is between end and start. */
struct deque
{
  struct node *start, *end;
  size_t start_free, end_free;
  size_t data_size, block_size;
};

static struct node *
create_node (size_t data_size, size_t block_size)
{
  void *data = malloc (data_size * block_size);
  if (!data)
    return NULL;

  struct node *node = malloc (sizeof (struct node));
  if (!node)
    {
      free (data);
      return NULL;
    }

  node->prev = node->next = NULL;
  node->data = data;

  return node;
}

static void
release_node (struct node *node)
{
  free (node->data);
  free (node);
}

deque_t
create_deque (size_t data_size, size_t block_size)
{
  struct node *node = create_node (data_size, block_size);
  node->prev = node->next = node;

  deque_t deque = malloc (sizeof (struct deque));
  if (!deque)
    {
      release_node (node);
      return NULL;
    }

  deque->start = deque->end = node;
  deque->start_free = 0;
  deque->end_free = block_size;
  deque->data_size = data_size;
  deque->block_size = block_size;

  return deque;
}

void
release_deque (deque_t deque)
{
  struct node *node = deque->start;
  do
    {
      node = node->next;        // Start node will be done last
      release_node (node);
    }
  while (node != deque->start);

#ifndef NDEBUG
  deque->start = deque->end = NULL;     // Break the deque to prevent accidental reuse
#endif

  free (deque);
}

bool
is_deque_empty (deque_t deque)
{
  assert (deque);

  if (deque->start != deque->end)
    return false;

  return deque->block_size == (deque->start_free + deque->end_free);
}

#define FREE_AFTER_PTR(deque) \
    (deque->end->data + (deque->data_size * (deque->block_size - deque->end_free)))

#define FREE_BEFORE_PTR(deque,n) \
    (deque->start->data + (deque->data_size * (deque->start_free - n)))

/* Add to the end of the deque.  Returns the number of items added, or 0 if an
 * error occurred.  (Adding 0 items is safe, but will return 0.)  This function
 * may add less than the requested amount if it is convenient for the
 * implementation to do so.  */
static size_t
add_some_to_end (deque_t deque, const void *src, size_t offset, size_t len)
{
  assert (deque);

  if (0 == len)
    return 0;

  const size_t data_size = deque->data_size;

  // Check for space in the current block
  if (0 == deque->end_free)
    {
      // Need to find space
      if (deque->end->next == deque->start)
        {
          // Worst case: there are no free blocks after the end block, so we need to allocate one
          struct node *new_node = create_node (data_size, deque->block_size);
          if (!new_node)
            return 0;           // out of memory!

          // Insert into chain after end
          new_node->prev = deque->end;
          new_node->next = deque->end->next;
          deque->end->next->prev = new_node;
          deque->end->next = new_node;
        }

      deque->end = deque->end->next;
      deque->end_free = deque->block_size;
    }

  if (deque->end_free < len)
    len = deque->end_free;

  memcpy (FREE_AFTER_PTR (deque), src + (offset * data_size),
          len * data_size);
  deque->end_free -= len;

  return len;
}

/* return number of items actualled added, which will be equal to len unless we
 * ran out of memory. */
size_t
add_to_deque_end (deque_t deque, const void *src, const size_t len)
{
  size_t running_len = len;
  while (running_len > 0)
    {
      size_t added =
        add_some_to_end (deque, src, len - running_len, running_len);
      if (0 == added)
        break;
      running_len -= added;
    }
  return (len - running_len);
}

bool
single_add_to_deque_end (deque_t deque, const void *src)
{
  return 0 != add_to_deque_end (deque, src, 1);
}

static size_t
add_some_to_start (deque_t deque, const void *src, size_t offset, size_t len)
{
  assert (deque);

  if (0 == len)
    return 0;

  const size_t data_size = deque->data_size;

  // Check for space in the current block
  if (0 == deque->start_free)
    {
      // No space, so we need a new node
      if (deque->start->prev == deque->end)
        {
          // Worst case: there are no free blocks after the end block, so we need to allocate one
          struct node *new_node = create_node (data_size, deque->block_size);
          if (!new_node)
            return 0;           // out of memory!

          // Insert into chain before start
          new_node->next = deque->start;
          new_node->prev = deque->start->prev;
          deque->start->prev->next = new_node;
          deque->start->prev = new_node;
        }

      deque->start = deque->start->prev;
      deque->start_free = deque->block_size;
    }

  if (deque->start_free < len)
    {
      offset = len - deque->start_free;
      len = deque->start_free;
    }

  memcpy (FREE_BEFORE_PTR (deque, len), src + (offset * data_size),
          len * data_size);
  deque->start_free -= len;


  return len;
}

size_t
add_to_deque_start (deque_t deque, const void *src, size_t len)
{
  size_t running_len = len;
  while (running_len > 0)
    {
      size_t added = add_some_to_start (deque, src, 0, running_len);
      if (0 == added)
        break;
      running_len -= added;
    }
  return (len - running_len);
}

bool
single_add_to_deque_start (deque_t deque, const void *src)
{
  return 1 == add_to_deque_start (deque, src, 1);
}

static size_t
remove_some_from_deque_start (deque_t deque, void *dst, size_t offset,
                              size_t len)
{
  size_t data_len;
  if (deque->start == deque->end)
    data_len = (deque->block_size - deque->end_free) - (deque->start_free);
  else
    data_len = deque->block_size - deque->start_free;

  if (data_len < len)
    len = data_len;


  if (0 == len)
    return 0;

  const size_t data_size = deque->data_size;
  memcpy (dst + (data_size * offset),
          deque->start->data + (data_size * deque->start_free),
          len * data_size);
  deque->start_free += len;

  if (deque->start_free == deque->block_size)
    {
      if (deque->start != deque->end)
        {
          deque->start = deque->start->next;
          deque->start_free = 0;
        }
      else
        {
          deque->start_free = 0;
          deque->end_free = deque->block_size;
        }
    }

  return len;
}

size_t
remove_from_deque_start (deque_t deque, void *dst, size_t len)
{
  size_t running_len = len;
  while (running_len > 0)
    {
      size_t amount =
        remove_some_from_deque_start (deque, dst, len - running_len,
                                      running_len);
      if (0 == amount)
        break;
      running_len -= amount;
    }

  return len - running_len;
}

bool
single_remove_from_deque_start (deque_t deque, void *dst)
{
  return 1 == remove_from_deque_start (deque, dst, 1);
}

static size_t
remove_some_from_deque_end (deque_t deque, void *dst, size_t offset,
                            size_t len)
{
  assert (len > 0);

  if (deque->end_free == deque->block_size)
    {
      if (deque->start != deque->end)
        {
          deque->end = deque->end->prev;
          deque->end_free = 0;
        }
    }

  size_t data_len;
  if (deque->start == deque->end)
    data_len = (deque->block_size - deque->end_free) - (deque->start_free);
  else
    data_len = deque->block_size - deque->end_free;

  if (data_len < len)
    {
      offset += len - data_len;
      len = data_len;
    }

  if (0 == len)
    return 0;

  const size_t data_size = deque->data_size;
  memcpy (dst + (data_size * offset),
          deque->end->data +
          (data_size * (deque->block_size - deque->end_free - len)),
          len * data_size);
  deque->end_free += len;

  if (deque->end_free == deque->block_size)
    {
      if (deque->start != deque->end)
        {
          deque->end = deque->end->prev;
          deque->end_free = 0;
        }
      else
        {
          deque->end_free = 0;
          deque->start_free = deque->block_size;
        }
    }

  return len;
}

size_t
remove_from_deque_end (deque_t deque, void *dst, size_t len)
{
  size_t running_len = len;
  while (running_len > 0)
    {
      size_t amount = remove_some_from_deque_end (deque, dst, 0, running_len);
      if (0 == amount)
        {
          // printf("remove_from_deque_end: breaking\n") ;
          break;
        }
      running_len -= amount;
    }
  size_t total = len - running_len;
  if (total < len)
    {
      memmove (dst, dst + (deque->data_size * running_len),
               deque->data_size * total);
    }

  return total;
}

bool
single_remove_from_deque_end (deque_t deque, void *dst)
{
  return 1 == remove_from_deque_end (deque, dst, 1);
}

void
clear_deque (deque_t deque)
{
  assert (deque);
  deque->end = deque->start;
  deque->start_free = 0;
  deque->end_free = deque->block_size;
}
