/* $Id: hashtable.c 710 2006-05-28 23:11:46Z jim $
   teebu - An archiving tool
   Copyright (C) 2006 Jim Farrand

   This program is free software; you can redistribute it and/or modify it
   under the terms of the GNU General Public License as published by the Free
   Software Foundation; either version 2 of the License, or (at your option)
   any later version.

   This program is distributed in the hope that it will be useful, but WITHOUT
   ANY WARRANTY; without even the implied warranty of MERCHANTABILITY or
   FITNESS FOR A PARTICULAR PURPOSE.  See the GNU General Public License for
   more details.

   You should have received a copy of the GNU General Public License along with
   this program; if not, write to the Free Software Foundation, Inc., 51
   Franklin Street, Fifth Floor, Boston, MA  02110-1301  USA
*/

#include <assert.h>
#include <stdio.h>
#include <string.h>

#include "hashtable.h"

#define MIN_ORDER 5

#define LOAD_FACTOR(ht) ((float)ht->hashtable_key_count / (float)ht->hashtable_bucket_count)
#define IS_OVERLOADED(ht) (LOAD_FACTOR(ht) > ht->hashtable_max_load)

// TODO: Copy keys so they can be freed
// Will need to store key copy function and data :/
typedef struct
{
  void *entry_key;
  list_t entry_data;
} entry_t;

struct hashtable
{
  void            *hashtable_user_data;
  key_copy_f      hashtable_key_copy;
  is_key_equal_f  hashtable_is_key_equal;
  key_hash_f      hashtable_key_hash;
  size_t          hashtable_bucket_count, hashtable_key_count, hashtable_data_count;
  float           hashtable_max_load;
  list_t          *hashtable_buckets; // List of entry_t
};

/* Get the hashtable size for the given order. */
static size_t
get_size(unsigned order)
{
  return (1 << order) - 1;
}

/* Find hashtable size greater or equal to the given size. */
static size_t
fix_size(size_t size)
{
  unsigned order = MIN_ORDER;
  size_t result;
  while((result = get_size(order)) < size)
    order++;

  return result;
}

/* Get the next hashtable size. */
static size_t
next_size(size_t size)
{
  return ((size+1)*2)-1;
}

/* Create an empty hashtable. */
hashtable_t
create_hashtable (size_t size, void *user_data, key_copy_f key_copy,
                  is_key_equal_f is_key_equal, key_hash_f key_hash)
{
  hashtable_t ht = malloc (sizeof(struct hashtable));
  if (!ht)
    return NULL;

  size = fix_size (size);

  ht->hashtable_user_data = user_data;
  ht->hashtable_key_copy = key_copy;
  ht->hashtable_is_key_equal = is_key_equal;
  ht->hashtable_key_hash = key_hash;
  ht->hashtable_bucket_count = size;
  ht->hashtable_max_load = 0.85;
  ht->hashtable_key_count = 0;
  ht->hashtable_data_count = 0;
  ht->hashtable_buckets = calloc(size, sizeof (list_t));
  if(!ht->hashtable_buckets)
    {
      free(ht);
      return NULL;
    }

  return ht;
}

static list_t
get_bucket (hashtable_t ht, size_t n, bool create)
{
  list_t bucket = ht->hashtable_buckets[n];
  if (!bucket && create)
    {
      // Need to create the bucket
      bucket = ht->hashtable_buckets[n] = empty_list ();
      if (!bucket)
        return false;
    }
  return bucket;
}

/* Expand a hashtable to reduce the load factor. */
static bool
resize_hashtable (hashtable_t ht)
{
  size_t old_bucket_count = ht->hashtable_bucket_count,
         new_bucket_count = next_size(ht->hashtable_bucket_count);
  assert(new_bucket_count > old_bucket_count);

  list_t *buckets = realloc (ht->hashtable_buckets,
                             new_bucket_count * sizeof (entry_t));
  if (!buckets)
    return false;

  ht->hashtable_buckets = buckets;

  // Initialise the new buckets
  for (size_t i = old_bucket_count; i < new_bucket_count; i++)
    buckets[i] = NULL;

  // Loop through buckets
  for (size_t i = 0; i < old_bucket_count; i++)
    {
      if (buckets[i])
        {
          // Loop through entries in this bucket
          iterator_t it = list_iterator(buckets[i]);
          while (iterator_has_next (it))
            {
              entry_t *entry = iterator_next (it);
              size_t new_bucket_no =
                ht->hashtable_key_hash (ht->hashtable_user_data, entry->entry_key) % new_bucket_count;
              if(new_bucket_no != i)
                {
                  // Need to move this entry to a new bucket
                  iterator_remove (it);
                  list_snoc (get_bucket (ht, new_bucket_no, true), entry);
                }

            }
          release_iterator(it);
        }
    }

  ht->hashtable_bucket_count = new_bucket_count;
  return true;
}

/* Find the entry for the given key, creating it if necessary. */
static entry_t *
find_entry (hashtable_t ht, const void *key, bool create)
{
  // Find the appropriate bucket
  size_t bucket_no = ht->hashtable_key_hash(ht->hashtable_user_data, key) % ht->hashtable_bucket_count;
  list_t bucket = get_bucket (ht, bucket_no, create);
  if (!bucket)
    return NULL;

  // Find the appropriate entry in the bucket
  iterator_t it = list_iterator (bucket);
  entry_t *entry = NULL;
  while (!entry && iterator_has_next (it))
    {
      entry_t *cur_entry = iterator_next (it);
      if (ht->hashtable_is_key_equal (ht->hashtable_user_data, key, cur_entry->entry_key))
        entry = cur_entry;
    }
  release_iterator (it);

  if (!entry && create)
    {
      // Need to create an entry structure
      entry = malloc (sizeof (entry_t));
      if (!entry)
        return NULL;

      if (ht->hashtable_key_copy)
        entry->entry_key = ht->hashtable_key_copy (ht->hashtable_user_data, key);
      else
        entry->entry_key = (void *)key;

      entry->entry_data = empty_list ();
      if (!entry->entry_data)
        {
          free (entry);
          return NULL;
        }

      ht->hashtable_key_count++;
      list_snoc(bucket, entry);
    }

  return entry;
}

void
release_entry(hashtable_t ht, entry_t *entry)
{
  // If the keep was copied, we need to free it
  if (ht->hashtable_key_copy)
    free (entry->entry_key);
  release_list (entry->entry_data);
  free(entry);
}

void
remove_entry (hashtable_t ht, const void *key)
{
  // Find the appropriate bucket
  size_t bucket_no = ht->hashtable_key_hash(ht->hashtable_user_data, key) % ht->hashtable_bucket_count;
  list_t bucket = ht->hashtable_buckets[bucket_no];
  assert(bucket);

  // Find the appropriate entry in the bucket
  iterator_t it = list_iterator (bucket);
  entry_t *entry = NULL;
  while (!entry && iterator_has_next (it))
    {
      entry_t *cur_entry = iterator_next (it);
      if (ht->hashtable_is_key_equal (ht->hashtable_user_data, key, cur_entry->entry_key))
        {
          assert(is_empty_list(cur_entry->entry_data));
          iterator_remove (it);
          release_entry (ht, cur_entry);
          ht->hashtable_key_count--;
          break;
        }
    }
  release_iterator (it);
}

/* Add an entry to a hashtable */
bool
hashtable_add(hashtable_t ht, const void *key, void *data)
{
  entry_t *entry = find_entry (ht, key, true);
  if (!entry)
    return false;

  list_snoc (entry->entry_data, data);
  ht->hashtable_data_count++;
  if (IS_OVERLOADED (ht))
    resize_hashtable (ht);
  return true;
}

/* Replace all entries for the given key. */
bool
hashtable_replace(hashtable_t ht, const void *key, void *data)
{
  entry_t *entry = find_entry (ht, key, true);
  if (!entry)
    return false;

  ht->hashtable_data_count-= list_length (entry->entry_data);
  clear_list (entry->entry_data);
  list_snoc (entry->entry_data, data);
  ht->hashtable_data_count++;

  return true;
}

void *
hashtable_find(hashtable_t ht, const void *key)
{
  entry_t *entry = find_entry (ht, key, false);
  if (!entry)
    return NULL;

  return list_head(entry->entry_data);
}

bool
hashtable_contains_key (hashtable_t ht, const void *key)
{
  entry_t *entry = find_entry (ht, key, false);
  if (!entry)
    return false;

  return true;
}

/* Find and remove an entry. */
void *
hashtable_remove(hashtable_t ht, const void *key)
{
  entry_t *entry = find_entry (ht, key, false);
  if (!entry)
    return NULL;

  if (is_empty_list (entry->entry_data))
    return NULL;

  void *data = remove_list_head (entry->entry_data);
  ht->hashtable_data_count--;
  if (is_empty_list (entry->entry_data))
    remove_entry (ht, key);

  return data;
}

void
hashtable_release(hashtable_t ht)
{
  for (int i = 0; i < ht->hashtable_bucket_count; i++)
    {
      list_t bucket = ht->hashtable_buckets[i];
      if (bucket)
        {
          iterator_t bucket_it = list_iterator(bucket);
          while (iterator_has_next (bucket_it))
            release_entry(ht, iterator_next (bucket_it));
          release_iterator (bucket_it);
          release_list (bucket);
        }
    }
  free (ht->hashtable_buckets);
  free (ht);
}

/*
 * Some hash and comparison functions
 */

bool
is_int_equal(void *ignore, const void * a, const void *b)
{
  return *(int *)a == *(int *)b;
}

bool
is_long_equal(void *ignore, const void * a, const void *b)
{
  return *(long *)a == *(long *)b;
}

size_t
int_hash (void *ignore, const void * a)
{
  return (size_t)(*(int *)a);
}

size_t
long_hash (void *ignore, const void * a)
{
  return (size_t)(*(long *)a);
}

static const unsigned SIZE_T_BITS = 8 * sizeof(size_t);

bool
is_string_equal (void *ignore, const void *a, const void *b)
{
  return 0 == strcmp ((const char *)a, (const char *)b);
}

static size_t
rotate_left(size_t v, unsigned n)
{
  return (v << (n % SIZE_T_BITS)) | (v >> (SIZE_T_BITS - (n % SIZE_T_BITS)));
}

size_t
string_hash (void *ignore, const void *a)
{
  const char *str = a;
  size_t hash = 0;
  for(int i = 0; str[i] != '\0'; i++)
    {
      unsigned rotate = 1 + (str[i] % (SIZE_T_BITS-1));
      hash = rotate_left(hash, rotate);
      hash ^= str[i];
    }

  // printf ("string hash = %zu\n", hash);
  return hash;
}

void *
string_key_copy (void *ignore, const void *a)
{
  return strdup(a);
}

void *
fixed_size_key_copy (void *data, const void *a)
{
  size_t size = (size_t)data;
  void *copy = malloc(size);
  if (!copy)
    return NULL;

  memcpy(copy, a, size);
  return copy;
}

hashtable_t
create_string_hashtable (size_t size, bool copy)
{
  return create_hashtable (size, NULL, copy ? string_key_copy : NULL, is_string_equal, string_hash);
}

hashtable_t
create_int_hashtable (size_t size)
{
  void *data = (void *)(sizeof (int));
  assert (sizeof (int) == (size_t)data);

  return create_hashtable (size, data, fixed_size_key_copy, is_int_equal, int_hash);
}

hashtable_t
create_long_hashtable (size_t size)
{
  void *data = (void *)(sizeof (long));
  assert (sizeof (long) == (size_t)data);

  return create_hashtable (size, data, fixed_size_key_copy, is_long_equal, long_hash);
}

