/* $Id: infpipeio.c 658 2006-05-13 14:50:30Z jim $
   teebu - An archiving tool
   Copyright (C) 2006 Jim Farrand

   This program is free software; you can redistribute it and/or modify it
   under the terms of the GNU General Public License as published by the Free
   Software Foundation; either version 2 of the License, or (at your option)
   any later version.

   This program is distributed in the hope that it will be useful, but WITHOUT
   ANY WARRANTY; without even the implied warranty of MERCHANTABILITY or
   FITNESS FOR A PARTICULAR PURPOSE.  See the GNU General Public License for
   more details.

   You should have received a copy of the GNU General Public License along with
   this program; if not, write to the Free Software Foundation, Inc., 51
   Franklin Street, Fifth Floor, Boston, MA  02110-1301  USA
 */

#include <assert.h>

#include "logging.h"
#include "deque.h"

#include "infpipeio.h"

/* Constants for pipe_inuse. */
#define INUSE_OUT  0x1
#define INUSE_IN   0x2
#define INUSE_USER 0x4

struct infpipe
{
  unsigned pipe_inuse;          // Bit field of above constants
  deque_t pipe_data;
};

static void
my_release_infpipe (infpipe_t data)
{
  if (data->pipe_inuse)
    return;

  release_deque (data->pipe_data);
  free (data);
}

void
release_infpipe (infpipe_t data)
{
  assert (data);
  assert (data->pipe_inuse & INUSE_USER);

  data->pipe_inuse &= ~INUSE_USER;
  my_release_infpipe (data);
}

static output_err_t
my_output_limited (void *uncast_data, iobuffer_t * buf, size_t max)
{
  infpipe_t data = uncast_data;
  if (!data)
    return OUTPUT_ERR_BAD;

  size_t added =
    add_to_deque_end (data->pipe_data, iobuffer_data_pointer (buf), max);
  iobuffer_mark_taken (buf, added);
  return added > 0 ? OUTPUT_OK : OUTPUT_ERR_FAILED;
}

static input_err_t
my_input_limited (void *uncast_data, iobuffer_t * buf, size_t max)
{
  infpipe_t data = uncast_data;
  if (!data)
    return INPUT_ERR_BAD;

  if (is_deque_empty (data->pipe_data))
    return INPUT_EOF;

  size_t removed =
    remove_from_deque_start (data->pipe_data, iobuffer_free_pointer (buf),
                             iobuffer_free_size (buf));
  iobuffer_mark_added (buf, removed);
  return INPUT_OK;
}

static void
my_release_out (void *uncast_data)
{
  infpipe_t data = uncast_data;
  assert (data);
  assert (data->pipe_inuse & INUSE_OUT);

  data->pipe_inuse &= ~INUSE_OUT;
  my_release_infpipe (data);
}

static void
my_release_in (void *uncast_data)
{
  infpipe_t data = uncast_data;
  assert (data);
  assert (data->pipe_inuse & INUSE_IN);

  data->pipe_inuse &= ~INUSE_IN;
  my_release_infpipe (data);
}

static out_stream_type_t my_out_type = {
  .output_limited = my_output_limited,
  .output_mark = NULL,          // Unsupported
  .flush = NULL,
  .close_out = NULL,
  .release_out = my_release_out
};

static in_stream_type_t my_in_type = {
  .input_limited = my_input_limited,
  .input_mark = NULL,           // Unimplemented
  .close_in = NULL,             // Unimplemented
  .release_in = my_release_in
};

bool
infpipe_open (out_stream_t * outs, in_stream_t * ins, infpipe_t * pipe_p)
{
  assert (outs);
  assert (ins);

  infpipe_t pipe = malloc (sizeof (struct infpipe));
  if (!pipe)
    return false;

  pipe->pipe_data = create_deque (1, 4096);
  if (!pipe->pipe_data)
    {
      free (pipe);
      return false;
    }

  pipe->pipe_inuse = INUSE_OUT | INUSE_IN;
  if (pipe_p)
    pipe->pipe_inuse |= INUSE_USER;

  *outs = open_out (&my_out_type, pipe);
  if (!*outs)
    {
      release_deque (pipe->pipe_data);
      free (pipe);
      return false;
    }

  *ins = open_in (&my_in_type, pipe);
  if (!*ins)
    {
      close_and_release_out (*outs);
      release_deque (pipe->pipe_data);
      free (pipe);
      return false;
    }

  if (pipe_p)
    *pipe_p = pipe;

  return true;
}
