/* $Id: iobuffer.h 659 2006-05-13 14:51:08Z jim $
   teebu - An archiving
   Copyright (C) 2006 Jim Farrand

   This program is free software; you can redistribute it and/or modify it
   under the terms of the GNU General Public License as published by the Free
   Software Foundation; either version 2 of the License, or (at your option)
   any later version.

   This program is distributed in the hope that it will be useful, but WITHOUT
   ANY WARRANTY; without even the implied warranty of MERCHANTABILITY or
   FITNESS FOR A PARTICULAR PURPOSE.  See the GNU General Public License for
   more details.

   You should have received a copy of the GNU General Public License along with
   this program; if not, write to the Free Software Foundation, Inc., 51
   Franklin Street, Fifth Floor, Boston, MA  02110-1301  USA
 */


#ifndef IOBUFFER_H
#define IOBUFFER_H

#include <stdbool.h>
#include <malloc.h>
#include <string.h>
#include <stdint.h>

struct iobuffer
{
  size_t size;
  size_t data_start;
  size_t free_start;
  char *data;
};

typedef struct iobuffer iobuffer_t;

void init_iobuffer_with (iobuffer_t *, size_t size, size_t data_size,
                         char *str);

/* Create a buffer with the given size. */
iobuffer_t *create_iobuffer (size_t);

/* Release the buffer. */
void free_iobuffer (iobuffer_t *);

/* Query total size of buffer. */
size_t iobuffer_size (iobuffer_t *);

/* Query free size in buffer. */
size_t iobuffer_free_size (iobuffer_t *);

/* Query used size in buffer. */
size_t iobuffer_data_size (iobuffer_t *);

/* Get a pointer to the free space in the buffer. */
char *iobuffer_free_pointer (iobuffer_t *);

/* Get a pointer to the data in the buffer. */
char *iobuffer_data_pointer (iobuffer_t *);

/* Mark that we have added data to the buffer. */
void iobuffer_mark_added (iobuffer_t *, size_t);

/* Mark that we have removed data from the buffer. */
void iobuffer_mark_taken (iobuffer_t *, size_t);

/* Mark that we have removed all data from the buffer. */
void iobuffer_mark_all_taken (iobuffer_t *);

/* Add the given data to the buffer. */
bool iobuffer_add_data (iobuffer_t *, const char *, size_t);

/* Single character version of iobuffer_add_data. */
#define IOBUFFER_ADD_CHAR(buf,c) iobuffer_add_data((buf), &(c), 1)

/** Return a point to the data block.  NB this is not the data itself (but does
 * contain the data).  Eg call this if you init_withed the buffer, and want to
 * deallocate. */
char *iobuffer_data_block (iobuffer_t *);

/* Copy from src to dst, until either src is empty or dst is full.
 * Return the number of bytes actually copied. */
size_t iobuffer_copy (iobuffer_t * src, iobuffer_t * dst);

/* Copy up to the given number of bytes from the first buffer to the second.
 * Return the number of bytes actually copied. */
size_t iobuffer_copy_limited (iobuffer_t * src, iobuffer_t * dst, size_t len);

/* Shunt all data to the front of the buffer.  This can be used to make space
 * in a buffer that has been read from, but probably should be avoided for
 * performance reasons.  Returns the distance shifted (which is the amount of
 * extra space available in the buffer). */
size_t iobuffer_shunt (iobuffer_t * buf);

#endif
