/* $Id: nonstdio.c 707 2006-05-27 22:36:00Z jim $
   teebu - An archiving tool
   Copyright (C) 2006 Jim Farrand

   This program is free software; you can redistribute it and/or modify it
   under the terms of the GNU General Public License as published by the Free
   Software Foundation; either version 2 of the License, or (at your option)
   any later version.

   This program is distributed in the hope that it will be useful, but WITHOUT
   ANY WARRANTY; without even the implied warranty of MERCHANTABILITY or
   FITNESS FOR A PARTICULAR PURPOSE.  See the GNU General Public License for
   more details.

   You should have received a copy of the GNU General Public License along with
   this program; if not, write to the Free Software Foundation, Inc., 51
   Franklin Street, Fifth Floor, Boston, MA  02110-1301  USA
 */


#include <stdbool.h>
#include <stdio.h>
#include <assert.h>

#include "logging.h"
#include "darray.h"
#include "nonstdio.h"

#define DEFAULT_CALLBACK_SPACE 8

const char *OUTPUT_ERR_NAME[] = {
  "OUTPUT_OK",                  // Output completed succesfully
  "OUTPUT_ERR_CLOSED",          // Output stream is closed
  "OUTPUT_ERR_EMPTY",           // The given buffer had no data
  "OUTPUT_ERR_UNSUPPORTED",     // Stream does not support this operation
  "OUTPUT_ERR_BAD",             // Output stream descriptor is NULL or otherwise bad
  "OUTPUT_ERR_REFUSED",         // Refused, eg no more data allowed
  "OUTPUT_ERR_FAILED"           // Output failed for some other reason
};

const char *INPUT_ERR_NAME[] = {
  "INPUT_OK",
  "INPUT_EOF",
  "INPUT_MARK",
  "INPUT_ERR_CLOSED",
  "INPUT_ERR_FULL",
  "INPUT_ERR_NO_MARK",
  "INPUT_ERR_UNSUPPORTED",
  "INPUT_ERR_BAD",
  "INPUT_ERR_DATALOSS",
  "INPUT_ERR_FAILED"
};

struct out_stream
{
  void *out_data;
  bool out_open;
  out_stream_type_t *out_type;
  on_close_out_t *out_on_close;
};

struct in_stream
{
  void *in_data;
  bool in_open;
  in_stream_type_t *in_type;
};

out_stream_t
open_out (out_stream_type_t * out_type, void *out_data)
{
  assert (out_type);
  out_stream_t r = malloc (sizeof (struct out_stream));
  if (!r)
    return NULL;

  r->out_data = out_data;
  r->out_type = out_type;
  r->out_open = true;
  r->out_on_close = NULL;

  return r;
}

output_err_t
output_limited (out_stream_t outs, iobuffer_t * buf, size_t amount)
{
  if (!outs)
    {
      // printf("output_limited: null\n") ;
      return OUTPUT_ERR_BAD;
    }

  assert (buf);

  if (!outs->out_open)
    return OUTPUT_ERR_CLOSED;

  size_t size = iobuffer_data_size (buf);
  if (size < amount)
    amount = size;

  if (0 == amount)
    return OUTPUT_ERR_EMPTY;

  if (!outs->out_type->output_limited)
    return OUTPUT_ERR_UNSUPPORTED;

  output_err_t err =
    (outs->out_type->output_limited) (outs->out_data, buf, amount);
  if (OUTPUT_OK != err)
    return err;

  assert (iobuffer_data_size (buf) < size);
  return OUTPUT_OK;
}


output_err_t
output (out_stream_t outs, iobuffer_t * buf)
{
  return output_limited (outs, buf, iobuffer_data_size (buf));
}

output_err_t
close_out (out_stream_t outs)
{
  if (!outs)
    return OUTPUT_ERR_BAD;

  if (!outs->out_open)
    return OUTPUT_ERR_CLOSED;

  outs->out_open = false;

  if (outs->out_on_close)
    {
      const size_t hook_count = darray_size (outs->out_on_close);
      for (int i = 0; i < hook_count; i++)
        (outs->out_on_close[i]) (outs);
    }

  if (outs->out_type->close_out)
    return outs->out_type->close_out (outs->out_data);

  return OUTPUT_OK;
}

bool
is_out_open (out_stream_t outs)
{
  assert (outs);

  return outs->out_open;
}

output_err_t
flush (out_stream_t outs)
{
  if (!outs)
    return OUTPUT_ERR_BAD;

  if (!outs->out_open)
    return OUTPUT_ERR_CLOSED;

  if (outs->out_type->flush)
    return outs->out_type->flush (outs->out_data);

  // Default implementation: do nothing
  return OUTPUT_OK;
}

void
release_out (out_stream_t outs)
{
  if (!outs)
    return;

  assert (!outs->out_open);

  if (outs->out_type->release_out)
    outs->out_type->release_out (outs->out_data);

  if (outs->out_on_close)
    release_darray (outs->out_on_close);

  free (outs);
}

output_err_t
close_and_release_out (out_stream_t outs)
{
  if (!outs)
    return OUTPUT_ERR_BAD;
  output_err_t err = close_out (outs);
  release_out (outs);
  return err;
}

output_err_t
output_mark (out_stream_t outs)
{
  if (!outs)
    return OUTPUT_ERR_BAD;

  if (outs->out_type->output_mark)
    return (outs->out_type->output_mark) (outs->out_data);
  else
    return OUTPUT_ERR_UNSUPPORTED;
}

output_err_t
output_all (out_stream_t outs, iobuffer_t * buf)
{
  output_err_t err;

  do
    {
      err = output (outs, buf);
    }
  while (OUTPUT_OK == err && iobuffer_data_size (buf) > 0);

  return err;
}

bool
register_on_close_out (out_stream_t outs, on_close_out_t callback)
{
  assert (outs);

  if (NULL == outs->out_on_close)
    {
      outs->out_on_close =
        create_darray (sizeof (on_close_out_t), DEFAULT_CALLBACK_SPACE, 0);
    }

  outs->out_on_close = darray_add (outs->out_on_close, &callback);
  return NULL != outs->out_on_close;
}

in_stream_t
open_in (in_stream_type_t * in_type, void *in_data)
{
  assert (in_type);
  in_stream_t r = malloc (sizeof (struct in_stream));
  if (!r)
    return NULL;

  r->in_data = in_data;
  r->in_type = in_type;
  r->in_open = true;

  return r;
}

input_err_t
input_limited (in_stream_t ins, iobuffer_t * buf, size_t amount)
{
  if (!ins)
    return INPUT_ERR_BAD;

  size_t size = iobuffer_free_size (buf);
  if (size < amount)
    amount = size;

  if (0 == amount)
    return INPUT_ERR_FULL;

  if (!ins->in_type->input_limited)
    return INPUT_ERR_UNSUPPORTED;

  return ins->in_type->input_limited (ins->in_data, buf, amount);
}

input_err_t
input (in_stream_t ins, iobuffer_t * buf)
{
  return input_limited (ins, buf, iobuffer_free_size (buf));
}

input_err_t
input_all (in_stream_t ins, iobuffer_t * buf)
{
  input_err_t err;

  do
    {
      err = input (ins, buf);
    }
  while (INPUT_OK == err && iobuffer_free_size (buf) > 0);

  return err;
}

input_err_t
input_mark (in_stream_t ins)
{
  if (!ins)
    return INPUT_ERR_BAD;

  if (!ins->in_type->input_mark)
    return INPUT_ERR_UNSUPPORTED;

  return (ins->in_type->input_mark) (ins->in_data);
}

#define SKIP_BUFFER_LEN 4096

input_err_t
skip_in (in_stream_t ins, size_t amount)
{
  char buf[SKIP_BUFFER_LEN];
  iobuffer_t iobuf;
  while (amount > 0)
    {
      init_iobuffer_with (&iobuf, SKIP_BUFFER_LEN, 0, buf);
      input_err_t err = input_limited (ins, &iobuf, amount);
      if (INPUT_OK != err)
        return err;
      amount -= iobuffer_data_size (&iobuf);
    }

  return INPUT_OK;
}

input_err_t
skip_all (in_stream_t ins)
{
  char buf[SKIP_BUFFER_LEN];
  iobuffer_t iobuf;
  while (true)
    {
      init_iobuffer_with (&iobuf, SKIP_BUFFER_LEN, 0, buf);
      input_err_t err = input_limited (ins, &iobuf, SKIP_BUFFER_LEN);
      if (INPUT_OK != err)
        return err;
    }

}

input_err_t
close_in (in_stream_t ins)
{
  if (!ins)
    return INPUT_ERR_BAD;

  if (!ins->in_open)
    return INPUT_ERR_CLOSED;

  ins->in_open = false;

  if (ins->in_type->close_in)
    return ins->in_type->close_in (ins->in_data);

  return INPUT_OK;
}

bool
is_in_open (in_stream_t ins)
{
  if (!ins)
    return INPUT_ERR_BAD;

  return ins->in_open;
}

void
release_in (in_stream_t ins)
{
  if (!ins)
    return;

  assert (!ins->in_open);

  if (ins->in_type->release_in)
    ins->in_type->release_in (ins->in_data);

  free (ins);
}

input_err_t
close_and_release_in (in_stream_t ins)
{
  if (!ins)
    return INPUT_ERR_BAD;

  input_err_t err = close_in (ins);
  release_in (ins);
  return err;
}

input_err_t
input_recover (in_stream_t ins)
{
  if (!ins)
    return INPUT_ERR_BAD;

  if (!ins->in_type->input_recover)
    return INPUT_ERR_UNSUPPORTED;

  return ins->in_type->input_recover (ins->in_data);
}

bool
copy_stream (size_t bufsize, size_t *output_count,
             input_err_t *in_err_ptr, output_err_t *out_err_ptr,
             in_stream_t ins, out_stream_t outs)
{
  char buf[bufsize];
  iobuffer_t iobuf;
  init_iobuffer_with (&iobuf, bufsize, 0, buf);

  if (output_count)
    *output_count = 0;

  input_err_t in_err = INPUT_OK;
  output_err_t out_err = OUTPUT_OK;
  do
    {
      in_err = input (ins, &iobuf);
      if (INPUT_OK == in_err)
        {
          size_t before = iobuffer_data_size (&iobuf);
          out_err = output_all (outs, &iobuf);
          size_t after = iobuffer_data_size (&iobuf);
          if (output_count)
            *output_count += before - after;
        }
    }
  while (INPUT_OK == in_err && OUTPUT_OK == out_err);

  if (in_err_ptr)
    *in_err_ptr = in_err;

  if (out_err_ptr)
    *out_err_ptr = out_err;

  return (INPUT_EOF == in_err || INPUT_MARK == in_err)
    && OUTPUT_OK == out_err;
}

bool
compare_stream (size_t bufsize,
                size_t *compared, in_stream_t ins1, in_stream_t ins2)
{
  char buf1[bufsize], buf2[bufsize];
  iobuffer_t iobuf1, iobuf2;
  init_iobuffer_with (&iobuf1, bufsize, 0, buf1);
  init_iobuffer_with (&iobuf2, bufsize, 0, buf2);

  if (compared)
    *compared = 0;

  input_err_t err1 = INPUT_OK, err2 = INPUT_OK;
  do
    {
      if (0 == iobuffer_data_size (&iobuf1))
        err1 = input (ins1, &iobuf1);

      if (0 == iobuffer_data_size (&iobuf2))
        err2 = input (ins2, &iobuf2);

      if (INPUT_OK == err1 && INPUT_OK == err2)
        {
          if (*iobuffer_data_pointer (&iobuf1) !=
              *iobuffer_data_pointer (&iobuf2))
            {
              DEBUG ("compare_stream: Data not equal");
              return false;
            }

          iobuffer_mark_taken (&iobuf1, 1);
          iobuffer_mark_taken (&iobuf2, 1);

          if (compared)
            (*compared)++;
        }
      else
        {
          DEBUGF ("compare_stream: inputs returned %s %s",
                  INPUT_ERR_NAME (err1), INPUT_ERR_NAME (err2));
        }
    }
  while (INPUT_OK == err1 && INPUT_OK == err2);

  return (INPUT_EOF == err1 || INPUT_MARK == err1) && (INPUT_EOF == err2
                                                       || INPUT_MARK == err2);
}
