/* $Id: nonstdio.h 707 2006-05-27 22:36:00Z jim $
   teebu - An archiving tool
   Copyright (C) 2006 Jim Farrand

   This program is free software; you can redistribute it and/or modify it
   under the terms of the GNU General Public License as published by the Free
   Software Foundation; either version 2 of the License, or (at your option)
   any later version.

   This program is distributed in the hope that it will be useful, but WITHOUT
   ANY WARRANTY; without even the implied warranty of MERCHANTABILITY or
   FITNESS FOR A PARTICULAR PURPOSE.  See the GNU General Public License for
   more details.

   You should have received a copy of the GNU General Public License along with
   this program; if not, write to the Free Software Foundation, Inc., 51
   Franklin Street, Fifth Floor, Boston, MA  02110-1301  USA
 */


#ifndef NONSTDIO_H
#define NONSTDIO_H

#include "iobuffer.h"

/* This type is the return value of several IO function. */
typedef enum
{
  OUTPUT_OK,                    // Output successful
  OUTPUT_ERR_CLOSED,            // Operation failed because stream has been closed
  OUTPUT_ERR_EMPTY,             // Output failed because supplied buffer is empty
  OUTPUT_ERR_UNSUPPORTED,       // Operation not supported by this implementation
  OUTPUT_ERR_BAD,               // Operation failed because output stream
  // structure is invalid
  OUTPUT_ERR_REFUSED,           // Operation refused for implementation depended reason
  OUTPUT_ERR_FAILED             // Operation failed for another unspecified reason
    // Must be last!
} output_err_t;

/* Constant defining the number of different output errors. */
#define OUTPUT_ERR_COUNT (OUTPUT_ERR_FAILED+1)

/* array containing namaes of errors, as strings */
extern const char *OUTPUT_ERR_NAME[OUTPUT_ERR_COUNT];

/* macro which produces the name of an output error */
#define OUTPUT_ERR_NAME(err) (OUTPUT_ERR_NAME[err])

/* A struct of functions which define a particular out stream implementation.
 * This structure is passed to open_out to create an output channel. */
struct out_stream_type
{
  /* output_limited(impl, buf, n) is called to output up to n characters.  n
   * is garunteed to be greater than 0 and at most the amount of data in the
   * buf. */
  output_err_t (*output_limited) (void *, iobuffer_t *, size_t);
  /* called to output a mark */
  output_err_t (*output_mark) (void *);
  /* called to flush a channel (optional) */
  output_err_t (*flush) (void *);
  /* called to close a channel.  (optional) */
  output_err_t (*close_out) (void *);
  /* called to release resources associated with a channel */
  void (*release_out) (void *);
};

typedef struct out_stream_type out_stream_type_t;

/* Here is a blank instanciation of out_stream_type you can copy and paste.

static out_stream_type_t my_out_type = {
  .output_limited  = NULL,     // Unimplemented
  .output_mark     = NULL,     // Unimplemented
  .flush           = NULL,     // Unimplemented
  .close_out       = NULL,     // Unimplemented
  .release_out     = NULL      // Unimplemented
} ;

*/

/* An output stream type. */
typedef struct out_stream *out_stream_t;

/* A callback triggered on file out close. */
typedef void (*on_close_out_t) ();

typedef enum
{
  INPUT_OK,                     // Input sucessful
  INPUT_EOF,                    // Cannot input because EOF found in data
  INPUT_MARK,                   // Cannot input because mark found in data
  INPUT_ERR_CLOSED,             // Operation failed because stream is closed
  INPUT_ERR_FULL,               // Input failed because supplied input buffer
  // has no room
  INPUT_ERR_NO_MARK,            // input_mark called when no mark present in stream
  INPUT_ERR_UNSUPPORTED,        // Operation not supported by this implementation
  INPUT_ERR_BAD,                // Operation failed because the input stream
  // data structure was invalid in some way
  INPUT_ERR_DATALOSS,           // An error occurred which may have resulted in
  // data loss, but may be recoverable
  INPUT_ERR_FAILED              // Input failed for another unspecified reason
    // Must be last!
} input_err_t;

/* Constant defining the number of different input errors. */
#define INPUT_ERR_COUNT (INPUT_ERR_FAILED+1)

/* Macro which produces the name of an input error */
#define INPUT_ERR_NAME(err) (INPUT_ERR_NAME[err])

extern const char *INPUT_ERR_NAME[INPUT_ERR_COUNT];

struct in_stream_type
{
  input_err_t (*input_limited) (void *, iobuffer_t *, size_t);
  // Input data
  input_err_t (*input_mark) (void *);   // Input a mark
  input_err_t (*close_in) (void *);     // Close the channel
  void (*release_in) (void *);  // Release allocated resources
    input_err_t (*input_recover) (void *);      // Attempt to recover after INPUT_ERR_DATALOSS
};

typedef struct in_stream_type in_stream_type_t;

/* Here is a blank instanciation of in_stream_type you can copy and paste.

static in_stream_type_t my_in_type = {
    .input_limited = NULL,     // Unimplemented
    .input_mark    = NULL,     // Unimplemented
    .close_in      = NULL,     // Unimplemented
    .release_in    = NULL,     // Unimplemented
    .input_recover = NULL      // Unimplemented
} ;
*/

/* An input stream type. */
typedef struct in_stream *in_stream_t;

/* Open an output stream. */
out_stream_t open_out (out_stream_type_t *, void *);

/* Send some or all of buffer to output stream.  IO is performed in whatever amount is
 * natural to the underlying implementation. */
output_err_t output (out_stream_t, iobuffer_t *);

/* As output, but takes an additional parameter which is the maximum number of
 * bytes to write.  May be greater than the amount of data in the buffer, in
 * which case will attempt to output the entire buffer. */
output_err_t output_limited (out_stream_t, iobuffer_t *, size_t);

/* As output() above, but keeps going until the whole buffer has been output,
 * or an error occurs. */
output_err_t output_all (out_stream_t, iobuffer_t *);

/* Flush any pending data. */
output_err_t flush (out_stream_t);

/* Close the stream (but do not release the associated memory). */
output_err_t close_out (out_stream_t);

/* Return true iff output stream is open. */
bool is_out_open (out_stream_t);

/* Register a callback triggered when the out stream closes. */
bool register_on_close_out (out_stream_t, on_close_out_t);

/* Release memory associate with the stream.  Do not use the out stream after
 * calling this function. */
void release_out (out_stream_t);

/* Close and release the given out stream. */
output_err_t close_and_release_out (out_stream_t);

/* Output a file mark. */
output_err_t output_mark (out_stream_t);

/* Create an input stream. */
in_stream_t open_in (in_stream_type_t *, void *);

/* Input some data into the given buffer. */
input_err_t input (in_stream_t, iobuffer_t *);

/* Input up to the given amount of data into the given buffer. */
input_err_t input_limited (in_stream_t, iobuffer_t *, size_t);

/* As input above, but keeps going until buffer is filled or an error occurs. */
input_err_t input_all (in_stream_t, iobuffer_t *);

/* Input a mark, if one is pending in the stream. */
input_err_t input_mark (in_stream_t);

/* Skip the given number of bytes. */
input_err_t skip_in (in_stream_t, size_t);

/* Skip until eof, mark or error */
input_err_t skip_all (in_stream_t);

/* Close the stream (but do not release the associated memory). */
input_err_t close_in (in_stream_t);

/* Attempt to recover from an INPUT_ERR_DATALOSS */
input_err_t input_recover (in_stream_t);

/* Return true iff the stream is open. */
bool is_in_open (in_stream_t);

/* Release memory associate with the stream.  Do not use the in stream after
 * calling this function. */
void release_in (in_stream_t);

/* Close and release the given in stream. */
input_err_t close_and_release_in (in_stream_t);

/* Copy all data until EOF or mark from the first stream to the second, in
 * chunks of the given size.  Return true if all data is copied successfully.
 * If the err pointers are not NULL, the last input and output error are placed
 * in the locations pointed to. */
bool copy_stream (size_t, size_t *output_count, input_err_t *, output_err_t *,
                  in_stream_t, out_stream_t);

/** Compare two streams, consuming contents of both, and returning true iff all
 * data is identical.   If compared is non-null, the referred value is set
 * to the number of bytes compared. */
bool compare_stream (size_t buffsize, size_t * compared, in_stream_t,
                     in_stream_t);

#endif
