/* $Id: teebumain.c 737 2006-06-13 20:19:04Z jim $
   teebu - An archiving tool
   Copyright (C) 2006 Jim Farrand

   This program is free software; you can redistribute it and/or modify it
   under the terms of the GNU General Public License as published by the Free
   Software Foundation; either version 2 of the License, or (at your option)
   any later version.

   This program is distributed in the hope that it will be useful, but WITHOUT
   ANY WARRANTY; without even the implied warranty of MERCHANTABILITY or
   FITNESS FOR A PARTICULAR PURPOSE.  See the GNU General Public License for
   more details.

   You should have received a copy of the GNU General Public License along with
   this program; if not, write to the Free Software Foundation, Inc., 51
   Franklin Street, Fifth Floor, Boston, MA  02110-1301  USA
 */


#include <assert.h>
#include <fcntl.h>
#include <regex.h>
#include <signal.h>
#include <stdio.h>
#include <stdlib.h>
#include <sys/types.h>
#include <sys/wait.h>
#include <unistd.h>

#include "baseio.h"
#include "bufferio.h"
#include "bzip2io.h"
#include "catio.h"
#include "cmdline.h"
#include "debugio.h"
#include "filestore.h"
#include "gpgio.h"
#include "gzipio.h"
#include "list.h"
#include "logging.h"
#include "nomarkio.h"
#include "remarkio.h"
#include "selfsyncio.h"
#include "simulateerrorio.h"
#include "statio.h"
#include "unixio.h"
#include "unixmisc.h"

#define SETUP_FAILED 64
#define ACTION_FAILED 32

// Uncomennt this to get stream debugging
// #define DEBUG_STREAM

/* Identify the action requested by the user. */
typedef enum
{
  CREATE,
  EXTRACT,
  IDENTIFY,
  SELFTEST,
  TEST,
  VERIFY
} action_t;

static bool
setup_out_stream (struct gengetopt_args_info *args, iostat_t * stats,
                  out_stream_t * outs)
{
  assert (outs);

#ifdef DEBUG_STREAM
  {
    out_stream_t debug_outs =
      debugio_open_out (*outs, "Data ", 16, true, 0, DEBUGIO_MAX_LEN, true);
    if (!debug_outs)
      {
        LOG (ERROR, "Couldn't open debug stream");
        return false;
      }

    *outs = debug_outs;
  }
#endif

  if (args->tape_blocks_given)
    {
      out_stream_t bufferio_outs =
        bufferio_open_out (*outs, args->tape_blocks_arg, true, true);
      if (!bufferio_outs)
        {
          LOG (ERROR, "Couldn't open bufferio stream");
          return false;
        }

      *outs = bufferio_outs;
    }

#ifdef DEBUG_STREAM
  {
    out_stream_t debug_outs =
      debugio_open_out (*outs, "Blocks", 16, true, 0, DEBUGIO_MAX_LEN, true);
    if (!debug_outs)
      {
        LOG (ERROR, "Couldn't open debug stream");
        return false;
      }

    *outs = debug_outs;
  }
#endif

  // Outut stats
  {
    out_stream_t stat_outs = statio_open_out (*outs, stats, false, true);
    if (!stat_outs)
      {
        LOG (ERROR, "Couldn't open stat stream");
        return false;
      }
    LOG (DEBUG, "Created stats output layer");
    *outs = stat_outs;
  }

  return true;
}

static const float MEG_F = 1024 * 1024;

static void
do_output_stats (time_t time_d, iostat_t * stats, bool out)
{
  if (out)
    {
      LOGF (NORMAL,
            "Chnk: %3zu  In: %5.0fM (%5.2fM/s) Out: %5.0fM (%5.2fM/s) Cmpr: %4.1f%%",
            stats->stats_output_marks + stats->stats_input_marks,
            (float) stats->stats_bytes_read / MEG_F,
            (float) stats->stats_bytes_read / (MEG_F * time_d),
            (float) stats->stats_bytes_written / MEG_F,
            (float) stats->stats_bytes_written / (MEG_F * time_d),
            100.0 * (1.0 -
                     ((float) stats->stats_bytes_written /
                      (float) stats->stats_bytes_read)));
    }
  else
    {
      LOGF (NORMAL,
            "Chnk: %3zu Out: %5.0fM (%5.2fM/s)  In: %5.0fM (%5.2fM/s) Cmpr: %4.1f%%",
            stats->stats_output_marks + stats->stats_input_marks,
            (float) stats->stats_bytes_written / MEG_F,
            (float) stats->stats_bytes_written / (MEG_F * time_d),
            (float) stats->stats_bytes_read / MEG_F,
            (float) stats->stats_bytes_read / (MEG_F * time_d),
            100.0 * (1.0 -
                     ((float) stats->stats_bytes_read /
                      (float) stats->stats_bytes_written)));
    }

  if (stats->stats_path)
    {
      TLOGF(NORMAL, "Processed files in: %.40s", path_str(stats->stats_path));
      clear_stats_path (stats);
    }
}

static void
stats_callback (void *uncast_data, path_t path)
{
  assert (uncast_data);
  void **callback_data = uncast_data;
  iostat_t *stats = callback_data[0];
  size_t *chunks_output = callback_data[1];
  bool out = *(bool *) callback_data[2];
  assert (stats);
  assert (chunks_output);

  time_t time_d = time (NULL) - stats->stats_created;
  const size_t total_marks =
    stats->stats_output_marks + stats->stats_input_marks;
  if (total_marks > *chunks_output)
    {
      *chunks_output = total_marks;
      do_output_stats (time_d, stats, out);
    }
}

/* Read a line from the input stream.  buf is a pointer to a buffer to store
 * the line in.  buf_size is a pointer to the size of the buffer.  input_line
 * will reallocate the buffer if it is too small to fit the line, in which case
 * the size will be updated.  Returns a pointer to the buffer, or NULL if there
 * are no more lines.  In this case, the buffer will also be freed.  */
static char *
input_line (in_stream_t ins, char *buf, size_t * buf_size)
{
  assert (ins);
  assert (buf_size);

  char iobuffer_data[1];
  iobuffer_t iobuffer;
  init_iobuffer_with (&iobuffer, 1, 0, iobuffer_data);

  size_t buf_pos = 0;
  input_err_t err;
  while (INPUT_OK == (err = input (ins, &iobuffer)))
    {
      // +1 for terminator
      if (buf_pos + 1 >= *buf_size)
        {
          *buf_size *= 2;
          if (0 == *buf_size)
            *buf_size = 256;

          buf = realloc (buf, *buf_size);
          if (!buf)
            return NULL;
        }

      buf[buf_pos] = *iobuffer_data_pointer (&iobuffer);
      iobuffer_mark_taken (&iobuffer, 1);

      if ('\n' == buf[buf_pos])
        break;

      buf_pos++;
    }

  buf[buf_pos] = '\0';

  if (0 == buf_pos && INPUT_OK != err)
    {
      free (buf);
      return NULL;
    }

  return buf;
}

/* Create an archive, outputting to given output stream, using the given args
 * structure.  All args related to the output location are ignored (because
 * they should have be processed before now. */
static void
create_to (struct gengetopt_args_info *args, filestore_stats_t * fsstats,
           out_stream_t outs)
{
  LOG (DEBUG, "Creating archive");

  iostat_t *stats = create_stats ();

  if (!setup_out_stream (args, stats, &outs))
    {
      LOG (ERROR, "Error: Couldn't setup output");
      close_and_release_out (outs);
      return;
    }

  size_t chunks_output = 0;
  bool out = true;
  void *callback_data[] = { stats, &chunks_output, &out };

  void (*my_stats_callback) (void *, path_t);
  void *my_callback_data;
  if (args->show_progress_given || args->verbose_given)
    {
      my_stats_callback = stats_callback;
      my_callback_data = callback_data;
    }
  else
    {
      my_stats_callback = NULL;
      my_callback_data = NULL;
    }

  filestore_out_params_t params;
  init_filestore_out_params(&params, outs);

  params.fop_label = args->label_arg;
  params.fop_md5sum = args->md5_given;
  params.fop_sha1sum = args->sha1_given;
  params.fop_gzip = args->gzip_given;
  params.fop_bzip2 = args->bzip2_given;
  params.fop_gpg_passphrase = args->gpg_arg;
  params.fop_chunk_size = args->chunk_arg;
  params.fop_iostats = stats;

  if (args->hide_info_given)
    {
      if (!args->gpg_arg)
        {
          LOG (WARNING, "Warning: --hide-info without --gpg does not secure archive info");
        }

      params.fop_info_after_filters = args->hide_info_given;
    }

  if (args->hide_id_given)
    {
      if (!args->gpg_arg)
        {
          LOG (WARNING, "Fatal: --hide-id without --gpg does not secure archive id");
        }

      params.fop_id_after_filters = args->hide_id_given;
    }

  filestore_out_t fso =
    open_filestore_out (&params);

  if (0 == args->inputs_num && !args->input_filelist_given)
    {
      LOG (WARNING, "Warning: No files to write");
    }
  else
    {
      // Files given on the command line
      for (int i = 0; i < args->inputs_num; i++)
        {
          path_t path = make_path (args->inputs[i]);
          filestore_output_path (fso, fsstats, path, true, my_stats_callback,
                                 my_callback_data);
          release_path (path);
        }

      if (args->input_filelist_given)
        {
          LOG (DEBUG, "Reading files from stdin");
          in_stream_t ins = baseio_stdin ();
          assert (ins);

          char *line = NULL;
          size_t line_len = 0;
          while (NULL != (line = input_line (ins, line, &line_len)))
            {
              path_t path = make_path (line);
              filestore_output_path (fso, fsstats, path, false, my_stats_callback,
                                     my_callback_data);
              release_path (path);
            }
        }
    }

  LOG (DEBUG, "Write done");

  close_filestore_out (fso);
  release_filestore_out (fso);

  close_and_release_out (outs);

  if (args->show_progress_given || args->show_final_given
      || args->verbose_given)
    {
      time_t time_d = time (NULL) - stats->stats_created;
      do_output_stats (time_d, stats, true);
    }

}

/* Setup input layers.  The argument ins must point to the base stream when
 * calling, and is updated with the resulting stream.  False is returned on
 * error, but ins may still have een modified (eg if one stream is opened but
 * not the other.
 */
static bool
setup_in_stream (struct gengetopt_args_info *args, iostat_t * stats,
                 in_stream_t * ins)
{
  assert (ins);

  {
    // Simulated input errors
    for (int i = 0; i < args->simulate_error_given; i++)
      {
        failure_mode_t fm;
        if (!parse_failure_mode (args->simulate_error_arg[i], &fm))
          {
            LOG (FATAL, "Couldn't parse failure mode");
            return false;
          }
        in_stream_t error_ins = simulateerrorio_open_in (&fm, *ins, true);
        if (!error_ins)
          {
            LOG (FATAL, "Couldn't open error stream");
            return false;
          }
        LOG (DEBUG, "Created error simulating input layer");
        *ins = error_ins;
      }
  }

#ifdef DEBUG_STREAM
  {
    in_stream_t debug_ins =
      debugio_open_in (*ins, "Data ", 16, true, 0, DEBUGIO_MAX_LEN, true);
    if (!debug_ins)
      {
        LOG (ERROR, "Couldn't open debug stream");
        LOG (DEBUG, "Created debug input layer");
        return false;
      }

    *ins = debug_ins;
  }
#endif

  return true;
}

static void
input_from (struct gengetopt_args_info *args,
             filestore_stats_t * fsstats, in_stream_t ins,
             action_t action)
{
  LOG (DEBUG, "Extracting archive");

  iostat_t *iostats = create_stats ();

  if (!setup_in_stream (args, iostats, &ins))
    {
      close_in (ins);
      release_in (ins);
      return;
    }

  filestore_in_params_t params;
  init_filestore_in_params (&params, ins);
  params.fip_md5sum = args->md5_given;
  params.fip_sha1sum = args->sha1_given;
  params.fip_gpg_passphrase = args->gpg_arg;
  params.fip_gzip = args->gzip_given;
  params.fip_bzip2 = args->bzip2_given;
  params.fip_ignore_header = args->ignore_header_given;
  params.fip_iostats = iostats;

  // Compile regexps given on the command line
  if (args->inputs_num > 0)
    {
      list_t regexs = empty_list ();
      bool added = false;

      for (int i = 0; i < args->inputs_num; i++)
        {
          regex_t *preg = malloc (sizeof (regex_t));
          if (!preg)
            MEMFAILED();

          if (0 != regcomp(preg, args->inputs[i], REG_NOSUB | REG_EXTENDED))
            {
              LOGF (WARNING, "Invalid regexp: %s", args->inputs[i]);
              free (preg);
            }
          else
            {
              list_snoc (regexs, preg);
              added = true;
            }
        }

      if (added)
        params.fip_regexs = regexs;
      else
        {
          // None found - free the list
          release_list (regexs);
          params.fip_regexs = NULL;
        }
    }

  filestore_in_t fsi =
    open_filestore_in (&params);

  if (!fsi)
    {
      LOG (FATAL, "Couldn't open archive");
      close_and_release_in (ins);
      return;
    }

  size_t chunks_output = 0;
  bool out = false;
  void *callback_data[] = { iostats, &chunks_output, &out };
  void (*my_stats_callback) (void *, path_t);
  void *my_callback_data;
  if (args->show_progress_given || args->verbose_given)
    {
      my_stats_callback = stats_callback;
      my_callback_data = callback_data;
    }
  else
    {
      my_stats_callback = NULL;
      my_callback_data = NULL;
    }


  path_t root = make_path (args->target_arg);
  bool ok ;
  switch (action)
    {
    case EXTRACT:
      ok = filestore_extract_in (fsi, fsstats, root, my_stats_callback,
                                 my_callback_data);
      break;
    case TEST:
      filestore_check_in (fsi, fsstats, root, my_stats_callback,
                          my_callback_data);
      break;
    case VERIFY:
      ok = filestore_verify_in (fsi, fsstats, root, my_stats_callback,
                                my_callback_data);
      break;
    case IDENTIFY:
      filestore_identify_in (fsi, fsstats);
      break;
    default:
      // SNAFU
      STAT_INCR (fsstats, STAT_INTERNAL_ERROR);
      LOG(FATAL, "Requested action not implemented (in perform_input_action)");
      break;
    }
  
  release_path (root);

  LOG (DEBUG, "Extraction complete");

  close_filestore_in (fsi);
  release_filestore_in (fsi);
  close_in (ins);

  release_in (ins);

  if (params.fip_regexs)
    {
      iterator_t it = list_iterator (params.fip_regexs);
      while (iterator_has_next (it))
        regfree (iterator_next (it));
      release_iterator (it);
      release_list (params.fip_regexs);
    }

  if (args->show_progress_given || args->show_final_given
      || args->verbose_given)
    {
      time_t time_d = time (NULL) - iostats->stats_created;
      do_output_stats (time_d, iostats, true);
    }
}

/* Show final status and compute return value. */
static int
show_final_status (filestore_stats_t *stats)
{
  print_filestore_stats (stats);
  return compute_return_code (stats);
}

static int
selftest_thread (struct gengetopt_args_info *args, filestore_stats_t * stats,
                 out_stream_t outs)
{
  assert (args);
  assert (outs);

  create_to (args, stats, outs);
  show_final_status (stats);
  DEBUG ("Selftest write thread done");
  return 0;
}

/* Perform a self test.  An archive is created and verified (without writing to
 * disk). */
static int
perform_selftest (struct gengetopt_args_info *args, filestore_stats_t * fsstats)
{

  if (args->file_arg)
    LOG (WARNING, "Warning: Ignoring -f/--file due to selftest");

  // Create a pipe
  int filedes[2];

  if (!close_on_exec_pipes (filedes))
    {
      return SETUP_FAILED;
    }

  in_stream_t pipe_in = unixio_open_in_fd (filedes[0]);
  out_stream_t pipe_out = unixio_open_out_fd (filedes[1]);

  const int pid = fork ();
  if (-1 == pid)
    {
      // It's all gone horribly wrong
      close_and_release_in (pipe_in);
      close_and_release_out (pipe_out);
      LOG (FATAL, "Error: Failed to create thread (in selftest)");
      return SETUP_FAILED;
    }
  else if (0 == pid)
    {
      // In the child thread
      DEBUG ("Selftest write thread started");

      // Close the end of the pipe we dont want
      close_and_release_in (pipe_in);   // Child does the output, not the input

      set_logging_prefix ("C: ");
      // Write the archive to the pipe
      // Return something that indicated success
      exit (selftest_thread (args, fsstats, pipe_out));
    }
  else
    {
      // In the parent thread

      // Close the end of the pipe we dont want
      close_and_release_out (pipe_out); // Child does the output, not the input

      set_logging_prefix ("V: ");
      // Verify the archive from the pipe
      input_from (args, fsstats, pipe_in, VERIFY);

      // Cleanup the child thread
      int status;
      if (pid != waitpid (pid, &status, 0))
        {
          LOG (FATAL, "Error waiting for selfest write thread");
          return ACTION_FAILED;
        }

      return status;
    }
}

/* Perform the specified input action, reading needed paramaters from args, and
 * adding stats to fsstats if non-NULL. */
static void
perform_input_action (struct gengetopt_args_info *args, filestore_stats_t *fsstats, action_t action)
{
  in_stream_t base_ins;

  if (args->file_arg && 0 != strcmp ("-", args->file_arg))
    {
      base_ins = baseio_open_in (args->file_arg, "r");
    }
  else
    {
      base_ins = unixio_open_in_fd (0);
    }

  if (!base_ins)
    {
      LOG (ERROR, "Unable to open input file");
      return;
    }

  input_from (args, fsstats, base_ins, action);
}

/* Perform the specified output action, reading needed paramaters from args, and
 * adding stats to fsstats if non-NULL. */
static void
perform_output_action (struct gengetopt_args_info *args, filestore_stats_t *fsstats, action_t action)
{
  out_stream_t base_outs;

  if (args->file_arg && 0 != strcmp ("-", args->file_arg))
    {
      base_outs = baseio_open_out (args->file_arg, "w");
    }
  else
    {
      base_outs = unixio_open_out_fd (1);
    }

  if (!base_outs)
    {
      LOG (ERROR, "Unable to open output file");
      return;
    }

  switch (action)
    {
    case CREATE:
      create_to (args, fsstats, base_outs);
      break;
    default:
      // SNAFU
      STAT_INCR (fsstats, STAT_INTERNAL_ERROR);
      LOG(FATAL, "Requested action not implemented (in perform_output_action)");
      break;
    }
}

/* Perform the specified action, reading needed paramaters from args, and
 * adding stats to fsstats if non-NULL. */
static void
perform_action (struct gengetopt_args_info *args, filestore_stats_t *fsstats, action_t action)
{
  switch (action)
    {
    case EXTRACT:
    case VERIFY:
    case TEST:
    case IDENTIFY:
      perform_input_action (args, fsstats, action);
      break;
    case CREATE:
      perform_output_action (args, fsstats, action);
      break;
    case SELFTEST:
      perform_selftest (args, fsstats);
      break;
    default:
      // SNAFU
      STAT_INCR (fsstats, STAT_INTERNAL_ERROR);
      LOG(FATAL, "Requested action not implemented (in perform_action)");
      break;
    }
}

int
main (int argc, char **argv)
{
  struct gengetopt_args_info args_info;

  // We prefer to deal with this synchronously
  signal (SIGPIPE, SIG_IGN);

  // Call our cmdline parser
  if (cmdline_parser (argc, argv, &args_info) != 0)
    exit (1);

  if (args_info.debug_given)
    init_logging (stderr, DEBUG);
  else if (args_info.verbose_given)
    init_logging (stderr, VERBOSE);
  else if (args_info.silent_given)
    init_logging (stderr, FATAL);
  else if (args_info.quiet_given)
    init_logging (stderr, ERROR);
  else
    init_logging (stderr, NORMAL);

  filestore_stats_t stats;
  init_filestore_stats (&stats);

  if (args_info.selftest_given)
    perform_action (&args_info, &stats, SELFTEST);
  else if (args_info.create_given)
    perform_action (&args_info, &stats, CREATE);
  else if (args_info.verify_given)
    perform_action (&args_info, &stats, VERIFY);
  else if (args_info.test_given)
    perform_action (&args_info, &stats, TEST);
  else if (args_info.extract_given)
    perform_action (&args_info, &stats, EXTRACT);
  else if (args_info.identify_given)
    perform_action (&args_info, &stats, IDENTIFY);
  else
    {
      LOG (FATAL, "Action not implemented, please try again later");
    }

  return show_final_status (&stats);
}
