/* $Id: teeio.c 658 2006-05-13 14:50:30Z jim $
   teebu - An archiving tool
   Copyright (C) 2006 Jim Farrand

   This program is free software; you can redistribute it and/or modify it
   under the terms of the GNU General Public License as published by the Free
   Software Foundation; either version 2 of the License, or (at your option)
   any later version.

   This program is distributed in the hope that it will be useful, but WITHOUT
   ANY WARRANTY; without even the implied warranty of MERCHANTABILITY or
   FITNESS FOR A PARTICULAR PURPOSE.  See the GNU General Public License for
   more details.

   You should have received a copy of the GNU General Public License along with
   this program; if not, write to the Free Software Foundation, Inc., 51
   Franklin Street, Fifth Floor, Boston, MA  02110-1301  USA
 */


#include <assert.h>

#include "teeio.h"

struct teeio_out_data
{
  out_stream_t out_base, out_copy;
  bool out_own_base, out_own_copy;
};

static output_err_t
my_output_limited (void *uncast_data, iobuffer_t * buf, size_t max)
{
  struct teeio_out_data *data = uncast_data;
  if (!data)
    return OUTPUT_ERR_BAD;

  char *data_ptr = iobuffer_data_pointer (buf);
  size_t size_before = iobuffer_data_size (buf);
  output_err_t err1 = output_limited (data->out_base, buf, max);
  size_t total = size_before - iobuffer_data_size (buf);

  output_err_t err2 = OUTPUT_OK;
  if (total > 0)
    {
      iobuffer_t buf2;
      init_iobuffer_with (&buf2, total, total, data_ptr);
      err2 = output_all (data->out_copy, &buf2);
    }

  if (OUTPUT_OK != err1)
    return err1;

  return err2;
}

static output_err_t
my_close_out (void *uncast_data)
{
  struct teeio_out_data *data = uncast_data;
  if (!data)
    return OUTPUT_ERR_BAD;

  output_err_t
    err1 = close_out (data->out_base), err2 = close_out (data->out_copy);

  if (OUTPUT_OK != err1)
    return err1;

  return err2;
}

static void
my_release_out (void *uncast_data)
{
  struct teeio_out_data *data = uncast_data;
  assert (data);

  if (data->out_own_base)
    release_out (data->out_base);

  if (data->out_own_copy)
    release_out (data->out_copy);

  free (data);
}

static out_stream_type_t my_out_type = {
  .output_limited = my_output_limited,
  .output_mark = NULL,          // Unimplemented
  .flush = NULL,                // Unimplemented
  .close_out = my_close_out,
  .release_out = my_release_out
};

out_stream_t
teeio_open_out (out_stream_t base, bool own_base, out_stream_t copy,
                bool own_copy)
{
  struct teeio_out_data *data = malloc (sizeof (struct teeio_out_data));
  if (!data)
    return NULL;

  data->out_base = base;
  data->out_own_base = own_base;
  data->out_copy = copy;
  data->out_own_copy = own_copy;

  return open_out (&my_out_type, data);
}

struct teeio_in_data
{
  in_stream_t in_base;
  out_stream_t in_copy;
  bool in_own_base, in_own_copy;
};

static input_err_t
my_input_limited (void *uncast_data, iobuffer_t * buf, size_t max)
{
  struct teeio_in_data *data = uncast_data;
  if (!data)
    return INPUT_ERR_BAD;

  char *free_ptr = iobuffer_free_pointer (buf);
  size_t space_before = iobuffer_free_size (buf);
  input_err_t err1 = input_limited (data->in_base, buf, max);
  size_t total = space_before - iobuffer_free_size (buf);

  output_err_t err2 = OUTPUT_OK;
  if (total > 0)
    {
      iobuffer_t buf2;
      init_iobuffer_with (&buf2, total, total, free_ptr);
      err2 = output_all (data->in_copy, &buf2);
    }

  if (INPUT_OK != err1)
    return err1;

  if (OUTPUT_OK != err2)
    return INPUT_ERR_FAILED;

  return INPUT_OK;
}

static input_err_t
my_close_in (void *uncast_data)
{
  struct teeio_in_data *data = uncast_data;
  if (!data)
    return INPUT_ERR_BAD;

  input_err_t err1 = close_in (data->in_base);
  output_err_t err2 = close_out (data->in_copy);

  if (INPUT_OK != err1)
    return err1;

  if (OUTPUT_OK != err2)
    return INPUT_ERR_FAILED;

  return INPUT_OK;
}

static void
my_release_in (void *uncast_data)
{
  struct teeio_in_data *data = uncast_data;
  assert (data);

  if (data->in_own_base)
    release_in (data->in_base);

  if (data->in_own_copy)
    release_out (data->in_copy);

  free (data);
}

static in_stream_type_t my_in_type = {
  .input_limited = my_input_limited,
  .input_mark = NULL,           // Unimplemented
  .close_in = my_close_in,
  .release_in = my_release_in
};

in_stream_t
teeio_open_in (in_stream_t base, bool own_base, out_stream_t copy,
               bool own_copy)
{
  struct teeio_in_data *data = malloc (sizeof (struct teeio_in_data));
  if (!data)
    return NULL;

  data->in_base = base;
  data->in_own_base = own_base;
  data->in_copy = copy;
  data->in_own_copy = own_copy;

  return open_in (&my_in_type, data);
}
