/* $Id: check_md5sumio.c 659 2006-05-13 14:51:08Z jim $
   teebu - An archiving tool
   Copyright (C) 2006 Jim Farrand

   This program is free software; you can redistribute it and/or modify it
   under the terms of the GNU General Public License as published by the Free
   Software Foundation; either version 2 of the License, or (at your option)
   any later version.

   This program is distributed in the hope that it will be useful, but WITHOUT
   ANY WARRANTY; without even the implied warranty of MERCHANTABILITY or
   FITNESS FOR A PARTICULAR PURPOSE.  See the GNU General Public License for
   more details.

   You should have received a copy of the GNU General Public License along with
   this program; if not, write to the Free Software Foundation, Inc., 51
   Franklin Street, Fifth Floor, Boston, MA  02110-1301  USA
 */

#include <stdlib.h>
#include <stdio.h>
#include <string.h>
#include <assert.h>

#include "baseio.h"
#include "nullio.h"
#include "md5sumio.h"
#include "check_io.h"
#include "check_md5sumio.h"


#define TEST_FILE "/tmp/bumtest.md5sumio.tmp"

static int
read_write_test_size ()
{
  switch (get_test_size ())
    {
    case QUICK_TEST:
      return 256;
    case NORMAL_TEST:
      return 1024;
    case LONG_TEST:
      return 10240;
    default:
      fail ("Invalid test size");
      return -1;                // Not reached, prevent compiler warning
    }
}

static void
check_read_write (int block_size)
{
  const int
    test_size = read_write_test_size (), count = test_size / block_size;

  assert (count > 0);
  assert (test_size > 0);

  out_stream_t outs = baseio_open_out (TEST_FILE, "w");
  fail_if (!outs);

  char buffer1[MD5_SUM_LEN];
  outs = md5sumio_open_out (outs, true, buffer1);
  fail_if (!outs);

  ascending_output_half_test (false, outs, block_size, count);
  release_out (outs);

  mark_point ();

  in_stream_t ins = baseio_open_in (TEST_FILE, "r");
  fail_if (!ins);

  char buffer2[MD5_SUM_LEN];
  ins = md5sumio_open_in (ins, true, buffer2);
  fail_if (!ins);

  ascending_input_half_test (true, false, ins, block_size, count);
  release_in (ins);

  if (0 != strcmp (buffer1, buffer2))
    {
      char msg[256];
      snprintf (msg, 256, "Expected \"%s\" but got \"%s\"", buffer1, buffer2);
      fail (msg);
    }                           /* else {
                                   printf("\nout sum = %s\n", buffer1) ;
                                   printf("in  sum = %s\n", buffer2) ;
                                   } */
}

START_TEST (check_read_write_small_blocks)
{
  check_read_write (1);
}

END_TEST
START_TEST (check_read_write_medium_blocks)
{
  check_read_write (32);
}

END_TEST
START_TEST (check_read_write_large_blocks)
{
  check_read_write (256);
}
END_TEST static void
check_known_string (char *str, char *expected_sum)
{
  out_stream_t null_out = nullio_open_out ();
  fail_if (!null_out);

  char buffer1[MD5_SUM_LEN];
  out_stream_t outs = md5sumio_open_out (null_out, true, buffer1);
  fail_if (!outs);

  iobuffer_t buf;
  init_iobuffer_with (&buf, strlen (str), strlen (str), str);
  fail_unless_out_status (OUTPUT_OK, output_all (outs, &buf));

  close_and_release_out (outs);

  if (0 != strcmp (buffer1, expected_sum))
    {
      char msg[256];
      snprintf (msg, 256, "Expected \"%s\" but got \"%s\"", buffer1,
                expected_sum);
      fail (msg);
    }                           /* else {
                                   printf("\nGot sum      = %s\n", buffer1) ;
                                   printf("Expected sum = %s\n", expected_sum) ;
                                   } */
}

START_TEST (check_hello)
{
  check_known_string ("Hello", "8b1a9953c4611296a827abf8c47804d7");
}

END_TEST
START_TEST (check_bum)
{
  check_known_string ("backupmonster - An archiving tool",
                      "e83791b256f9caab9882350483caa390");
}
END_TEST void
add_md5sumio_tests (Suite * s)
{
  TCase *tc_core = tcase_create ("MD5IO");
  tcase_add_test (tc_core, check_read_write_small_blocks);
  tcase_add_test (tc_core, check_read_write_medium_blocks);
  tcase_add_test (tc_core, check_read_write_large_blocks);
  tcase_add_test (tc_core, check_hello);
  tcase_add_test (tc_core, check_bum);

  suite_add_tcase (s, tc_core);
}

/*

   0f343b0931126a20f133d67c2b018a3b
   0f343b0931126a20f133d67c2b018a3b0f343b0931126a20f133d67c2b018a3b:
*/
