/* tensor/tensor_char.h
 *
 * Copyright (C) 2003, 2004, 2007 Jordi Burguet-Castell
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or (at
 * your option) any later version.
 *
 * This program is distributed in the hope that it will be useful, but
 * WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to:
 *   Free Software Foundation, Inc.
 *   51 Franklin Street, Fifth Floor, Boston, MA  02110-1301  USA.
 */

/*
 * This file contains the basic declarations for a tensor.
 */
#ifndef __TENSOR_char_H__
#define __TENSOR_char_H__

#include <stdlib.h>


#include <gsl/gsl_types.h>
#include <gsl/gsl_errno.h>
#include <gsl/gsl_check_range.h>
#include <gsl/gsl_matrix.h>
#include <gsl/gsl_vector.h>

#include "tensor_utilities.h"

#undef __BEGIN_DECLS
#undef __END_DECLS
#ifdef __cplusplus
# define __BEGIN_DECLS extern "C" {
# define __END_DECLS }
#else
# define __BEGIN_DECLS /* empty */
# define __END_DECLS /* empty */
#endif

__BEGIN_DECLS


/*
 * A tensor is a struct with the rank of the tensor (number of indices
 * it has), the dimension of the vectorial space it is in (number of
 * different possible values for each index) and an array to store the
 * dimension^rank values.
 *
 * For the moment, there is no tda, as opossed to matrices, because it
 * would complicate quite a bit the algorithms and probably it is not
 * worth it.
 */
typedef struct
{
  unsigned int rank;
  size_t dimension;
  size_t size;
  char * data;
} tensor_char;


/*
 * There is not such a thing as "tensor views", in contrast with the
 * case for gsl_matrix.
 */

/* Allocation */

tensor_char *
tensor_char_alloc(const unsigned int rank, const size_t dimension);

tensor_char *
tensor_char_calloc(const unsigned int rank, const size_t dimension);

tensor_char *
tensor_char_copy(tensor_char * t);

void tensor_char_free(tensor_char * t);


/* Views */

/*
 * There are no views.
 */


/* Conversions */

gsl_matrix_char * tensor_char_2matrix(tensor_char * t);
gsl_vector_char * tensor_char_2vector(tensor_char * t);


/* Operations */

char tensor_char_get(const tensor_char * t, const size_t * indices);
void tensor_char_set(tensor_char * t, const size_t * indices, const char x);


char * tensor_char_ptr(tensor_char * t, const size_t * indices);
const char * tensor_char_const_ptr(const tensor_char * t, const size_t * indices);

void tensor_char_set_zero(tensor_char * t);
void tensor_char_set_all(tensor_char * t, char x);

int tensor_char_fread(FILE * stream, tensor_char * t);
int tensor_char_fwrite(FILE * stream, const tensor_char * t);
int tensor_char_fscanf(FILE * stream, tensor_char * t);
int tensor_char_fprintf(FILE * stream, const tensor_char * t,
                        const char * format);

int tensor_char_memcpy(tensor_char * dest, const tensor_char * src);
int tensor_char_swap(tensor_char * t1, tensor_char * t2);

tensor_char *
tensor_char_swap_indices(const tensor_char * t, size_t i, size_t j);

char tensor_char_max(const tensor_char * t);
char tensor_char_min(const tensor_char * t);
void tensor_char_minmax(const tensor_char * t,
                        char * min_out, char * max_out);

void tensor_char_max_index(const tensor_char * t, size_t * indices);
void tensor_char_min_index(const tensor_char * t, size_t * indices);
void tensor_char_minmax_index(const tensor_char * t,
                              size_t * imin, size_t * imax);

int tensor_char_isnull(const tensor_char * t);

int tensor_char_add(tensor_char * a, const tensor_char * b);
int tensor_char_sub(tensor_char * a, const tensor_char * b);
int tensor_char_mul_elements(tensor_char * a, const tensor_char * b);
int tensor_char_div_elements(tensor_char * a, const tensor_char * b);
int tensor_char_scale(tensor_char * a, const double x);
int tensor_char_add_constant(tensor_char * a, const double x);
int tensor_char_add_diagonal(tensor_char * a, const double x);
tensor_char * tensor_char_product(const tensor_char * a,
                                  const tensor_char * b);
tensor_char * tensor_char_contract(const tensor_char * t_ij,
                                   size_t i, size_t j);


/* inline functions if you are using GCC */

#ifdef HAVE_INLINE
extern inline
size_t
tensor_char_position(const size_t * indices, const tensor_char * t)
{
  size_t shift, position;
  unsigned int i;

  shift = t->size/t->dimension;  /* == quick_pow(t->dimension, t->rank-1) */

  position = 0;
  for (i = 0; i < t->rank; i++)
    {
#if GSL_RANGE_CHECK
      if (indices[i] >= t->dimension)
        return t->size;
#endif

      position += shift * indices[i];
      shift /= t->dimension;
    }

  return position;
}


extern inline 
char
tensor_char_get(const tensor_char * t, const size_t * indices)
{
  size_t position;

  position = tensor_char_position(indices, t);
#if GSL_RANGE_CHECK
  if (position >= t->size)
    GSL_ERROR_VAL("index out of range", GSL_EINVAL, 0);
#endif

  return t->data[position];
}


extern inline
void
tensor_char_set(tensor_char * t, const size_t * indices, const char x)
{
  size_t position;
  
  position = tensor_char_position(indices, t);
#if GSL_RANGE_CHECK
  if (position >= t->size)
    GSL_ERROR_VOID("index out of range", GSL_EINVAL);
#endif

  t->data[position] = x;
}


extern inline 
char *
tensor_char_ptr(tensor_char * t, const size_t * indices)
{
  size_t position;

  position = tensor_char_position(indices, t);
#if GSL_RANGE_CHECK
  if (position >= t->size)
    GSL_ERROR_NULL("index out of range", GSL_EINVAL);
#endif

  return (char *) (t->data + position);
}


extern inline 
const char *
tensor_char_const_ptr(const tensor_char * t, const size_t * indices)
{
  size_t position;

  position = tensor_char_position(indices, t);
#if GSL_RANGE_CHECK
  if (position >= t->size)
    GSL_ERROR_NULL("index out of range", GSL_EINVAL);
#endif

  return (const char *) (t->data + position);
} 

#endif

__END_DECLS

#endif /* __TENSOR_char_H__ */
