/****************************************************************************
** Copyright (C) 2006  Xavier Cremaschi (omega.xavier@gmail.com)
** This file is part of teXswitcher.
** teXswitcher is a LaTeX document modifier.
**
** teXswitcher is free software; you can redistribute it and/or modify
** it under the terms of the GNU General Public License as published by
** the Free Software Foundation; either version 2 of the License, or
** (at your option) any later version.
**
** This program is distributed in the hope that it will be useful,
** but WITHOUT ANY WARRANTY; without even the implied warranty of
** MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
** GNU General Public License for more details.
**
** You should have received a copy of the GNU General Public License along
** with this program; if not, write to the Free Software Foundation, Inc.,
** 51 Franklin Street, Fifth Floor, Boston, MA 02110-1301 USA.
****************************************************************************/


#include "Gui.h"

#include "TreeModel.h"
#include "Node.h"
#include "Parser.h"


#include <QTreeView>
#include <QMenuBar>
#include <QAction>
#include <QFileDialog>
#include <QString>
#include <QMenu>
#include <QPushButton>
#include <QVBoxLayout>
#include <QLabel>
#include <QStatusBar>
#include <QKeyEvent>

#include <fstream>
#include <sstream>
#include <iostream>
using namespace std;

const string Gui::CTRL_PRESSED  = "CTRL pressed, buttons act on ITEM + SUBTREE ";
const string Gui::CTRL_RELEASED = "CTRL not pressed, buttons act on ITEM";
const string Gui::VERSION       = "teXswitcher (0.1beta)";




Gui::Gui(QApplication* app) : QMainWindow(NULL)
, root(NULL)
, treemodel(NULL)
, treeview(NULL) {

    /********
     * MENU *
     ********/
    QMenu* fileMenu = menuBar()->addMenu(tr("&File"));

    QAction* openAct = new QAction(tr("&Open"), this);
    openAct->setShortcut(tr("Ctrl+O"));
    openAct->setStatusTip(tr("Open a LaTeX file"));
    connect(openAct, SIGNAL(triggered()), this, SLOT(openFile()));

    this->saveAct = new QAction(tr("&Save As"), this);
    this->saveAct->setShortcut(tr("Ctrl+S"));
    this->saveAct->setStatusTip(tr("Save in a LaTeX file"));
    this->saveAct->setEnabled(false);
    connect(this->saveAct, SIGNAL(triggered()), this, SLOT(saveFile()));

    QAction* quitAct = new QAction(tr("&Quit"), this);
    quitAct->setShortcut(tr("Ctrl+Q"));
    quitAct->setStatusTip(tr("Exit the application"));
    connect(quitAct, SIGNAL(triggered()), app, SLOT(quit()));

    fileMenu->addAction(openAct);
    fileMenu->addAction(saveAct);
    fileMenu->addSeparator();
    fileMenu->addAction(quitAct);


    /********
     * BODY *
     ********/
    this->layout = new QVBoxLayout(this);
    this->treeview = new QTreeView(this);
    this->treeview->setDragEnabled(true);
    this->treeview->setAcceptDrops(true);
    this->treeview->setDropIndicatorShown(true);
    this->treeview->setSelectionBehavior(QAbstractItemView::SelectRows);

    this->layout->addWidget(this->treeview);


    /***********
     * BUTTONS *
     ***********/
    QPushButton* keyword_plus = new QPushButton(tr("Keyword++"), this);
    connect(keyword_plus, SIGNAL(clicked()), this, SLOT(increase_selected_keyword()));
    this->layout->addWidget(keyword_plus);

    QPushButton* keyword_minus = new QPushButton(tr("Keyword--"), this);
    connect(keyword_minus, SIGNAL(clicked()), this, SLOT(decrease_selected_keyword()));
    this->layout->addWidget(keyword_minus);


    /**************
     * STATUS BAR *
     *************/
    this->status_label = new QLabel();
    this->status_label->setText(CTRL_RELEASED.c_str());
    this->statusBar()->addWidget(this->status_label, 1);


    /*********
     * FINAL *
     *********/
    QWidget* window = new QWidget();
    window->setLayout(this->layout);
    this->setWindowTitle(VERSION.c_str());
    this->setFocusPolicy(Qt::StrongFocus); // to listen keyboard events
    this->setCentralWidget(window);
    this->show();

}



Gui::~Gui() {
    if (this->treeview != NULL) {
        delete this->treeview; // TODO : check if this calls the destructor of the treeview's model
        this->treeview = NULL;
    }
}



void Gui::openFile(void) {

    QString fileName = QFileDialog::getOpenFileName(
                    this,
                    "Choose a file to open",
                    "",
                    "Well-formed LaTeX Documents (*.tex)" );


    if (fileName != QString::null) {

        // ifstream ctor to open a file in read mode
        ifstream file( fileName.toStdString().c_str(), ios::in );

        if ( file ) { // failed if not open

            // enabling File/Save menu
            this->saveAct->setEnabled(true);

            // reading
            stringstream buffer;
            buffer << file.rdbuf();
            file.close();

            // parsing
            this->root = Node::from_string(buffer.str());

            // for status bar
            string docclass = Parser::extract_document_class(buffer.str());
            string message = "Document class is " + docclass + ", chapter ";
            if (docclass == "article") {
                message += "disabled.";
            }
            else {
            message += "enabled.";
            }
            QLabel* label = new QLabel();
            label->setText(message.c_str());
            this->statusBar()->addWidget(label, 1);


            // replacing old with new
            TreeModel* old_tree = NULL;
            if (this->treemodel != NULL) {
                old_tree = this->treemodel;
            }
            this->treemodel = new TreeModel(this->root);
            delete old_tree;


            this->treeview->setModel(this->treemodel);


        }
    }
}



void Gui::saveFile(void) {
    QString fileName = QFileDialog::getSaveFileName(
                    this,
                    "Save file",
                    "",
                    "Well-formed LaTeX Documents (*.tex)" );


    if (fileName != QString::null) {
        ofstream file( fileName.toStdString().c_str(), ios::out);
        if ( file ) {
            if (this->root != NULL) {
                file << this->root->to_string();
            }
            file.close();
        }
    }
}


void Gui::increase_selected_keyword(void) {
    if (this->treeview != NULL) {
        QItemSelectionModel* model = this->treeview->selectionModel();
        if (model != NULL) {
            QModelIndex index = model->currentIndex();
            if (index.isValid() && this->treemodel != NULL) {
                this->treemodel->change(index, +1);
            }
        }
    }
}

void Gui::decrease_selected_keyword(void) {
    if (this->treeview != NULL) {
        QItemSelectionModel* model = this->treeview->selectionModel();
        if (model != NULL) {
            QModelIndex index = model->currentIndex();
            if (index.isValid() && this->treemodel != NULL) {
                this->treemodel->change(index, -1);
            }
        }
    }
}



void Gui::keyPressEvent(QKeyEvent* e) {
    if (this->treemodel != NULL && e->key() == Qt::Key_Control) {
        e->accept();
        this->status_label->setText(CTRL_PRESSED.c_str());
        this->treemodel->act_on_subtree(true);
    }
    else {
        e->ignore(); // propagated to parent until someone accepts it
    }
}

void Gui::keyReleaseEvent(QKeyEvent* e) {
    if (this->treemodel != NULL && e->key() == Qt::Key_Control) {
        e->accept();
        this->status_label->setText(CTRL_RELEASED.c_str());
        this->treemodel->act_on_subtree(false);
    }
    else {
        e->ignore(); // propagated to parent until someone accepts it
    }
}
