/****************************************************************************
** Copyright (C) 2006  Xavier Cremaschi (omega.xavier@gmail.com)
** This file is part of teXswitcher.
** teXswitcher is a LaTeX document modifier.
**
** teXswitcher is free software; you can redistribute it and/or modify
** it under the terms of the GNU General Public License as published by
** the Free Software Foundation; either version 2 of the License, or
** (at your option) any later version.
**
** This program is distributed in the hope that it will be useful,
** but WITHOUT ANY WARRANTY; without even the implied warranty of
** MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
** GNU General Public License for more details.
**
** You should have received a copy of the GNU General Public License along
** with this program; if not, write to the Free Software Foundation, Inc.,
** 51 Franklin Street, Fifth Floor, Boston, MA 02110-1301 USA.
****************************************************************************/


#include <iostream>
#include <string>
using namespace std;

#include "Parser.h"
#include "Node.h"
#include "Keyword.h"



/**
 * This method build a tree of Node* from a string and return the root of the
 * tree
 * PRECONDITION : the first block of the string has the maximal keyword value
 *
 * description of the parsing algorithm :
 * - go to the beginning of 'text' ;
 * - determine if 'text' is a full LaTeX document 
 *   (ie beginning with \documentclass)
 *   or just a piece of one (beginning with a keyword);
 * - search the position of the first keyword : 'end_of_first_block' ;
 * - create the first node with the string from the beginning of text to
 *   'end_of_first_block' ;
 * - execute main loop (see below);
 * - if it is a full LaTeX document, add \end{document} after the last block.
 *
 * description of main loop :
 * - we are at position 'begin' in text
 * - we search the next keyword position, 'end'
 * - if found, create a node with the string between 'begin' and 'end'
 * - if not, this is the last node before the end of 'text', create a node with
 *   the lasting text
 */
Node* Parser::parse(string& text) {
    Node* result = NULL;
    Keyword::use_chapter(true); // we suppose we are going to work will all keywords

    // is this a full document or just a piece of ?
    bool is_full_latex_document = false;
    string docclass = Parser::extract_document_class(text);
    if (docclass != "") {
        is_full_latex_document = true;
        if (docclass == "article") {
            Keyword::use_chapter(false);
        }
    }

    // the 1st char != of one of these 3 char => beginning of file
    string::size_type begin = text.find_first_not_of(" \t\n"); 

    // we search the first keyword in order to extract the first block
    string::size_type end_of_first_block = Parser::find_next_keyword_position(text, begin+1);

    // here between begin and end_of_header we have the header
    string first_block = text.substr(begin, end_of_first_block - begin);

    // the root node
    result = new Node(first_block);

    // initialization for the while
    Node* current   = result;
    begin           = end_of_first_block;

    // an iteration searchs where is the beginning of the next block, and
    // creates a node using the string between begin and the beginning of this
    // next block
    while (begin != string::npos) {
        string::size_type end = Parser::find_next_keyword_position(text, begin+1);

        // compute the length of the block
        string::size_type block_length;
        if (end != string::npos) {
            block_length = end - begin;
        }
        else { // we find the end of the document : this is the last block.
            if (!is_full_latex_document) {
                block_length = text.size() - begin;
            }
            else {
                block_length = (text.size() - Keyword::LATEX_DOCUMENT_LAST_KEYWORD().size() - 1) - begin;
            }
        }

        string block    = text.substr(begin, block_length);
        Node* n         = new Node(block);
        int priority    = n->get_keyword_value();
        Node* good_one  = Parser::find_good_node(current, priority);
        good_one->add_son(n); // update of the current tree

        current = n;
        begin = text.find_first_not_of(" \t\n", end);
    }

    if (is_full_latex_document) {
        result->set_footer(Keyword::LATEX_DOCUMENT_LAST_KEYWORD());
    }

    return result;
}


/** to find the position of the first next keyword */
string::size_type Parser::find_next_keyword_position(string text, string::size_type begin) {
    string::size_type result = text.find(Keyword::KEYWORD(0), begin);

    for (int i = 1 ; i < Keyword::NB_OF_KEYWORDS() ; i++) {
        string::size_type pos = text.find(Keyword::KEYWORD(i), begin);
        result = (pos < result) ? pos : result;
    }
    return result;
}







/**
 * when we add a teX section to our tree, we have to find the good place, depending
 * on priorities
 * example : 
 * a subsection after a section => subsection IS A SON OF section, easy
 * a section after a subsubsection => we have to go up to find the good node, recursivity
 *
 * PRECONDITION : only the root of the tree have a NULL parent
 * PRECONDITION : the root has a keyword value superior to any keyword value of
 *                any node we try to add
 */
Node* Parser::find_good_node(Node* current_node, int priority_to_place) {

    Node* parent = current_node->get_parent();

    // current_node is the root, the only solution is to use it
    if (parent == NULL) { 
        return current_node;
    }

    int current_priority = current_node->get_keyword_value();

    // a son : current_node will be its parent
    if (priority_to_place < current_priority) { 
        return current_node;
    }

    // a brother : current_node's parent will be its parent
    if (priority_to_place == current_priority && parent != NULL) {
        return parent;
    }

    // this is a new part with a higher priority, we have to seek up for the
    // good parent
    if (priority_to_place > current_priority && parent != NULL) {
        return Parser::find_good_node(parent, priority_to_place);
    }

    return NULL; // should never happened
}



string Parser::extract_document_class(string text) {
    string result = "";

    string::size_type position = text.find(Keyword::LATEX_DOCUMENT_FIRST_KEYWORD());
    if (position != string::npos) {
        string::size_type begin = text.find("{", position+1) + 1;
        string::size_type end   = text.find("}", begin);
        result                  = text.substr(begin, end - begin);
    }

    return result;
}
