/****************************************************************************
** Copyright (C) 2006  Xavier Cremaschi (omega.xavier@gmail.com)
** This file is part of teXswitcher.
** teXswitcher is a LaTeX document modifier.
**
** teXswitcher is free software; you can redistribute it and/or modify
** it under the terms of the GNU General Public License as published by
** the Free Software Foundation; either version 2 of the License, or
** (at your option) any later version.
**
** This program is distributed in the hope that it will be useful,
** but WITHOUT ANY WARRANTY; without even the implied warranty of
** MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
** GNU General Public License for more details.
**
** You should have received a copy of the GNU General Public License along
** with this program; if not, write to the Free Software Foundation, Inc.,
** 51 Franklin Street, Fifth Floor, Boston, MA 02110-1301 USA.
****************************************************************************/


#include "TreeModel.h"
#include "Node.h"
#include "Keyword.h"


const QString TreeModel::MIME_TYPE = "text/plain";
Node* TreeModel::moving_node = NULL;


TreeModel::TreeModel(Node* tree_, QObject *parent) 
    : QAbstractItemModel(parent), tree(tree_), is_recursive(false) {
}


TreeModel::~TreeModel() {
    if (this->tree != NULL) {
        delete this->tree;
        this->tree = NULL;
    }
}





/*************************************************
 * MANDATORY WHEN SUBCLASSING QAbstractItemModel *
 *************************************************/


QModelIndex TreeModel::index(int row
        , int column
        , const QModelIndex& parent) const {

    Node* parentItem = NULL;

    if (!parent.isValid()) {
        parentItem = this->tree;
    }
    else {
        parentItem = static_cast<Node*>(parent.internalPointer());
    }

    Node* childItem = parentItem->get_son(row);
    if (childItem != NULL) {
        return createIndex(row, column, childItem);
    }
    else {
        return QModelIndex();
    }
}


QModelIndex TreeModel::parent(const QModelIndex& index) const {
    if (!index.isValid()) {
        return QModelIndex();
    }

    Node* childItem  = static_cast<Node*>(index.internalPointer());
    Node* parentItem = childItem->get_parent();

    if (parentItem == this->tree) {
        return QModelIndex();
    }
    else {
        return createIndex(parentItem->get_number(), 0, parentItem);
    }
}


int TreeModel::columnCount(const QModelIndex&) const {
    return 2;
}


int TreeModel::rowCount(const QModelIndex& index) const {
    Node* n = NULL;

    if (!index.isValid()) {
        n = this->tree;
    }
    else {
        n = static_cast<Node*>(index.internalPointer());
    }

    return n->get_nb_of_nodes();
}


QVariant TreeModel::data(const QModelIndex& index, int role) const {
    if (!index.isValid() || role != Qt::DisplayRole) {
        return QVariant();
    }

    Node* item = static_cast<Node*>(index.internalPointer());
    if (index.column() == 0) {
        return QVariant(item->get_name().c_str());
    }
    else if (index.column() == 1) {
        return QVariant(item->get_keyword().c_str());
    }

    else {
        return QVariant();
    }
}







/**********************************
 * TO ENABLE DRAG & DROP FEATURES *
 **********************************/


/**
 * A delegate checks whether an item is editable before creating an editor. The
 * model must let the delegate know that its items are editable. We do this by
 * returning the correct flags for each item in the model; in this case, we
 * enable all items and make them both selectable and editable
 */
Qt::ItemFlags TreeModel::flags(const QModelIndex& index) const {
    if (!index.isValid()) {
        return Qt::ItemIsDropEnabled | Qt::ItemIsEnabled;
    }
    else {
        return Qt::ItemIsDragEnabled | Qt::ItemIsDropEnabled | Qt::ItemIsSelectable | Qt::ItemIsEnabled;
    }
}

Qt::DropActions TreeModel::supportedDropActions(void) const {
    return Qt::MoveAction | Qt::CopyAction; // Qt::CopyAction seems mandatory
}


/**
 * In copy/paste or drag&drop, we are expected to not communicate by QModelIndex (internal
 * stuff) but by QMimeData (more universal stuff)
 * TRICK : In fact, because i only want to drag&drop from this TreeModel to this TreeModel, i
 * store the internal pointer in a private static variable (static because
 * mimeData has to be const and so it can't modify this) 
 * XXX : because of the static aspect, it could be dangerous to instanciate more
 *       than one TreeModel
 * NB : the QMimeData returned MUST contain data.
 */
QMimeData* TreeModel::mimeData(const QModelIndexList& indexes) const {
    TreeModel::moving_node = NULL;

    // 2 elements, (row x, col 0) and (row x, col1)
    // row is the position in relation to the parent, knowing that row = 0 is not
    // enough to determine what was selected
    if (indexes.size() == 2) {
        TreeModel::moving_node = static_cast<Node*>(indexes[0].internalPointer());
    }
    QMimeData* fake_result = new QMimeData();
    fake_result->setText("42");
    return fake_result;
} 

/** a the drop moment (the reception), we execute this method */
bool TreeModel::dropMimeData(const QMimeData*, Qt::DropAction, int, int, const QModelIndex& parent_index) {
    if(TreeModel::moving_node != NULL && parent_index.isValid()) {
        Node* parent = static_cast<Node*>(parent_index.internalPointer());

        if (parent->get_keyword_value() > TreeModel::moving_node->get_keyword_value()) {
            // cutting old branch
            Node* old_parent = TreeModel::moving_node->get_parent();
            if (old_parent != NULL) {
                old_parent->remove_son(TreeModel::moving_node);
            }
            // adding to new branch
            parent->add_son(TreeModel::moving_node);

            emit layoutChanged();
            return true;
        }
        else {
            return false;
        }
    }
    else {
        return false;
    }
}

/** 
 * Returns a list of MIME types that can be used to describe a list of model indexes. 
 * Useful to propose copy/paste or drag & drop from here to the outside 
 */
QStringList TreeModel::mimeTypes(void) const {
    return (QStringList() << MIME_TYPE);
}








/** for header's display */
QVariant TreeModel::headerData(int section
        , Qt::Orientation orientation
        , int role) const {

    if (orientation == Qt::Horizontal 
            && role == Qt::DisplayRole) {

        if (section == 0) {
            return tr("Title");
        }
        else if (section == 1) {
            return tr("Type");
        }
        else {
            return QVariant();
        }
    }
    else {
        return QVariant();
    }
}


/** increase or decrease the rank of an item (ex: subsection->section) */
void TreeModel::change(QModelIndex& index, int count) {
    if (index.isValid()) {

        Node* item = static_cast<Node*>(index.internalPointer());

        string  old_keyword = item->get_keyword();
        int     old_value   = item->get_keyword_value();

        int new_value = old_value + count;
        if ( Keyword::MIN_VALUE() <= new_value
                && new_value < Keyword::MAX_VALUE()) {

            // a promotion
            if (new_value > old_value) {
                // we need to increase the item at this node.
                // we check if the node's father will be able to accept the
                // promotion of one of its sons
                Node* father = item->get_parent();
                if (father != NULL) { // father == NULL <=> father is the root, item can't be promoted
                    int father_value = father->get_keyword_value();
                    if (new_value < father_value) {
                        item->change_keyword(+1, this->is_recursive);
                        emit layoutChanged();
                    }
                }

            }

            // a reduction in rank
            else if (new_value < old_value) {
                // we need to reduce the item at this node.
                // we check if the node's sons will be able to accept the
                // reduction of their father
                bool is_reductible = false;
                if (this->is_recursive) { // will we be able to decrease the lowest son ?
                    int sons_minvalue = item->get_sons_lowest_keyword_value();
                    if (sons_minvalue > Keyword::MIN_VALUE()) {
                        is_reductible = true;
                    }
                }
                else { // after reduction, do we stay bigger than the biggest son ?
                    if (item->has_son()) {
                        int son_maxvalue = item->get_sons_biggest_keyword_value();
                        if (new_value > son_maxvalue) {
                            is_reductible = true;
                        }
                    }
                    else {
                        if (new_value >= Keyword::MIN_VALUE()) {
                            is_reductible = true;
                        }
                    }
                }

                if (is_reductible) {
                    item->change_keyword(-1, this->is_recursive);
                    emit layoutChanged();
                }
            }
        }
    }
}


void TreeModel::act_on_subtree(bool b) {
    this->is_recursive = b;
}
