##	Things Copyright(C) 2009 Donn.C.Ingle
##
##	Contact: donn.ingle@gmail.com - I hope this email lasts.
##
##  This file is part of Things.
##
##  Things is free software: you can redistribute it and/or modify
##  it under the terms of the GNU General Public License as published by
##  the Free Software Foundation, either version 3 of the License, or
##  (at your option) any later version.
##
##  Things is distributed in the hope that it will be useful,
##  but WITHOUT ANY WARRANTY; without even the implied warranty of
##  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
##  GNU General Public License for more details.
##
##  You should have received a copy of the GNU General Public License
##  along with Things.  If not, see <http://www.gnu.org/licenses/>.

"""
General drawing tools
=====================

In this module you will find a collection of tools for drawing.
"""

import math
from math import pi
from random import random

import cairo
import pangocairo, pango
import pyparsing as PP

pi2 = 2*pi
hp = pi/2
pi90 = pi + hp
pi3 = 3*pi

##   pi90
## pi    0
##    hp

class TextBlock(object):
	"""
	This will get you a quick block of text. Use setup() to get it going and draw() to display it.
	"""
	def __init__(self):
		self.setup()

	def setup(self, richtext="<span>Say what?</span>", x=0, y=0, align=pango.ALIGN_LEFT, width=-1):
		"""
		@param richtext: The text you want to show, marked-up with pango's span tag and params.
		 Some examples:
		  1. font_desc="Sans 12" -- Choose generic sans-serif at 12 point size.
		  2. foreground="#RRGGBB" -- Colour of span element. RRGGBB are hex couplets. #FF000 would be red.
		@param x: The x coord.
		@param y: The y coord.
		@param align: Pass pango.ALIGN_LEFT, pango.ALIGN_CENTER or pango.ALIGN_RIGHT
		@param width: A width in pixels that the text-block should wrap into.
		"""
		self._x = x
		self._y = y
		self._markup = richtext#'''<span font_desc="%s" %s>%s</span>''' % (font, spanargs, text)
		self._align = align

		self._width = -1
		if width != -1:
			self._width = width * pango.SCALE

	def draw(self, ctx):
		pctx = pangocairo.CairoContext (ctx)
		layout = pctx.create_layout ()
		layout.set_markup (self._markup)
		layout.set_alignment(self._align)
		if self._width > 0:
			layout.set_width(self._width)
		ctx.move_to (self._x, self._y)
		pctx.show_layout (layout)

class RoundRect(object):
	"""
	Use this to draw rectangles with rounded corners.
	Use the B{setup()} method and pass in the width, height and radius.
	After that, call B{draw(ctx)} with an optional x,y.

	The rect is drawn from x,y in its center.
	"""
	def __init__(self):
		self.dlRound=DrawLang()
		self._x=0
		self._y=0
	def setup(self, width=10, height=10, radius=20):
		width = width-(2*radius)
		height = height-(2*radius)
		self.dlRound.setup("r%(R)sf%(W)s r%(R)sf%(H)s r%(R)sf%(W)s r%(R)sf%(H)s" % {"R":radius,"W":width,"H":height} )
		self._x = (width + (2 * radius))/2
		self._y = (height)/2
	def draw(self, ctx, x=0, y=0):
		x,y = x-self._x,y-self._y
		self.dlRound.draw(ctx,x,y)

class DrawLang(object):
	"""
	Use this to generate stepped-paths with curved corners. It uses a simple 'language' comprised of:
	
	 1. B{l#}: Turn left, with an optional number which is the radius of the corner. 
	 A sharp-corner is zero (or leave it blank).
	 2. B{r#}: Turn right; same as left, only... it goes right.
	 3. B{f#}: Go forward, with a number of units to proceed. The units are not optional.
	
	The initial direction is North. Unless you begin with 'r' or 'l', you will be going forward
	directly upwards.

	To draw a square with round corners: B{r10 f50 r10 f50 r10 f50 r10 f50}
	(Of course, you should simply use the RoundRect() class for shapes like that.)

	B{r50 f10 r50 f60 r f110 r f60} : Would draw a square with two big curved corners on top (both of radius 50.) 
	Note from this example that the radius is part of the total width; so 50 + 10 + 50 = 110

	Use the B{setup()} method to pass your string in. This will create a list of python-cairo
	line and arc commands that get drawn when you call B{draw(ctx)}.
	"""
	## Pyparse draw strings : Class level vars.
	_fwd =  PP.Group( PP.Literal("f") + PP.Word(PP.nums) )
	_right = PP.Group( PP.Literal("r") + PP.Optional(PP.Word(PP.nums), default=0) )
	_left = PP.Group( PP.Literal("l") + PP.Optional(PP.Word(PP.nums), default=0) )
	_grp = _left | _right | _fwd 
	_seq = PP.OneOrMore( _grp )
	
	def __init__(self):
		pass
	def setup(self,str):
		# chop-up the string with pyparsing.
		tokens = DrawLang._seq.parseString(str)

		# global default direction 
		D="N"

		def doLeftRight(F):
			turn=F[0]
			radius=F[1]
			s1='x,y=ctx.get_current_point();r=%s;' % radius
			if D=="N":
				if turn=="r":s='xr=x+r;yr=y\nctx.arc(xr,yr,r,pi,pi90)\n'
				if turn=="l":s='xr=x-r;yr=y\nctx.arc_negative(xr,yr,r,0,pi90)\n'
			if D=="E":
				if turn=="r":s='xr=x;yr=y+r\nctx.arc(xr,yr,r,pi90,0)\n'
				if turn=="l":s='xr=x;yr=y-r\nctx.arc_negative(xr,yr,r,hp,0)\n'
			if D=="S":
				if turn=="r":s='xr=x-r;yr=y\nctx.arc(xr,yr,r,0,hp)\n'
				if turn=="l":s='xr=x+r;yr=y\nctx.arc_negative(xr,yr,r,pi,hp)\n'
			if D=="W":
				if turn=="r":s='xr=x;yr=y-r\nctx.arc(xr,yr,r,hp,pi)\n'
				if turn=="l":s='xr=x;yr=y+r\nctx.arc_negative(xr,yr,r,pi90,pi)\n'
			#print turn," from ",D
			return s1 + s

		def doFwd(F):
			ufwd=F[1]
			x='0'
			y='0'
			if D=="N": y='-%s' % ufwd
			if D=="E": x='%s' % ufwd
			if D=="S": y='%s' % ufwd
			if D=="W": x='-%s' % ufwd
			s = "ctx.rel_line_to(%s,%s)\n" % (x,y)
			#print "fwd"
			return s

		def NewCompassDirection(F):
			turn=F[0]
			ND=None
			if D=="N":
				if turn=="r": ND="E"
				if turn=="l": ND="W"
			if D=="E":
				if turn=="r": ND="S"
				if turn=="l": ND="N"
			if D=="S":
				if turn=="r": ND="W"
				if turn=="l": ND="E"
			if D=="W":
				if turn=="r": ND="N"
				if turn=="l": ND="S"
			return ND

		##[['r', '5'], ['f', '10'], ['r', 0], ['f', '20'], ['l', 0], ['f', '20'], ['f', '11'], ['r', '6']]
		self.paths="" # We will build the pycairo commands to draw this string.
		for F in tokens:
			if F[0] in ['r','l']:
				commands = doLeftRight(F)
				D=NewCompassDirection(F) # Switch to new direction *after* we draw the curve.
			if F[0]=="f":
				fwd = F[1]
				commands = doFwd(F)
			self.paths += commands

	def draw(self, ctx, x=0, y=0):
		ctx.move_to(x,y)
		exec(self.paths) #Just play them back. Fast as possible.

class fuzzydot(object):
	"""
	Ye olde generic fuzzy dot. 
	
	Use:
	====
	Make an instance: B{fd = fuzzydot()} and then use B{fd.draw(ctx)}
	"""
	rad = cairo.RadialGradient(0,0,2,0,0,20)
	rad.add_color_stop_rgba(0, 0,0,0, 1)
	rad.add_color_stop_rgba(1, 0,0,0, 0)
	def draw(self,ctx):
		ctx.set_source(fuzzydot.rad)
		ctx.arc ( 0, 0, 20, 0, pi * 2)
		ctx.fill()	

def zerodot(ctx):
	"""
	A big red circle for quick debugging. This is a function, so just call it with a context.
	"""
	ctx.set_source_rgb(1.0,0,0)
	ctx.arc ( 0, 0, 20, 0, pi * 2)
	ctx.fill()


def zerorect(ctx,w,h,nostroke=False):
	"""
	A function to draw a red rectangle at zero, of given width/height. nostroke False will stroke it.
	For debugging.
	"""
	ctx.set_source_rgb(1.0, 0.0, 0)
	ctx.rectangle(0-(w/2), 0-(w/2), w, h)
	if nostroke: return
	ctx.stroke()



## Mathification funcs

def hexfloat(s):
	"""
	Given an HTML-style hex colour, return the cairo r,g,b colour numbers for it.
	If given an alpha value, this returns r,g,b,a

	You can use this in places like B{ctx.set_source_rgb( *hexfloat("#FF00BB") )}
	-- the	* expands the return value into a list for the function it's passed-to.
	"""
	if s[0]=="#": s=s[1:]
	r,g,b =  int(s[:2],16),int(s[2:4],16),int(s[4:6],16)
 	r,g,b =  r / 255.0, g / 255.0, b / 255.0
	ret = (r,g,b)
	if len(s) > 6:
		a = int( s[6:],16 )
		a = a/255.0
		ret = (r,g,b,a)
	return ret

def circrandom(radius):
	"""
	Returns a random point (tuple) on a circle of given radius.
	"""
	i = math.radians(random()*360)
	x=math.cos(i)*radius
	y=math.sin(i)*radius
	return x,y

