##	Things Copyright(C) 2009 Donn.C.Ingle
##
##	Contact: donn.ingle@gmail.com - I hope this email lasts.
##
##  This file is part of Things.
##
##  Things is free software: you can redistribute it and/or modify
##  it under the terms of the GNU General Public License as published by
##  the Free Software Foundation, either version 3 of the License, or
##  (at your option) any later version.
##
##  Things is distributed in the hope that it will be useful,
##  but WITHOUT ANY WARRANTY; without even the implied warranty of
##  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
##  GNU General Public License for more details.
##
##  You should have received a copy of the GNU General Public License
##  along with Things.  If not, see <http://www.gnu.org/licenses/>.

"""
ThingObjects are essential Things; like ClipThing and FollowThing. For higher-level tools see Thinglets.
"""

import pyparsing as PP
import copy as objectduplicate
from timeline import *
from bugs import Bugs as _Bugs

import cairo

_MOUSENOTWITHIN = "MOUSENOTWITHIN"
_MOUSEWITHIN = "MOUSEWITHIN"


class _Frame( object ):
	"""Private: Holds the vital data for every frame of a thing."""
	def __init__(self):
		self.blank = False
		self.tag = ""
		self.frame = 1
		self.label = ""
		self.stop = False
		self.temporarilyIgnoreThisPropertyStop = False
		self.jump = None # A frame or label
		self.func = None 
		
		self.props = Props()
	def __repr__( self ):
		s="********\n_Frame:\n"
		for attr, value in sorted(self.__dict__.iteritems()):
			s +=  "%s:%s, " % ( attr, value )
		s += "\n" + repr(self.props)
		s += "\n***********\n"
		return s

class _Function(object):
	"""Private: Hold a ref to some other function as well as it's args. Used for callbacks to the user's code."""
	def __init__( self, funcref, args):
		self.funcref = funcref
		self.args = args	

class Props(object):
	"""
	A Property
	==========
	Make one (or more) of these to pass into thing.keys().
	Describes the visual data of a thing, like x and y.

	Use
	===
	prop = Props(x=100, y=20, a=0.2)

	@param x,y: Positional coordinate relative to the center of the screen.
	@param a:  Alpha; a float between 0 (transparent) and 1 (opaque).
	@param rot: Rotation; in radians -- a full turn is pi*2
	@param rx,ry: A point about which to rotate.
	@param sx, sy: Scale; setting them to different values will distort and stretch 
	 your Thing. You can use the param sz to set both at once.
	@param r,g,b,tint: RGB colour values; floats 0 to 1. Tint is amount of colour to use
	 over Thing. Used to fade from one colour to another, with varying amounts. This will
	 only take effect when tint is not 0.
	 B{WARNING:} Tinting it *slow*. It's better to have duplicate paths of differing colours,
	 or to draw your own paths and control the stroke/fill colour manually.
	"""
	def __init__( self,x=0,y=0,a=1.0,rot=0.0,sx=1,sy=1,sz=None,rx=0, ry=0.0, r=0.0, g=0.0, b=0.0, tint=0.0 ):
		self.x , self.y = x, y
		self.r, self.g, self.b = r, g, b
		self.tint = tint
		self.a = a
		self.rot = rot
		if sz:
			self.sx,  self.sy = sz,sz
		else:
			self.sx, self.sy = sx,sy
		self.rx, self.ry = rx, ry # point of rotation
	def tintlist(self):
		return [self.r,self.g,self.b,self.tint]
	def __repr__ (self):
		s = "self.props::\n"
		for attr, value in sorted(self.__dict__.iteritems()):
			s +=  "%s:%s, " % ( attr, value )
		return s

class Thing(Timeline):
	"""
	A Timeline
	==========

	A basic Thing -- it is a Timeline. This represents a container of other Things. Use add() to add 
	other Things to it. They will appear as per the keys() you supply to this Thing.
	
	If the user supplies a draw() method, then it will be called and that's where you put the cairo commands to do your 
	actual drawing.

	Drawing stuff
	=============
	If you define a draw() method, it will get called every frame. The signature is: def draw(self,context,frame_number)

	Use
	===

	Typical use::

	  class Alien(Thing):
	    def __init__(self,x, speed=1): # up to you
	    ## Now initialize the Thing: You *must* do this.
	    Thing.__init__( self, id="alien")

	ends.
	
	@ivar lifespan: If you need to start a sequence right after some other thing is done, then get
	 that thing's lifespan and start your thing there. See Thing.add() for the parentFrame argument.
	@ivar currentFrame: This provides a _Frame instance. Within that, the most useful property is 'props'. 
	 For example: B{boat.currentFrame.props.x} is the x coord of the boat right now.
	@ivar globalProps: This is a Props() object assigned to the entire Thing. Can be set from the
	 add() method of a Thing (see Timeline class). You can also simply use it in a Thing's init.
	@ivar frame: Just be aware there is a variable of this name and B{don't} use it.
	"""

	## Class Variables:
	## This part is just ... magical :) It sets up Pyparsing and I can't be more amazed.
	## pyparsing help courtesy of Paul McGuire <ptmcg@austin.rr.com> (http://groups.google.com)
	## Describe the permutations of the keyframing syntax.
	_frame = PP.Literal("#")
	_tween = PP.Word("-") # that is, it is a "word" composed of 1 or more -'s
	_copy = PP.Literal("=")
	_blank = PP.Literal(".")
	_animation = PP.Optional(PP.OneOrMore(_blank)) + PP.OneOrMore(_frame + PP.Optional((_tween + PP.FollowedBy(_frame)) | PP.OneOrMore(_copy | _blank)))

	def __init__( self, id="thing_id_not_set"):
		"""
		@param id: A string to identify this Thing. It's mainly for debug purposes, but handy within your own code too.
		"""
		Timeline.__init__(self)
		self.frame = 1
		self.id = id
		self.frameobjectList =[] #A list of my frameProps objects.
		self.lifespan = 0
		self.loops = True
		self._isClip = False #If true, then all my children are masked by me.
		self.keyframelayoutstring = None #nothing yet
		self.currentFrame = _Frame() # give it an empty Frame
		self.stopped = False

		self.dirstore = dir(self)

		self.globalProps = Props()
		self.START_ALPHA = False
		self.START_TINT = False

	def keys(self, keystring, *stuff):
		"""This sets the keys that control the lifespan of Things 1)added B{to} this thing and 2)Whatever you draw().

		Key syntax
		==========

		 1. #: Keyframe. Things will appear on a keyframe.
		 2. =: A 'same' frame. It must follow a keyframe. This indicates that the previous frame
		 simply continues and does B{not} need another Props() object to describe it.
		 3. -: A 'tween'. This must follow a # (keyframe) and end on same. Each in-between frame
		 is given Props that are interpolated. Use tweens to move and fade Things.
		 4. .: A blank frame. Everthing goes away on a blank frame.

		Use
		===

		Typical use::
		
		  #Here we have only one keyframe, hence we need one Props() object.
		  p1 = Props ( x=100, y=20, a=0.5 )
		  self.keys ( "#=.", p1 )
		  
		  # If you have more than one keyframe, have a prop for each one:
		  self.keys ( "#---#", Props(), Props(rot=pi * 2) )

		  #To tween movement do something like this:
		  self.keys( "#----------------------#", Props(x=0), Props(x=100) ) #move from 0 to 100 gradually.


		ends.

		@param keystring: A string that represents the keyframes, blanks and tweens of this Thing.
		 This string dictates where/when B{children} of this Thing will appear.
		@param stuff: Any number of Props() objects. Have as many as you have keyframes.

		"""
		if len(keystring) == 0: 
			raise _Bugs("EMPTY_KEY_STRING") 
		if len(stuff) < 1:
			raise _Bugs("KEYS_NEED_STUFF")
		#Must vet the string for bad chars... later.
		self.keyframelayoutstring = keystring
		self.lifespan = len( keystring )
		self._parseKeys( stuff )
		
		## Now make sure that we have a valid currentFrame
		self.currentFrame = self.frameobjectList[self.frame-1]
		
	def labels(self, labelstring, *labels):
		"""This is used to assign labels to frames.
		Wherever you use ^ it will place a label you name in the args.
		(The idea is that it I{points} to the frame in the keys() command above it.)
		
		Use
		===
		Typical use::

		  self.keys  ( "#---------#.", ...)
		  self.labels( "...........^", "die" )

		  Later, you can say:
		  self.goPlay('die')
		
		ends.

		@param labelstring: A string of dots (.) and carets (^). The carets point where 
		 labels will be assigned.
		@param labels: Any number of strings providing the label names for the ^ points
		 in the label string.
		"""
		if not self.frameobjectList:
			raise _Bugs("SET_KEYFRAMES_FIRST")
		
		if len(labelstring) == 0: 
			raise _Bugs("EMPTY_LABEL_STRING") 
		if len(labels) < 1:
			raise _Bugs("MISSING_LABELS")			
		if labelstring.count("^") != len(labels):
			raise _Bugs("MISSING_LABELS")
			
		## Process the string
		## It looks like this:
		## "^__^__^" : ^ = label here, _=nothing
		fr = 0
		i = 0
		for c in labelstring:
			if c == "^":
				lab = labels[i]
				i += 1
				self.frameobjectList[fr].label = lab # Should check for duped labels...
			fr += 1

	def stops(self, stopstring):
		"""This is used to set stops in particular frames. Wherever you use ^ it will place a stop.

		Use
		===
		Typical use::

		  self.keys ( "#--------#...",...)
		  self.stops( "...^.....^" )

		ends.

		@param stopstring: A dot (.) is a blank. A caret (^) points at the frame to stop at.
		"""		
		if not self.frameobjectList:
			raise _Bugs("SET_KEYFRAMES_FIRST")
		
		if len(stopstring) == 0: 
			raise _Bugs("EMPTY_STOP_STRING") 
		if stopstring.count("^") == 0:
			raise _Bugs("NO_STOPS_FOUND")
			
		## Process the string
		## It looks like this:
		## "^__^__^" : ^ = stop here, _=nothing
		fr = 0
		for c in stopstring:
			if c == "^":
				try:
					self.frameobjectList[fr].stop = True
				except:
					# I noticed a keyError when I had stops and my keys were 
					# badly formed. So, this error should help in future.
					raise _Bugs("GENERAL_BAD_KEYS")
			fr += 1
			
	def jumps(self, jumpstring, *targets):
		"""This is used to set jumps to other frames.
		Wherever you use ^ it will place a jump to a frame that	you name (a label or a frame number) in the args.
		
		Use
		===
		Typical use::
		  self.keys ( "#--#----#",...)
		  self.jumps( "........^", "start" ) #look for label 'start' and jump to it.
		  self.jumps( "...^", 1 ) # keep looping back to frame 1.
		
		ends.

		@param jumpstring: A dot (.) is a blank. A caret (^) points at the frame where the jump happens.
		@param targets: Any number of strings (labels) or integers (frame numbers). One per jump.
		"""		
		if not self.frameobjectList:
			raise _Bugs("SET_KEYFRAMES_FIRST")
		
		if len(jumpstring) == 0: 
			raise _Bugs("EMPTY_JUMP_STRING") 
		if jumpstring.count("^") != len(targets):
			raise _Bugs("MISSING_JUMP_TARGETS")
			
		## Process the string
		## It looks like this:
		## "____^_" : ^ = Jump(frame), _=nothing
		fr = 0
		i = 0
		for c in jumpstring:
			if c == "^":
				jump = targets[i]
				i += 1
				self.frameobjectList[fr].jump = jump # a frame or label
			fr += 1			

	def funcs(self, funcstring, *funcs):
		"""This is used to call functions from frames. Wherever you use ^ it will place a function that
		you name in the args.

		NOTE
		====
		If you have a jump in the same frame, the function will not get run. You can always avoid a jump; instead have a function that
		uses goPlay(x).

		WARNING
		=======
		Make sure you B{name} the function; not B{call} it. It's easy to make a mistake. 
		Use funcName and not funcName(). 
		If there are arguments then use a tuple	like (funcName, arg1,arg2,...)

		Use
		===
		Typical use::

		  self.funcs( "........^", somefunc )
		  self.funcs( "...^", (self.goPlay,10) ) # Here we pass 10 to goPlay.

		ends.
		""" 
		if not self.frameobjectList:
			raise _Bugs("SET_KEYFRAMES_FIRST")
		
		if len(funcstring) == 0: 
			raise _Bugs("EMPTY_FUNC_STRING") 
		if len(funcs) < 1:
			raise _Bugs("MISSING_FUNCS")			
		if funcstring.count("^") != len(funcs):
			raise _Bugs("MISSING_FUNCS")
			
		## Process the string
		fr = 0
		i = 0
		for c in funcstring:
			if c.upper() == "^":
				args = None
				funcy=funcs[i]
				if type(funcy) == tuple:
					## it's (funcname,arg1,arg2,...)
					func = funcy[0]
					args = funcy[1:]
				else:
					func = funcy
				i += 1
				## Only allow funcs where there is no blank
				if not self.frameobjectList[fr].blank:
					self.frameobjectList[fr].func = _Function(func, args)
				else:
					raise _Bugs("FUNC_ON_BLANK")
			fr += 1
		
	def _findFrame(self, LF):
		"""Private: Given a frame number or a label, return the number."""
		if type(LF) == int: return LF # do no bounds test to keep it fast
		## Looking for labels is going to be slow
		for f in self.frameobjectList:
			if f.label == LF:
				return f.frame
		raise _Bugs("NO_SUCH_LABEL", label = LF)

	def _getFrameAndAdvanceFrame(self):
		"""Private: Advance a thing's frame, if all the gods and winds are favourable."""
		##
		## NB:: This func is part of TIMELINE TICK - it happens from in there
		##	  therefore we have YET TO reach draw. 
		## NB:: Do not put any callbacks to the Thing in this routine.
		##	  ONLY put that kind of stuff in expose


		## The currentFrameData is always one behind the self.frame 
		thisFrame = self.frameobjectList[self.frame - 1]#bloody fenceposts :|
		
		##Is there a JUMP in this frameProps?
		if thisFrame.jump:
			##self.goPlay( thisFrame.jump ) # This forces a Play, and that's unwanted
			frame =  self._findFrame( thisFrame.jump )
			self.frame = frame
			thisFrame = self.frameobjectList[frame - 1 ] # get the new one
			
		## It's new, from one ahead in time.
		self.currentFrame = thisFrame # It may be new, so update the thing
		
		## Has the thing been goStop() stopped?
		if self.stopped: 
			return self.currentFrame
		
		## Is there a stop in the key? Ignorestop is a way to temporarily disable the stop. See goPlay()
		## This lets us pass through a stop frame.
		if thisFrame.stop and not thisFrame.temporarilyIgnoreThisPropertyStop:
			return self.currentFrame
		thisFrame.temporarilyIgnoreThisPropertyStop = False #reset it.
		
		## Put the next frameProps into myself.
		self.currentFrame = thisFrame


		self.frame += 1
		if self.frame > self.lifespan:
			if self.loops:  self.frame = 1
			else: self.frame = self.lifespan
		return thisFrame 

	def changeLayer(self,newlayer):
		"""This will change the layer (z-order) of this thing. Zero is the bottom layer.
		If you find this is not working, then make sure you give layer numbers, to each
		thing you add(), that are larger than zero.
		"""
		# my thinglist is sorted by layer -- this is what _tick() loops.
		# To change *MY* layer, I have to find my parent, then find myself in that
		# thinglist and then change the layer number and re-order.
		
		#if isinstance(self,ThingsApp.AllThings):

		myparent = self.parentThing
		for tup in myparent.thinglist:
			# The tup is (layer, thing, parentframe)
			if tup[1] is self:
				i = myparent.thinglist.index(tup)
				pf = tup[2]
				newtup = (newlayer,self,pf)
				myparent.thinglist[i] = newtup
				break
		myparent.thinglist.sort()
		#for tup in myparent.thinglist:
		#	print tup
		#print

	def toTop(self):
		"""
		Move this Thing to the very top layer.
		"""
		for tup in self.parentThing.thinglist:
			if tup[1] is self:
				i = self.parentThing.thinglist.index(tup)
				del self.parentThing.thinglist[i] # Remove it from whence it came
				self.parentThing.thinglist.append(tup) #force it onto end of list: ergo top layer
				break
		# no sort, cos that would move the layer back again.

	def nextFrame(self):
		"""Send a thing to its next frame. Kind of untested really..."""
		self._getFrameAndAdvanceFrame()
		
	def stop(self):
		"""Tell this Thing to stop playing. This is different
		from a stop in a keyframe in that it's not recorded
		in the key's properties -- so it won't be remembered.
		
		Use
		===
		thing.stop()
		"""
		self.stopped = True
		self.currentFrame.temporarilyIgnoreThisPropertyStop = True
	def play(self):
		"""Tell this Thing to start playing again.
		However it was stopped, this will cause it to start again.

		Use
		===
		thing.play()
		"""
		self.stopped = False
		self.currentFrame.temporarilyIgnoreThisPropertyStop = True
		
	def loop(self, bool = True):
		"""Set to True to have this Thing keep looping around(the default.)"""
		self.loops = bool
		
	def changeProps(self, **args):
		"""Changes the current key's properties.
		This is one way to move/scale things around.

		Use
		===
		thing.changeProps( x=10, y=20 )

		Better
		======
		Use thing.currentFrame.props.x = 10, it's faster.
		"""
		#A way for user to change frame data in a current keyframe
		## Change currentFrame.props as per the values that come in.
		for k,v in args.iteritems():
			if k in self.currentFrame.props.__dict__:
				setattr(self.currentFrame.props, k, v)

	def getProps( self, *args ):
		"""
		Fetch the named properties of a Thing.
		
		Use
		===
		x,y,a = self.getProps("x","y","a")

		Note
		====

		It's quicker to use: self.currentFrame.props.BLAH, where BLAH
		is a property like x,y or rot.

		@param args: Takes any number of string argument naming properties.
		@return: A list.

		"""
		retlist = []
		for k in args:
			if k in self.currentFrame.props.__dict__:
				retlist.append(getattr( self.currentFrame.props,k ))
		if not retlist:
			raise _Bugs("BAD_PROPS")
		return retlist
		
	def showData( self ):
		"""Useful to debug things.

		Use
		===
		thing.showData()
		"""
		print self.currentFrame
		
	def _parseKeys(self, props):
		"""Private: Pass in a list of frame PROP objects, one for each key in the layout."""
		#Now, I must parse the keyframelayoutstring in tandem with props[] and build keys[]
		#This must do the linear interpolation between each tweened key frame,
		#as well as insert blank keys and check syntax. Gulp.
		
		#The form is:
		# "#-----#=====..#.#=="
		# : # : a thing(There must be at least one)
		# : ## : would imply two props that differ in some way. It's not the same as "#=", but might be.
		# : - : a tween frame(must be within two #'s)
		# : = : a frame same as the one before(must follow a #)
		# : . : a blank frame
		# There must be a frameProps object for every "#"
		
		#First, kick out the obvious
		s = self.keyframelayoutstring
		lens = len(s)
		if not("#" in s): raise _Bugs("MISSING_HASH")
		if s.count("#") != len(props): raise _Bugs("NOT_ENOUGH_KEYS")
			
		#Now the typical start errors
		if s[0] == "-" or s[0] == "=":
			raise _Bugs("BAD_KEY_STRING")
		#Now the end errors
		if s[lens-1] == "-":
			raise _Bugs("ENDS_IN_TWEEN")
		#Now look for error patterns
		#[-=], [=-], [.=], [=.], [.-], [-.]
		for e in range(0,lens-1):
			#Check one against the next, then shift by one.
			test = s[e] + s[e+1]
			if test == "-=" or test == "=-" or test == ".=" or test == ".-" or test == "-.":
				raise _Bugs("BAD_KEY_COMBO", test = test, pos = e + 1)
		
		#At this point the string is perfect.(For variable values of 'perfect' ...)
		
		#Pyparsing miracle ... I am still stunned by this :)
		cutup = Thing._animation.parseString(s)
	   
	   ## NOTE:
	   ##  After a loooooong debug session, I realised that the frameProps that come
	   ##  into this routine must be handled with some suspicion. Esp. the 
	   ##  zeroprop frameProps. If a frameProps is *already* an instance and it gets shared
	   ##  among many Things, then changing one naturally changes all the others!
	   ##  I want each frameProps to be unique to the Thing, so I must make new objects
	   ##  and break any links to the originals.
	   
	   ## Dec 8 change: I put the props within a _Frame object, so it's _Frame.props.whatever now.
	   ## This is so that *some* frames can *share* common props. For example "#==", the
	   ## two "==" objects must be unique *but* share the same x,y,etc. as the "#" does.
	   ## If the "#" key is changed by the user, then all the linked "=" keys must reflect
	   ## the same values of x,y,etc.
	   ## To do this I can now use a shallow copy of a frame object.
	   
		newprops =[]
		for p in props: #what comes into this method
		   dup = objectduplicate.deepcopy(p)
		   newprops.append(dup)
		## Now put them back into props just because frameobjectList[] is already in the code below
		del  props 
		props = newprops
		del  newprops
		
		## Sorted.
		
		pindex = 0
		#self.frameobjectList =[] #reset it. This may be needless.
		for run in cutup:
			#print run,pindex
			#The only multi-runs are tweens
			if run == "#":
				po = props[pindex] #also used in '-' test
				fo = _Frame()
				fo.props = po
				fo.tag="#"
				pindex += 1
				self.frameobjectList.append( fo )
			if run == "=":
				## '=' follows '#' so we COPY(a new instance) the p from above
				##
				## I must make a copy of p, but the visual data
				## like x,y,sx,sy etc, must be the LINKED to the original
				## '#' before me... damn... fixed by new _Frame->Props plan
				
				newfo = objectduplicate.copy( fo ) #shallow copy of p, leaving props tied to p
				newfo.tag="="
				self.frameobjectList.append( newfo )

			if run == ".":
				fo = _Frame()
				fo.blank = True
				fo.tag="."
				self.frameobjectList.append( fo )
				
			if len(run) > 1 or run == "-":
				average_length = len(run) + 1
				first_prop = po #got it already
				last_prop = props[ pindex ]
				#print "pindex + 1 ", pindex
				#Make the average frameProps.
				#vars = self.currentFrameData.varlist 
				avp = Props() #For tmp use
				#Set it up with all the average skips. It's a mere holder of values.
				for v in self.currentFrame.props.__dict__:
					diff = getattr(last_prop,v) - getattr(first_prop,v) #subtract end from start
					av = float(diff)/float(average_length) #get the average
					setattr(avp,v,av) #put average into that property var
				#Now add first_frameobj, then make props for those inbetween
				prev_prop = first_prop
				for c in run:
					tween_prop = Props()
					a_frame = _Frame()
					for v in self.currentFrame.props.__dict__: #:vars:
						av = float(getattr(avp,v))# Fetch var from our "average" frameProps
						calc = float((getattr(prev_prop,v)+av))
						setattr(tween_prop,v,calc)
					tween_prop.tag = "-"
					a_frame.props = tween_prop
					self.frameobjectList.append( a_frame ) #add it to my keys
					#print "Add tween"
					prev_prop = tween_prop   
		
		#Put frame numbers into them.
		fr = 1
		for f in self.frameobjectList:
			f.frame = fr
			fr += 1
			
##		print "setProps completed okay for %s THESE ARE ITS PROPS:" % self.id
		#print self.keyframelayoutstring
##		for p in self.frameobjectList:
##			print p
####			
####		print
##		
##		#print "TEST::", self.id
##		#print self.frameobjectList[0]
##		#raise SystemExit
		
		return True
		
	def __len__( self ):
		return self.lifespan

	## Finally got this working.
	def goPlay(self, frameorlabel):
		"""
		Tell this Thing to go to a certain frame number(or label) and play.

		Use
		===
		
		thing.goPlay(7)
		
		thing.goPlay("die")
		"""
		frame =  self._findFrame(frameorlabel)
		self.frame = frame
		self.stopped = False
		self.currentFrame = self.frameobjectList[frame - 1 ]
		self.currentFrame.temporarilyIgnoreThisPropertyStop = True #This will be reset as soon as it's processed.
		
	def goStop(self, frameorlabel):
		"""
		Tell this Thing to go to a certain frame number(or label) and stop.

		Use
		===
		
		thing.goStop(2)
		
		thing.goStop("hide")
		"""
		frame =  self._findFrame(frameorlabel)
		self.frame = frame
		self.stopped = True
		self.currentFrame = self.frameobjectList[frame - 1 ]

## April 2009
## deepcopy stopped working :(
## I get this error:
##  File "/usr/lib/python2.5/copy_reg.py", line 92, in __newobj__
##    return cls.__new__(cls, *args)
##  TypeError: instancemethod expected at least 2 arguments, got 0
##
#	def duplicate( self, masterKey=None, id="duplicate_id_not_set" ):
#		"""Use this to duplicate this Thing.
#		You can pass a Prop that will be used to alter all the
#		keys of the original.
#		
#		Use:
#			newthing = someOtherThing.duplicate( newProp )
#		"""
#		dup = objectduplicate.deepcopy(self)
#		dup.id = id
#		
#		## Deepcopy appears to live up to its rep, it has also made *copies* of all the props!
#		
#		if masterKey:
#			for frameObj in dup.frameobjectList:
#				for v in self.currentFrame.props.__dict__:
#					new = getattr(masterKey, v); old = getattr(frameObj.props, v)
#					calc = float((new - old) + old)
#					if calc != 0:
#						if calc < 0: #Some vars should not be negative.
#							if v == "a": calc = 0
#							if v == "sx": calc = 0
#							if v == "sy": calc = 0
#						org = getattr(frameObj.props, v)
#						setattr(frameObj.props, v, calc) #Change the frameProps
#		return dup






		
class DrawThing(Thing):
	"""
	A DrawThing is a very simple Thing: You need only supply the draw() method.
	(Don't use an __init__ method at all.)
	"""
	def __init__(self):
		Thing.__init__(self)
		self.keys("#",Props())


class HitThing(Thing):
	"""
	Subclass this to have your object draw a hit area and detect hits.
	This gives your Thing the power to receive all kinds of mouse events.

	This is a normal Thing, and can have other things added to it. See the help for Thing and Timeline.

	Events
	======
	These events are fired. Catch them by defining a def with one of these names.

	 1. onDrag(x,y)
	 2. onDrop(dx,dy,x,y) : Use (dx,dy) to change props with. (x,y) are useful for hit testing.
	 3. onDown(x,y)
	 4. onRelease()/onClick()
	 5. onEnter(x,y)
	 6. onRollOver(x,y)
	 7. onLeave(x,y)
	 
	Hit Area
	========
	Define a drawHitarea method where you then draw (using paths, not stroked or filled) the hit area.

	The signature is: def drawHitarea(self,context,frame_number)

	Use
	===
	Typical use::

	  class Touchy(HitThing):
	    def __init__(self,x, y): # this is up to you
	      ## Init the class - very important.
	      HitThing.__init__( self, id="button" )
	      self.keys( "#", Props() )
	    def onRollOver(self,x,y):
	      print "You are in ",x,y
	    def drawHitarea(self,ctx,fr):
	      ctx.rectangle(0,0,50,50) # this rectangle will be the hit-area
	.

	"""
	def __init__( self, id="Hitter" ):
		Thing.__init__( self, id)
		self._state = _MOUSENOTWITHIN
		self.dx, self.dy = 0, 0
		self.is_being_dragged = False # Keep tabs on my state.
		self.mydict = self.dirstore#__class__.__dict__ # What methods are in my instance?
		self.fireEventOnce = False
		self.oldx, self.oldy = self.currentFrame.props.x, self.currentFrame.props.y
		
	def _handleRollover(self, hit, userspace_x, userspace_y):
		if hit: # YES, we are INSIDE the thing
			if self._state == _MOUSENOTWITHIN: # the state is currently this, so:
				## We have JUST entered the thing, fire this once:
				if "onEnter" in self.mydict:
					self.onEnter(userspace_x, userspace_y)
					
			self._state = _MOUSEWITHIN

			if "onRollover" in self.mydict:
				self.onRollover(userspace_x,userspace_y)		

		else: # NO, we are NOT inside anymore.
			if self._state == _MOUSEWITHIN:  # the mouse WAS inside, so:
				## We have JUST left the thing, fire this once:
				if "onLeave" in self.mydict:
					self.onLeave(userspace_x, userspace_y)
			
			self._state = _MOUSENOTWITHIN


		
	def _handleMouseButtons(self,  x, y):
		if Stack.stack.button_is_pressed_down:
			## Record the gap between the mouse and the Thing.
			self.dx = x - self.currentFrame.props.x
			self.dy = y - self.currentFrame.props.y
			## I want to fire this ONCE, so I need a flag to control that
			if self.fireEventOnce  == False:
				## It's not been done yet
				if "onDown" in self.mydict:
					self.onDown(x, y)
				self.fireEventOnce = True 
			return
			
		if Stack.stack.button_has_just_been_released:
			if self.fireEventOnce: # only notice if button *was* just down.
				if "onRelease" in self.mydict:
					self.onRelease()
				## onclick() is more natural to use:
				if "onClick" in self.mydict:
					self.onClick()					
			if self.is_being_dragged:
				if "onDrop" in self.mydict:
					self.onDrop(x, y)
				self.is_being_dragged = False						

			self.fireEventOnce = False
			
	def _handleDragging(self, hit, x, y, userspace_x, userspace_y):
		if hit and Stack.stack.button_is_pressed_down:
			self.is_being_dragged = True
			
		if Stack.stack.button_is_pressed_down and self.is_being_dragged:
			if "onDrag" in self.mydict:
				self.onDrag( x, y, userspace_x, userspace_y )



class ClipThing(Thing):
	"""
	Use this to have this Thing become a clip for all its children.
	If you want to animate this entire ClipThing, then wrap it in another Thing.

	This ClipThing can animate via its draw method, this will affect the clip(mask) 
	itself -- create animated masks in this way. You would do this by making sure a 
	slightly different path was drawn each frame. You could use a BagOfStuff path and
	draw a new path based on the 'frame_number' argument that gets passed to draw().

	Use
	===
	Typical use::

	  class ChopChop(ClipThing):
	    def __init__(self,x, y): # the params are up to you
	      ## Init the class
	      ClipThing.__init__( self, id="button" )
	      self.add( SomeOtherThing ) #Will be clipped by what I draw() below.
	    def draw(self,ctx,fr):
	      ## Let's make a circle mask:
	      drawACircle() #Do not stroke/fill it.
		  
	.

	"""
	def __init__( self, id="Clipper"):
		Thing.__init__( self, id)
		self._isClip = True




class FollowThing( Thing ):
	"""
	In a FollowThing, you provide a path drawing function 
	and then whatever you add to this Thing will follow that path.

	Use
	===
	Typical use::

	  class FollowPath(FollowThing):
		  def __init__(self ):
			  FollowThing.__init__(self)
			  self.keys  (40, self.path)
			  self.dot = Dot() # This will follow the path!
			  self.add(self.dot)
		  def path(self,ctx):
		      ## This is where you draw the path.
			  ## Here we use a path id taken from an Inkscape SVG file.
			  ## However, you can also use the normal pyCairo drawing commands
			  ## and construct a path in any way you like.
			  ## Simply remember -- do not stroke or fill the path!
			  bos["foll:swoopy"].draw(ctx)
	.	

	Notes
	=====
	There is no notion (yet) of orienting the thing that follows (the path) to the path itself. The maths is over my head! Waaay over.

	"""

	def keys(self, numFrames, pathFunc, startAtFirstNode=True):
		"""
		This keys() method is slightly different to the normal one.

		@param numFrames: Use a number greater than zero. This is the 'smoothness' of the follow animation.
		 You might get an error related to the path and it's number of nodes. Try increasing the numFrames,
		 as this code is still not working properly. In fact, I could do with major help in this dept.
		@param pathFunc: Supply a function which can be called where you draw a path.
		@param startAtFirstNode: True starts at one end of the path, False at the other.
		"""
		if numFrames < 1: 
			raise _Bugs("FOLLOWTHING_NEEDS_FRAMES")

		if not callable(pathFunc):
			raise _Bugs("FOLLOWTHING_NEEDS_A_PATH_FUNCTION")

		## We need to interrogate the path the user wants drawn.
		## To do this we use cairo and that means we need a context
		## Since the expose event is not yet running, we need to create
		## a temporary surface to get a contxt to draw the path to get the nodes.
		WIDTH,HEIGHT = Stack.stack._windowSize ## Get it from there.
		tmpsurface = cairo.ImageSurface (cairo.FORMAT_ARGB32, WIDTH, HEIGHT)
		ctx = cairo.Context (tmpsurface)		
		#ctx.set_tolerance(0.1) 
		pathFunc(ctx) # path() is a method in the user's object - it draws the path
		path = ctx.copy_path_flat() # no curve_to's
		del ctx

		## Now to use the code and idea that Gerdus Van Zyl helped me with
		## on the cairo list. gerdusvanzyl@gmail.com. Many thanks!
		##
		## The idea is to use what's in path now, but insert new nodes as needed.
		## Then go through it all and make a new path list of evenly spaced nodes
		## and give those to the normal keys() method.

		path = [a[1] for a in path if a[0] != 3] # Get all but type 3 nodes
		nenodes = len(path)
		#print "Existing Nodes:",nenodes

		## First -- get an estimation of how long the path is.
		totaldistance = 0
		idx = 0
		for p in path:
			if idx == 0:
				prevpoint = path[idx]
			else:
				prevpoint = path[idx-1]
			## This calculation is knows as the 'Manhattan distance" or "Taxi-cab" 
			## http://en.wikipedia.org/wiki/Manhattan_distance
			distance = abs(p[0]-prevpoint[0]) + abs(p[1]-prevpoint[1])
			idx += 1
			totaldistance += distance

		## Now -- chop it up and spread the nodes evenly throughout it.
		newpath = []
		idx = 0
		for p in path:
			if idx == 0:
				prevpoint = path[idx]
			else:
				prevpoint = path[idx-1]

			distance = abs(p[0]-prevpoint[0]) + abs(p[1]-prevpoint[1])
			#print "distance:",distance, totaldistance
			nodestoinsert = distance / totaldistance * (numFrames-nenodes)
			nodestoinsert = int(round(nodestoinsert))
			#print "nodes to insert:",nodestoinsert

			xdistance = p[0]-prevpoint[0]
			ydistance = p[1]-prevpoint[1]
			for i in range(1,nodestoinsert+1):
				x = prevpoint[0] + (xdistance/(nodestoinsert+1)*i)
				y = prevpoint[1] + (ydistance/(nodestoinsert+1)*i)
				newpoint = (x,y)
				newpath.append(newpoint)
			newpath.append(p)
			
			idx += 1
		
		## Now to decide if we have to skip nodes, or if we can use them all
		lnp = len(newpath)
		#print "len newpath:",lnp

		if numFrames > lnp:
			raise _Bugs("FOLLOW_PATH_NEEDS_MORE_NODES",has=lnp,want=numFrames)


		#skip = 1
		#if numFrames < lnp:
			## If there are fewer frames requested by user than what's actually available
			## then we have to skip bunches of them, so work out that skip:
		skip = float(lnp)/numFrames
		## If there are more frames req by user than are in the newpath, it does not seem
		## to cause a problem. I am not sure why. It could be the normal looping nature
		## of the timeline. I have not looked-into this possible FollowThing bug.

		props=[]
		for l in range(0,numFrames):
			s = skip * l
			i=int(s)
			tup=newpath[i]
			props.append(Props(x=tup[0],y=tup[1]))

		kfs=len(props)
		if not startAtFirstNode:
			props.reverse()
		## Make the Props() to set x,y for each keyframe
		Thing.keys(self,"#" * kfs, *props)
		del(newpath)

class SceneThing( Thing ):
	"""
	A SceneThing is a simple collection for other Things. It starts on a blank frame
	with a stop, so it's invisible and nothing is happening. In order to start it, 
	call the play() method.

	You can make many SceneThings and add them to your AllThings instance (in my examples
	and demos it's called 'app'). They get recorded in sequence and you can get one
	to start the next by calling the Allthings.playNextScene() method.

	I{See also} AllThings.startScene().
	"""
	def __init__(self):
		Thing.__init__(self)
		self.keys(  ".#", Props() )
		self.stops( "^")
	def add(self, *args, **kwargs):
		"""
		Override of Timeline.add to cater for adding things to a SceneThing.

		See B{Timeline.add}

		Note
		====
		This adds Things to a SceneThing: for this operation you B{only use the thing argument} to add.

		Note2
		=====
		You can add more than one Thing to a Scene (in many separate add calls), but be sure to control when stuff plays or they will all play at once.
		"""
		# We are adding something to a SceneThing. For this I keep a sep. list and we need only the thing
		# This is used in AllThings.playNextScene()
		thing=args[0]
		self.scenelist.append( thing )

		# Now continue the add call...
		Thing.add(self, *args, **kwargs)

	def play(self):
		self.goStop(2)


