<?php
/**
 * Upload all the image files contained in an archive file (tar or zip)
 *
 * @param file local filename of uploaded file
 * @param file_name filename of uploaded file on remote computer
 * @param file_size size of the file (unused)
 * @param file_type mime_type of the file (unused)
 * @param rollID roll in which to put uploaded pictures
 * @param parentTopicID topic in which to put uploaded pictures
 * @param groups groups to add to the picture
 */
function uploadArchive(
	&$tof,
	$file,$file_name,$file_size,$file_type,
	$rollID,$parentTopicID,$groups) 
{
	$tof->debug("uploadArchive $file_name in $rollID");
    $dirname = $tof->basePath . $rollID;
    if (!$tof->checkDirectory($dirname)) {
		return false;
    }
   
	$tof->debug("dirname = $dirname");
    $tof->debug("file = $file ($file_type)");
    $tof->debug("file_name = $file_name");
	
    chdir($dirname);
    $files = array();
    if (eregi(".*\.tar",$file_name)) {
		exec("tar xf $file");
		exec("tar tf $file",$files);
		reset($files);
		while (list($key,$value) = each($files)) {
			if (eregi("(.*)\.jpe?g",$value,$match)) {
				$pictureFile = $dirname . "/" . $value;
				$frameID = basename($match[1]);
				$pictureID = uploadPicture(
					$tof,
					$pictureFile,$rollID,$frameID,$dirname,$parentTopicID,"",$groups);
				$tof->log(
					sprintf(_('Picture "%s" uploaded to "%s"'),
							$tof->getPictureLink($value,$pictureID,$rollID),
							$tof->getRollLink($rollID)));
			} else {
				$tof->warn(sprintf(_('Skipping file %s: not a jpeg picture'),$value));
			}
		}
    } else if (eregi(".*\.zip",$file_name)) {
		exec("unzip -j $file");
		exec("unzip -l $file",$files);
		reset($files);
		while (list($key,$pictureFile) = each($files)) {
			if (eregi("[0-9]* +[-0-9]+ +[0-9:]+ +(.*\.jpe?g)",$pictureFile,$match)) {
				$pictureFile = $match[1];
				eregi("(.*)\.jpe?g",$pictureFile,$match);
				$pictureFile = $dirname . "/" . $pictureFile;
				$frameID = basename($match[1]);
				uploadPicture($tof,
					$pictureFile,$rollID,$frameID,$dirname,$parentTopicID,"",$groups);
				$tof->log(
					sprintf(_('Picture "%s" uploaded to "%s"'),
							$tof->getPictureLink($match[1],$pictureID,$rollID),
							$tof->getRollLink($rollID)));
			} else {
				$tof->warn(sprintf(_('Skipping file %s: not a jpeg picture'),$pictureFile));
			}
		}
    } else {
		$tof->error(sprintf(_("unhandled file type: %s"),$file_name));
    }
}

/**
 * Upload a single picture
 *
 * @param file local filename of uploaded file
 * @param file_name filename of uploaded file on remote computer
 * @param file_size size of the file (unused)
 * @param file_type mime_type of the file (unused)
 * @param rollID roll in which to put uploaded pictures
 * @param parentTopicID topic in which to put uploaded pictures
 * @param description a description for the picture
 * @param groups groups to add to the picture
 */
function uploadFile(
	&$tof,
	$file,$file_name,$file_size,$file_type,
	$frameID,$rollID,$parentTopicID,$description,$groups) 
{
	if (!$tof->ct->getTopic($parentTopicID)) {
		$tof->error(sprintf(_("You must create the topic %s before you can add pictures in it"),
							 $parentTopicID));
		return false;
	}
	if (!is_array($tof->ct->getRoll($rollID))) {
		$tof->error(sprintf(_("You must create the roll %s before you can add pictures in it"),
							 $rollID));
		return false;
	}
	$dirname = $tof->basePath . $rollID;

	if (!$tof->checkDirectory($dirname)) {
		return false;
	}

	$pictureID = $tof->uploadPicture(
		$file,$rollID,$frameID,$dirname,
		$parentTopicID,$description,$groups);
	$tof->log(sprintf(_('Picture "%s" uploaded to "%s"'),
					   $tof->getPictureLink($file_name,$pictureID,$rollID),
					   $tof->getRollLink($rollID)));
}

/**
 * Resize picture (max 600x600) and create thumbnail after
 * upload. Also read exif date and ensure the picture is writable.
 *
 * @param file absolute pathname of uploaded file
 * @param rollID roll 
 * @param frameID picture ID in roll
 * @param dirname directory where to put resized picture and thumbnail
 * @return pictureID
 */  
function uploadPicture(
	&$tof,
	$file,$rollID,$frameID,$dirname,
	$parentTopicID, $description="", $groups=null) 
{
	// set mode to rw for user and group
	if (!chmod($file,0660)) { 
		$tof->error(sprintf(_('Failed to change file access permissions for \'%s\''),$file));
	}
	$date = $tof->readExifDate($file);
	$filename = $dirname . "/" . $frameID . $tof->orig_suffix;
	if (!file_exists($filename)) {
		if ($file!=$filename && !rename($file,$filename)) {
			$tof->error(sprintf(_("Failed to copy %s to %s"),$file,$filename));
			return false;
		}
	} else {
		$tof->warn(sprintf(_('File %s already exists'),$filename));
	}
	$picture = array('RollID' => $rollID, 
					 'FrameID' => $frameID,
					 'Description' => $description,
					 "Owner" => $tof->uid);
	if ($date)
		$picture['Date'] = $date;

	$tof->createThumb($picture);

	$size = $tof->getPictureDiskUsage($picture);
	$user = $tof->ct->getUser($tof->uid);
	if ($user['Quota'] && $user['DiskUsage'] + $size > $user['Quota']) {
		$tof->error(sprintf(_('Failed to upload picture %s/%s : quota exceeded'),$rollID,$frameID));
		unlink($tof->getOrigFilename($picture));
		unlink($tof->getPictureFilename($picture));
		unlink($tof->getThumbFilename($picture));
		return;
	}

	$pictureID = 
		 $tof->ct->addPicture(
			 $picture,
			 array($parentTopicID),
			 $groups);
	$picture['PictureID'] = $pictureID;
	$tof->increaseDiskUsage(
		$picture['Owner'],
		$size);
	return $pictureID;
}
?>